<?php
/**
 * i-doit - Documentation and CMDB solution for IT environments
 *
 * This file is part of the i-doit framework. Modify at your own risk.
 *
 * Please visit http://www.i-doit.com/license for a full copyright and license information.
 *
 * @version     1.10
 * @package     i-doit
 * @author      synetics GmbH
 * @copyright   synetics GmbH
 * @url         http://www.i-doit.com
 * @license     http://www.i-doit.com/license
 */
/**
 * i-doit
 *
 * Export helper for custom fields
 *
 * @package     i-doit
 * @subpackage  Export
 * @author      Van Quyen Hoang <qhoang@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_global_custom_fields_export_helper extends isys_export_helper
{
    /**
     * Export helper for yes-no values for custom categories
     *
     * @param $data
     *
     * @return array
     * @author   Van Quyen Hoang <qhoang@i-doit.com>
     */
    public function exportCustomFieldYesNoDialog($data)
    {
        if(!empty($data))
        {
            return [
                "title"     => $data,
                "prop_type" => 'yes-no'
            ];
        }
        return ["prop_type" => 'yes-no'];
    } // function

    /**
     * Import helper method for yes-no values for custom categories
     *
     * @param $data
     *
     * @return mixed|null
     * @author   Van Quyen Hoang <qhoang@i-doit.com>
     */
    public function exportCustomFieldYesNoDialog_import($data)
    {
        return (is_array($data)? $data[C__DATA__VALUE]: null);
    } // function

    /**
     * Export helper for calendar values for custom categories
     *
     * @param $p_value
     *
     * @return array|bool
     * @internal param $p_object_id
     */
    public function exportCustomFieldCalendar($data)
    {
        if (!empty($data))
        {
            return [
                "title"     => $data,
                "prop_type" => 'calendar'
            ];
        } // if

        return ["prop_type" => 'calendar'];
    }

    /**
     * Import method for calendar values (Custom categories).
     *
     * @param   array $p_value
     *
     * @return  mixed
     */
    public function exportCustomFieldCalendar_import($p_value)
    {
        if (is_array($p_value))
        {
            return $p_value[C__DATA__VALUE];
        } // if
        return null;
    }

    /**
     * Export helper for object connections for custom categories
     *
     * @param int|array $p_object_id
     *
     * @return array|bool
     */
    public function exportCustomFieldObject($objectId)
    {
        if (count($objectId) > 0)
        {
            $data = [];
            if(is_array($objectId))
            {
                foreach($objectId AS $l_obj_id)
                {
                    $data[] = $this->exportCustomFieldObjectHelper($l_obj_id, true);
                } // foreach
                $return = new isys_export_data($data);
            }
            else
            {
                $return = $this->exportCustomFieldObjectHelper($objectId, false);
            } // if
            return $return;
        } // if

        return ["prop_type" => 'browser_object'];
    }

    /**
     * Helper Method for method "exportCustomFieldObject"
     *
     * @param $p_object_id
     *
     * @return array|bool
     * @author   Van Quyen Hoang <qhoang@i-doit.com>
     */
    private function exportCustomFieldObjectHelper($objectId, $multiSelection = false)
    {
        $dao = isys_cmdb_dao::instance($this->m_database);
        $objectDataResult = $dao->get_object_by_id($objectId);

        if ($objectDataResult->num_rows() == 0)
        {
            return false;
        } // if

        $objectData = $objectDataResult->get_row();

        $objectTypeId = $objectData['isys_obj__isys_obj_type__id'];

        $cacheObjectType = $this->getCacheContent('object_type_rows', $objectTypeId);

        if(!$cacheObjectType)
        {
            $cacheObjectType = $dao->get_objtype($objectTypeId)->get_row();
            $this->setCacheContent('object_type_rows', $objectTypeId, $cacheObjectType);
        }

        $return = [
            "id"         => $objectId,
            "title"      => $objectData["isys_obj__title"],
            "sysid"      => $objectData["isys_obj__sysid"],
            "type"       => $cacheObjectType["isys_obj_type__const"],
            "type_title" => _L($objectData['isys_obj_type__title']),
            "prop_type"  => 'browser_object'
        ];

        if (isset($this->m_ui_info[C__PROPERTY__UI__PARAMS]['p_identifier']))
        {
            // Its a relation
            $return['identifier'] = $this->m_ui_info[C__PROPERTY__UI__PARAMS]['p_identifier'];
        } // if

        if ($multiSelection)
        {
            $return['multiselection'] = 1;
        }

        return $return;
    }

    /**
     * Import method for objects (Custom categories).
     *
     * @param   array $p_value
     *
     * @return  mixed
     */
    public function exportCustomFieldObject_import($value)
    {
        if (is_array($value))
        {
            if(isset($value['id']))
            {
                if ($value['id'] != $this->m_object_ids[$value['id']])
                {
                    return $this->m_object_ids[$value['id']];
                }
                else
                {
                    return $value['id'];
                } // if
            }
            elseif(isset($value[C__DATA__VALUE]) && is_array($value[C__DATA__VALUE]))
            {
                $return = [];
                foreach($value[C__DATA__VALUE] AS $data)
                {
                    if(isset($data['id']))
                    {
                        if($data['id'] != $this->m_object_ids[$data['id']])
                        {
                            $return[] = $this->m_object_ids[$data['id']];
                        }
                        else
                        {
                            $return[] = $data['id'];
                        } // if
                    } // if
                } // foreach
                return $return;
            }
        }

        return null;
    }

    /**
     * Get dialog plus information by id for custom categories
     *
     * @param int|array  $p_id
     * @param bool $p_table_name
     *
     * @return array
     */
    public function exportCustomFieldDialogPlus($dialogId, $tableName = false)
    {
        if (count($dialogId) > 0)
        {
            $return = $data = [];
            // Get corresponding table.
            if ($tableName)
            {
                $dialogTable = $tableName;
            }
            else
            {
                $dialogTable = $this->m_data_info[C__PROPERTY__DATA__REFERENCES][0];
            } // if

            if (empty($dialogTable))
            {
                // Data are generated in the ui
                if (isset($this->m_ui_info[C__PROPERTY__UI__PARAMS]['p_arData']))
                {
                    $dialogData = $this->m_ui_info[C__PROPERTY__UI__PARAMS]['p_arData'];

                    if (is_object($dialogData) && method_exists($dialogData, 'execute'))
                    {
                        $dialogData = $dialogData->execute();
                    } // if

                    if (is_string($dialogData))
                    {
                        $dialogData = unserialize($dialogData);
                    } // if

                    if(is_array($dialogId))
                    {
                        foreach($dialogId AS $id)
                        {
                            $data[] = $this->exportCustomFieldDialogPlusHelper($dialogData, $id, null, true);
                        }
                        $return = new isys_export_data($data);
                    }
                    else
                    {
                        $return = $this->exportCustomFieldDialogPlusHelper($dialogData, $dialogId, null, false);
                    }
                } // if
            }
            else
            {
                // Data is in the db.
                if(is_array($dialogId))
                {
                    foreach($dialogId AS $id)
                    {
                        $dialogData = isys_factory_cmdb_dialog_dao::get_instance($this->m_database, $dialogTable)
                            ->get_data($id);

                        if (!empty($dialogData))
                        {
                            $data[] = $this->exportCustomFieldDialogPlusHelper($dialogData, $id, $dialogTable, true);
                        } // if
                    } // foreach
                    $return = new isys_export_data($data);
                }
                else
                {
                    $dialogData = isys_factory_cmdb_dialog_dao::get_instance($this->m_database, $dialogTable)
                        ->get_data($dialogId);

                    if (!empty($dialogData))
                    {
                        $return = $this->exportCustomFieldDialogPlusHelper($dialogData, $dialogId, $dialogTable, false);
                    } // if
                } // if
            } // if

            return $return;
        } // if

        return ['identifier' => $this->m_ui_info[C__PROPERTY__UI__PARAMS]['p_identifier']];
    }

    /**
     * Helper Method for method exportCustomFieldDialogPlus
     *
     * @param      $p_data
     * @param      $dialogId
     * @param null $p_table
     *
     * @return array
     * @author   Van Quyen Hoang <qhoang@i-doit.com>
     */
    private function exportCustomFieldDialogPlusHelper($dialogData, $dialogId, $dialogTable = null, $multiSelection = false)
    {
        $return = [
            "id"         => $dialogId,
            "title"      => ($dialogTable !== null)? _L($dialogData[$dialogTable . "__title"]): _L($dialogData[$dialogId]),
            "const"      => ($dialogTable !== null)? $dialogData[$dialogTable . "__const"]: '',
            "title_lang" => ($dialogTable !== null)? $dialogData[$dialogTable . "__title"]: $dialogData[$dialogId],
            'identifier' => $this->m_ui_info[C__PROPERTY__UI__PARAMS]['p_identifier']
        ];

        if ($multiSelection)
        {
            $return['multiselection'] = 1;
        }
        return $return;
    }

    /**
     * Import method for dialog properties (Custom categories).
     *
     * @param      $p_title_lang
     *
     * @return null
     */
    public function exportCustomFieldDialogPlus_import($importData)
    {
        $data = [];
        if (is_array($importData))
        {
            if (isset($importData[C__DATA__VALUE]))
            {
                if(is_array($importData[C__DATA__VALUE]))
                {
                    foreach($importData[C__DATA__VALUE] AS $value)
                    {
                        $data[] = [
                            'title_lang' => ($value['title_lang'] != '' ? $value['title_lang'] : $value[C__DATA__VALUE]),
                            'identifier' => $value['identifier']
                        ];
                    }
                }
                else
                {
                    $data[] = [
                        'title_lang' => ($importData['title_lang'] != '' ? $importData['title_lang'] : $importData[C__DATA__VALUE]),
                        'identifier' => $importData['identifier']
                    ];
                }
            }
            else
            {
                return null;
            } // if
        }
        else
        {
            return null;
        } // if

        if(count($data) > 0)
        {
            $return = [];

            $dialogTable = $this->m_data_info[C__PROPERTY__DATA__REFERENCES][0];

            $dao = isys_cmdb_dao::instance($this->m_database);

            foreach($data AS $dialogData)
            {
                $query = 'SELECT * FROM ' . $dialogTable . ' WHERE ' . $dialogTable . '__identifier = ' . $dao->convert_sql_text(
                        $dialogData['identifier']
                    ) . ' ' . 'AND ' . $dialogTable . '__title = ' . $dao->convert_sql_text($dialogData['title_lang']);

                $result = $dao->retrieve($query);
                if ($result->num_rows() > 0)
                {
                    $return[] = $result->get_row_value($dialogTable . '__id');
                }
                else
                {
                    $l_insert = 'INSERT INTO ' . $dialogTable . ' (' . $dialogTable . '__identifier, ' . $dialogTable . '__title, ' . $dialogTable . '__status) ' . 'VALUES (' . $dao->convert_sql_text(
                            $dialogData['identifier']
                        ) . ',' . $dao->convert_sql_text($dialogData['title_lang']) . ',' . C__RECORD_STATUS__NORMAL . ')';

                    $dao->update($l_insert);
                    $return[] = $dao->get_last_insert_id();
                    $dao->apply_update();
                } // if
            }
            return (count($return) === 1)? array_pop($return): $return;
        }
        return null;
    } // function
} // class