<?php
/**
 * i-doit - Documentation and CMDB solution for IT environments
 *
 * This file is part of the i-doit framework. Modify at your own risk.
 *
 * Please visit http://www.i-doit.com/license for a full copyright and license information.
 *
 * @version     1.10
 * @package     i-doit
 * @author      synetics GmbH
 * @copyright   synetics GmbH
 * @url         http://www.i-doit.com
 * @license     http://www.i-doit.com/license
 */

namespace idoit\Module\QCW\Model;

use idoit\Model\Dao\Base;
use idoit\Module\QCW\Provider\CRUDCount;

/**
 * i-doit QCW object type group Model.
 *
 * @package     Modules
 * @subpackage  QCW
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class ObjectTypeGroup extends Base
{
    use CRUDCount;

    /**
     * @param  mixed $idConst May be the ID or constant.
     *
     * @return \isys_component_dao_result
     * @throws \isys_exception_database
     */
    public function load ($idConst = null)
    {
        $condition = '';

        if ($idConst !== null)
        {
            if (is_numeric($idConst))
            {
                $condition = 'WHERE isys_obj_type_group__id = ' . $this->convert_sql_id($idConst);
            }
            else
            {
                $condition = 'WHERE isys_obj_type_group__const = ' . $this->convert_sql_text($idConst);
            } // if
        } // if

        return $this->retrieve('SELECT *
            FROM isys_obj_type_group
            ' . $condition . '
            ORDER BY isys_obj_type_group__sort');
    } // function

    /**
     * Save method for a single object type group.
     *
     * @param  mixed $idConst May be the ID or constant.
     * @param  array $data
     *
     * @return bool
     */
    public function save($idConst = null, array $data)
    {
        $sqlData = [];

        if ($idConst === null)
        {
            $this->incrementCreate();

            // Use the standard way to create a new object type!
            return \isys_cmdb_dao::instance($this->m_db)
                ->insert_new_objtype_group(
                    $data['title'],
                    $data['const'],
                    65535,
                    ($data['visible'] ? C__RECORD_STATUS__NORMAL : C__RECORD_STATUS__BIRTH));
        } // if


        // ID-4369  This is necessary to prevent the user from (accidentally) overwriting language constants.
        if (isset($data['titleOrig']))
        {
            if ($data['title'] != $data['titleOrig'] && $data['title'] != _L($data['titleOrig']))
            {
                $sqlData[] = 'isys_obj_type_group__title = ' . $this->convert_sql_text($data['title']);
            } // if
        }
        else
        {
            $sqlData[] = 'isys_obj_type_group__title = ' . $this->convert_sql_text($data['title']);
        } // if

        if (isset($data['visible']))
        {
            $sqlData[] = 'isys_obj_type_group__status = ' . $this->convert_sql_int((isset($data['visible']) ? ($data['visible'] ? C__RECORD_STATUS__NORMAL : C__RECORD_STATUS__BIRTH) : C__RECORD_STATUS__NORMAL));
        } // if

        if (! count($sqlData))
        {
            return true;
        } // if

        $sql = 'UPDATE isys_obj_type_group SET ' . implode(', ', $sqlData);

        if (is_numeric($idConst))
        {
            $sql .= ' WHERE isys_obj_type_group__id = ' . $this->convert_sql_id($idConst);
        }
        else
        {
            $sql .= ' WHERE isys_obj_type_group__const = ' . $this->convert_sql_text($idConst);
        } // if

        $this->incrementUpdate();

        return $this->update($sql . ';');
    } // function

    /**
     * @param   array $data
     *
     * @return  array
     */
    public function saveMultiple(array $data)
    {
        $data = array_filter($data);

        foreach ($data as &$objTypeGroup)
        {
            try
            {
                $objTypeGroup = $this->save((($objTypeGroup['id'] > 0) ? $objTypeGroup['id'] : null), $objTypeGroup);
            }
            catch (\Exception $e)
            {
                $objTypeGroup = $e->getMessage();
            } // try
        } // foreach

        return $data;
    } // function

    /**
     * @param   mixed  $idConst  May be the ID or constant.
     *
     * @return  boolean
     * @throws  \isys_exception_dao
     */
    public function delete($idConst)
    {
        if (empty($idConst))
        {
            return true;
        } // if

        $sql = 'DELETE FROM isys_obj_type_group ';

        if (is_numeric($idConst))
        {
            $sql .= 'WHERE isys_obj_type_group__id = ' . $this->convert_sql_id($idConst) . ';';
        }
        else
        {
            $sql .= 'WHERE isys_obj_type_group__const = ' . $this->convert_sql_text($idConst) . ';';
        } // if

        $this->incrementDelete();

        return $this->update($sql);
    } // function

    /**
     * @param   array  $data
     *
     * @return  array
     */
    public function deleteMultiple (array $data)
    {
        $data = array_filter($data);

        foreach ($data as &$objTypeGroup)
        {
            try
            {
                if ($objTypeGroup['id'] > 0)
                {
                    $objTypeGroup = $this->delete($objTypeGroup['id']);
                } // if
            }
            catch (\Exception $e)
            {
                $objTypeGroup = $e->getMessage();
            } // try
        } // foreach

        return $data;
    } // function
} // class