<?php
/**
 * i-doit - Documentation and CMDB solution for IT environments
 *
 * This file is part of the i-doit framework. Modify at your own risk.
 *
 * Please visit http://www.i-doit.com/license for a full copyright and license information.
 *
 * @version     1.10
 * @package     i-doit
 * @author      synetics GmbH
 * @copyright   synetics GmbH
 * @url         http://www.i-doit.com
 * @license     http://www.i-doit.com/license
 */
/**
 *
 *
 * @package     i-doit
 * @subpackage
 * @author      Pavel Abduramanov <pabduramanov@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */

namespace idoit\Component\Table\Filter;

use idoit\Component\Table\Filter\Condition\CollectionCondition;
use idoit\Component\Table\Filter\Condition\ConstantCondition;
use idoit\Component\Table\Filter\Condition\InArrayCondition;
use idoit\Component\Table\Filter\Condition\IsSubstringCondition;
use idoit\Component\Table\Filter\Condition\NotCondition;
use idoit\Component\Table\Filter\Condition\TypeCondition;
use idoit\Component\Table\Filter\Formatter\ConstantFormatter;
use idoit\Component\Table\Filter\Operation\EqualsOperation;
use idoit\Component\Table\Filter\Operation\LikeOperation;
use idoit\Component\Table\Filter\Operation\HavingLikeOperation;
use idoit\Component\Table\Filter\Formatter\AppendFormatter;
use idoit\Component\Table\Filter\Formatter\CollectionFormatter;
use idoit\Component\Table\Filter\Formatter\IntFormatter;
use idoit\Component\Table\Filter\Formatter\PrependFormatter;
use idoit\Component\Table\Filter\Formatter\ReplaceFormatter;
use idoit\Component\Table\Filter\Formatter\SqlFormatter;
use idoit\Component\Table\Filter\Operation\OperationProvider;
use idoit\Component\Table\Filter\Operation\OrderByOperation;
use idoit\Component\Table\Filter\Operation\SkipOperation;
use idoit\Component\Table\Filter\Source\QueryMapSource;
use idoit\Component\Table\Filter\Source\QuerySource;
use idoit\Component\Table\Filter\Source\SessionSource;
use idoit\Component\Table\Filter\Source\SourceProvider;
use idoit\Component\Table\Filter\Source\TableConfigDefaultsSource;
use idoit\Component\Table\Table;
use idoit\Module\Cmdb\Model\Ci\Table\Config;
use isys_cmdb_dao_list_objects;
use isys_usersettings;

/**
 * Class Configuration
 * TODO: replace with DI
 *
 * @package idoit\Component\Table\Filter
 */
class Configuration
{
    /**
     * Get the configuration for the filters operations
     *
     * @param isys_cmdb_dao_list_objects $dao
     *
     * @return OperationProvider
     */
    public static function configureFilters(isys_cmdb_dao_list_objects $dao)
    {
        $tableConfig = $dao->get_table_config();
        $operations  = new OperationProvider($dao);

        $defaultFormatter = new CollectionFormatter();
        // We force the user to use "*" als wildchar by masking "%".
        $defaultFormatter->add(new ReplaceFormatter(['%', '*'], ['\%', '%']));
        // Add Wildcard on first position if in list config default wildcard is set
        if ($tableConfig->isFilterWildcard())
        {
            $defaultFormatter->add(new PrependFormatter('%'));
        }
        $defaultFormatter->add(new AppendFormatter('%'));
        $defaultFormatter->add(new SqlFormatter());

        // If we filter by object title we can can add it to the where condition because obj_main is the root table
        // apply Table::DEFAULT_FILTER_FIELD as title like 'value%'
        $default = new LikeOperation();
        $default->setFormatter($defaultFormatter)
            ->setColumnFormatter(ConstantFormatter::create('obj_main.isys_obj__title'))
            ->setCondition(new InArrayCondition([Table::DEFAULT_FILTER_FIELD]));

        // apply cmdb status as = (int)status_id
        $status = new EqualsOperation();
        $status->setFormatter(new IntFormatter())
            ->setColumnFormatter(ConstantFormatter::create('obj_main.isys_obj__isys_cmdb_status__id'))
            ->setCondition(new InArrayCondition(['isys_cmdb_dao_category_g_global__cmdb_status']));

        // default - having like 'value%'
        $fallback = new HavingLikeOperation();
        $fallback->setFormatter($defaultFormatter);

        $operations->addOperation($default);
        $operations->addOperation($status);
        $operations->addOperation($fallback);

        return $operations;
    }

    /**
     * Facade for filter
     * @param isys_cmdb_dao_list_objects $dao
     * @param                            $objectTypeID
     *
     * @return array
     */
    public static function filter(isys_cmdb_dao_list_objects $dao, $objectTypeID)
    {
        $filterSource = self::initFilterSource($dao->get_table_config(), $objectTypeID);
        $filter       = $filterSource->fetch();

        $filterProvider = self::configureFilters($dao);
        $filterProvider->apply($filter);

        return $filter;
    }

    /**
     * Facade for sort
     * @param isys_cmdb_dao_list_objects $dao
     *
     * @return array
     */
    public static function sort(isys_cmdb_dao_list_objects $dao)
    {
        $filterSource = self::initSortSource();
        $filter       = $filterSource->fetch();

        $filterProvider = self::configureSort($dao);
        $filterProvider->apply($filter);

        return $filter;
    }

    /**
     * Configure the sort operations
     *
     * @param isys_cmdb_dao_list_objects $dao
     *
     * @return static
     */
    public static function configureSort(isys_cmdb_dao_list_objects $dao)
    {
        $config     = $dao->get_table_config();
        $operations = new OperationProvider($dao);

        // ignore dialog fields, custom, descriptions from the sorting
        $ignoreCondition = new CollectionCondition(
            [
                new TypeCondition(
                    $config, new CollectionCondition(
                        [
                            new IsSubstringCondition('dialog'),
                            new IsSubstringCondition('isys_cmdb_dao_category_g_global')
                        ]
                    )
                ),
                new CollectionCondition(
                    [
                        new IsSubstringCondition('custom_fields'),
                        new IsSubstringCondition('description')
                    ]
                )
            ], false
        );
        $skip            = new SkipOperation();
        $skip->setCondition($ignoreCondition);
        $operations->addOperation($skip);

        // sort cmdb status by cmdb status id
        $cmdbStatus = new OrderByOperation();
        $cmdbStatus->setCondition(new InArrayCondition(['isys_cmdb_dao_category_g_global__cmdb_status']));
        $cmdbStatus->setColumnFormatter(ConstantFormatter::create('obj_main.isys_obj__isys_cmdb_status__id'));
        $operations->addOperation($cmdbStatus);

        // if its isys_cmdb_dao_category_g_global__ and not in table config columns - replace prefix to obj_main
        $replaceCondition = new CollectionCondition(
            [
                new IsSubstringCondition('isys_cmdb_dao_category_g_global__'),
                // type is not found in table config
                new NotCondition(new TypeCondition($config, new ConstantCondition(true)))
            ]
        );
        $replace          = new OrderByOperation();
        $replace->setCondition($replaceCondition);
        $replace->setColumnFormatter(new ReplaceFormatter(['isys_cmdb_dao_category_g_global__'], ['obj_main.isys_obj__']));
        $operations->addOperation($replace);

        // default - just sort
        $operations->addOperation(new OrderByOperation());

        return $operations;
    }

    public static function initSessionSource($objectTypeID)
    {
        $memorizeFilterTimer = (int) isys_usersettings::get('gui.objectlist.remember-filter', 300);

        return new SessionSource($memorizeFilterTimer, $objectTypeID);
    }

    /**
     * Configure the filters source. GET -> SESSION -> Table Config
     */
    public static function initFilterSource(Config $config, $objectTypeID)
    {
        $source = new SourceProvider();
        $source->addSource(new QuerySource('tableFilter'));
        $source->addSource(self::initSessionSource($objectTypeID));
        $source->addSource(new TableConfigDefaultsSource($config));

        return $source;
    }

    /**
     * Configure the sort source
     */
    public static function initSortSource()
    {
        $source   = new SourceProvider();
        $queryMap = new QueryMapSource();
        $queryMap->add('orderBy', 'orderByDir');
        $source->addSource($queryMap);

        return $source;
    }
}
