<?php
/**
 * i-doit - Documentation and CMDB solution for IT environments
 *
 * This file is part of the i-doit framework. Modify at your own risk.
 *
 * Please visit http://www.i-doit.com/license for a full copyright and license information.
 *
 * @version     1.10
 * @package     i-doit
 * @author      synetics GmbH
 * @copyright   synetics GmbH
 * @url         http://www.i-doit.com
 * @license     http://www.i-doit.com/license
 */

namespace idoit\Module\Check_mk\Console\Command;

use idoit\Console\Command\AbstractCommand;
use idoit\Module\Check_mk\Export;
use Monolog\Logger;
use Symfony\Component\Console\Input\InputDefinition;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class CheckMkExportCommand extends AbstractCommand
{
    const NAME = 'check_mk-export';

    /**
     * Get name for command
     *
     * @return string
     */
    public function getCommandName()
    {
        return self::NAME;
    }

    /**
     * Get description for command
     *
     * @return string
     */
    public function getCommandDescription()
    {
        return 'Exports i-doit objects and Host Tags to Check_MK WATO configuration files';
    }

    /**
     * Retrieve Command InputDefinition
     *
     * @return InputDefinition
     */
    public function getCommandDefinition()
    {
        $definition = new InputDefinition();

        $definition->addOption(new InputOption(
            'exportLanguage',
            'l',
            InputOption::VALUE_REQUIRED,
            'Use to export the configuration in a certain language',
            0
        ));

        $definition->addOption(new InputOption(
            'exportStructure',
            'x',
            InputOption::VALUE_REQUIRED,
            'Use to export the configuration in a certain language',
            Export::STRUCTURE_NONE
        ));

        return $definition;
    }

    /**
     * Checks if a command can have a config file via --config
     *
     * @return bool
     */
    public function isConfigurable()
    {
        return false;
    }

    /**
     * Returns an array of command usages
     *
     * @return string[]
     */
    public function getCommandUsages()
    {
        return [];
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        if ($input->getOption('exportLanguage') === 0) {
            $output->writeln("Use the '-l' parameter to export the configuration in a certain language:");
            $output->writeln(" 0: All available");

            foreach (isys_glob_get_language_constants() as $l_id => $l_lang)
            {
                $output->writeln(" " . $l_id . ": " . $l_lang);
            } // foreach

            $output->writeln("");
        }

        if ($input->getOption('exportStructure') === Export::STRUCTURE_NONE) {
            $output->writeln("Add the '-x' parameter to export the configuration files in a certain directory pattern:");

            foreach (Export::getStructureOptions() as $l_id => $l_option)
            {
                $output->writeln(" " . $l_id . ": " . $l_option);
            } // foreach

            $output->writeln("");
        }

        $output->writeln("Setting up system environment");

        $l_export = new Export(
            [
                'export_structure' => (int) $input->getOption('exportStructure'),
                'language'         => (int) $input->getOption('exportLanguage')
            ]
        );

        // Begin the export.
        $l_export->export();

        $l_logs = $l_export->getLogRecords();

        if (is_array($l_logs) && count($l_logs) > 0)
        {
            foreach ($l_logs as $l_log)
            {
                switch ($l_log['level'])
                {
                    case Logger::DEBUG:
                        $l_prefix = 'Debug: ';
                        break;

                    default:
                    case Logger::INFO:
                    case Logger::NOTICE:
                        $l_prefix = 'OK: ';
                        break;

                    case Logger::WARNING:
                    case Logger::ERROR:
                    case Logger::CRITICAL:
                    case Logger::ALERT:
                    case Logger::EMERGENCY:
                        $l_prefix = '!! ';
                        break;
                } // switch

                $output->writeln($l_prefix . $l_log['message']);
            } // foreach
        } // if

        $output->writeln('Done!');
    }
}
