<?php
/**
 * i-doit - Documentation and CMDB solution for IT environments
 *
 * This file is part of the i-doit framework. Modify at your own risk.
 *
 * Please visit http://www.i-doit.com/license for a full copyright and license information.
 *
 * @version     1.10
 * @package     i-doit
 * @author      synetics GmbH
 * @copyright   synetics GmbH
 * @url         http://www.i-doit.com
 * @license     http://www.i-doit.com/license
 */

namespace idoit\Module\JDisc\Console\Command;

use idoit\Console\Command\AbstractCommand;
use idoit\Console\Command\IsysLogWrapper;
use isys_helper_crypt;
use isys_jdisc_dao_discovery;
use isys_log;
use isys_module_jdisc;
use Symfony\Component\Console\Input\InputDefinition;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class JDiscDiscoveryCommand extends AbstractCommand
{
    const NAME = 'import-jdiscdiscovery';

    /**
     * Log instance for this handler.
     *
     * @var  isys_log
     */
    protected $m_log = null;

    /**
     * @var bool
     */
    private $m_show_log = false;

    /**
     * Get name for command
     *
     * @return string
     */
    public function getCommandName()
    {
        return self::NAME;
    }

    /**
     * Get description for command
     *
     * @return string
     */
    public function getCommandDescription()
    {
        return 'Triggers a JDisc discovery (API Access to the JDisc server is defined in the GUI)';
    }

    /**
     * Retrieve Command InputDefinition
     *
     * @return InputDefinition
     */
    public function getCommandDefinition()
    {
        $definition = new InputDefinition();

        $definition->addOption(new InputOption(
            'server',
            's',
            InputOption::VALUE_REQUIRED,
            'Selected "JDisc Server"'
        ));

        $definition->addOption(new InputOption(
            'discoveryJob',
            'j',
            InputOption::VALUE_REQUIRED,
            'Selected "Discovery Job"',
            'Discover all'
        ));

        $definition->addOption(new InputOption(
            'deviceHostname',
            'd',
            InputOption::VALUE_REQUIRED,
            'Selected device by "hostname"'
        ));

        $definition->addOption(new InputOption(
            'deviceHostAddress',
            'a',
            InputOption::VALUE_REQUIRED,
            'Selected device by "hostaddress"'
        ));

        $definition->addOption(new InputOption(
            'showLog',
            'l',
            InputOption::VALUE_NONE,
            'Show log while discovery'
        ));


        return $definition;
    }

    /**
     * Checks if a command can have a config file via --config
     *
     * @return bool
     */
    public function isConfigurable()
    {
        return false;
    }

    /**
     * Returns an array of command usages
     *
     * @return string[]
     */
    public function getCommandUsages()
    {
        return [];
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        /**
         * @param isys_jdisc_dao_discovery $p_discovery_obj
         */
        $handleOutput = function ($p_discovery_obj) use ($output) {
            if (defined('ISYS_VERBOSE') && $this->m_show_log === true)
            {
                // Get status logs only if verbose mode is set
                if (constant('ISYS_VERBOSE') === true)
                {
                    $l_running_data = $p_discovery_obj->get_running_discover_status();
                    $l_status       = $l_running_data['status'];
                    $l_last_log     = $l_running_data['log'];
                    $output->writeln($l_last_log, true);

                    while ($l_status === 'Running')
                    {
                        $l_running_data = $p_discovery_obj->get_running_discover_status();
                        $l_status       = $l_running_data['status'];
                        if ($l_last_log !== $l_running_data['log'] && $l_running_data['log'] !== '')
                        {
                            $l_last_log = $l_running_data['log'];
                            $output->writeln($l_last_log, true);
                        } // if
                    } // while
                    $output->writeln('Finished scanning device.', true);
                }
            }
        };

        // Start logging.
        $this->m_log = IsysLogWrapper::instance();
        $this->m_log->setOutput($output);

        // Retrieving the jdisc server.
        $l_jdisc_server = $input->getOption('server');

        // Retrieving the discovery job.
        $l_jdisc_discovery_job = $input->getOption('discoveryJob');

        // Retrieving the device hostname.
        $l_device_hostname = $input->getOption('deviceHostAddress');

        // Retrieving the device hostname.
        $l_device_hostaddress = $input->getOption('deviceHostname');

        $this->m_show_log = (bool) $input->getOption('showLog');

        // JDisc module
        $l_module = isys_module_jdisc::factory();

        $l_jdisc_server = $l_module->get_jdisc_discovery_data($l_jdisc_server, true)
            ->get_row();

        $l_host     = $l_jdisc_server['isys_jdisc_db__host'];
        $l_username = $l_jdisc_server['isys_jdisc_db__discovery_username'];
        $l_password = isys_helper_crypt::decrypt($l_jdisc_server['isys_jdisc_db__discovery_password']);
        $l_port     = $l_jdisc_server['isys_jdisc_db__discovery_port'];
        $l_protocol = $l_jdisc_server['isys_jdisc_db__discovery_protocol'];

        // JDisc Discovery object
        $l_discovery_obj = isys_jdisc_dao_discovery::get_instance();

        $l_discovery_obj->connect($l_host, $l_username, $l_password, $l_port, $l_protocol);

        if ($l_device_hostname === null && $l_device_hostaddress === null)
        {
            $l_discovery_jobs = $l_discovery_obj->get_discovery_jobs();
            foreach ($l_discovery_jobs AS $l_job)
            {
                if (strtolower($l_job['name']) == strtolower($l_jdisc_discovery_job))
                {
                    $l_discovery_obj->set_discovery_job($l_job);
                    break;
                }
            }
            if ($l_discovery_obj->get_discovery_job() === null)
            {
                $output->writeln('Discovery Job "' . $l_jdisc_discovery_job . '" not found.');
                return;
            }
            if ($l_discovery_obj->start_discovery_job())
            {
                $output->writeln('Discovery Job "' . $l_jdisc_discovery_job . '" has been triggered.');
                $handleOutput($l_discovery_obj);
            }
            else
            {
                $output->writeln('Failed to trigger the Discovery Job "' . $l_jdisc_discovery_job . '".');
            } // if
        }
        else
        {
            if ($l_device_hostaddress)
            {
                $l_discovery_obj->set_target($l_device_hostaddress);
            }
            elseif ($l_device_hostname)
            {
                $l_discovery_obj->set_target($l_device_hostname);
            } // if
            if ($l_discovery_obj->discover_device())
            {
                $output->writeln('Discovery of device "' . $l_discovery_obj->get_target() . '" started.', true);
                $handleOutput($l_discovery_obj);
            } // if
        } // if
    }
}
