<?php
/**
 * i-doit - Documentation and CMDB solution for IT environments
 *
 * This file is part of the i-doit framework. Modify at your own risk.
 *
 * Please visit http://www.i-doit.com/license for a full copyright and license information.
 *
 * @version     1.10
 * @package     i-doit
 * @author      synetics GmbH
 * @copyright   synetics GmbH
 * @url         http://www.i-doit.com
 * @license     http://www.i-doit.com/license
 */

namespace idoit\Console\Command\Import;

use Exception;
use idoit\Component\Helper\Ip;
use idoit\Console\Command\Import\Ocs\Hardware;
use idoit\Console\Command\AbstractCommand;
use idoit\Module\Cmdb\Search\IndexExtension\Signals;
use idoit\Module\Cmdb\Model\Matcher\Ci\CiMatcher;
use idoit\Module\Cmdb\Model\Matcher\Ci\MatchKeyword;
use idoit\Module\Cmdb\Model\Matcher\Identifier\Fqdn;
use idoit\Module\Cmdb\Model\Matcher\Identifier\Hostname;
use idoit\Module\Cmdb\Model\Matcher\Identifier\Mac;
use idoit\Module\Cmdb\Model\Matcher\Identifier\ModelSerial;
use idoit\Module\Cmdb\Model\Matcher\Identifier\ObjectTitle;
use idoit\Module\Cmdb\Model\Matcher\Identifier\ObjectTypeConst;
use idoit\Module\Cmdb\Model\Matcher\MatchConfig;
use isys_cmdb_dao_cable_connection;
use isys_cmdb_dao_category_g_application;
use isys_cmdb_dao_category_g_cpu;
use isys_cmdb_dao_category_g_drive;
use isys_cmdb_dao_category_g_global;
use isys_cmdb_dao_category_g_graphic;
use isys_cmdb_dao_category_g_guest_systems;
use isys_cmdb_dao_category_g_ip;
use isys_cmdb_dao_category_g_memory;
use isys_cmdb_dao_category_g_model;
use isys_cmdb_dao_category_g_network_interface;
use isys_cmdb_dao_category_g_network_port;
use isys_cmdb_dao_category_g_operating_system;
use isys_cmdb_dao_category_g_power_consumer;
use isys_cmdb_dao_category_g_relation;
use isys_cmdb_dao_category_g_sound;
use isys_cmdb_dao_category_g_stack_member;
use isys_cmdb_dao_category_g_stor;
use isys_cmdb_dao_category_g_ui;
use isys_cmdb_dao_category_g_version;
use isys_cmdb_dao_category_g_virtual_machine;
use isys_cmdb_dao_category_s_application;
use isys_cmdb_dao_category_s_net;
use isys_cmdb_dao_connection;
use isys_cmdb_dao_dialog;
use isys_component_signalcollection;
use isys_convert;
use isys_format_json;
use isys_import_handler;
use isys_import_handler_cmdb;
use isys_log;
use isys_module_logbook;
use SplFixedArray;
use Symfony\Component\Console\Input\InputDefinition;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\NullOutput;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Output\StreamOutput;
use Symfony\Component\Console\Formatter\OutputFormatterStyle;
use isys_tenantsettings;
use isys_component_dao_ocs;
use isys_event_manager;
use isys_cmdb_dao_category;
use isys_cmdb_dao;
use isys_component_database;
use isys_helper_crypt;
use idoit\Console\Command\IsysLogWrapper;

class ImportOcsCommand extends AbstractCommand
{
    const NAME = 'import-ocs';

    /**
     * @var \idoit\Module\Cmdb\Model\Matcher\Ci\CiMatcher
     */
    private $ciMatcher = [];

    /**
     * @var []
     */
    private $ocsDatabaseSettings = null;

    /**
     * @var array
     */
    private $categories = [];
    /**
     * @var array
     */
    private $m_already_imported_snmp_devices = [];

    /**
     * @var array
     */
    private $m_objtypes = [];

    /**
     * @var IsysLogWrapper
     */
    private $m_log = null;

    /**
     * @var array
     */
    private $m_objtype_blacklist = [
        C__OBJTYPE__PERSON,
        C__OBJTYPE__PERSON_GROUP,
        C__OBJTYPE__ORGANIZATION
    ];

    /**
     * Get name for command
     *
     * @return string
     */
    public function getCommandName()
    {
        return self::NAME;
    }

    /**
     * Get description for command
     *
     * @return string
     */
    public function getCommandDescription()
    {
        return 'Imports data from an OCS inventory NG server (SQL server access is defined in the GUI)';
    }

    /**
     * Retrieve Command InputDefinition
     *
     * @return InputDefinition
     */
    public function getCommandDefinition()
    {
        $definition = new InputDefinition();

        // Option which determines if all ips and ports of an updated object should be overwritten or not
        $definition->addOption(
            new InputOption(
                'ipPortOverwrite', null, InputOption::VALUE_REQUIRED, 'Determines if hostaddresses and ports should be deleted first for each imported device. 1 = Active; 0 = Inactive (Default)' , 0
            )
        );

        // Option for the default object type which will be used if no default object type has been defined
        $definition->addOption(
            new InputOption(
                'databaseSchema', null, InputOption::VALUE_REQUIRED, 'Import from selected database schema. If not set default database schema will be used in the configuration.'
            )
        );

        // Option for the default object type which will be used if no default object type has been defined
        $definition->addOption(
            new InputOption(
                'objectType', null, InputOption::VALUE_REQUIRED, 'Default objecttype constant from the object type configuration.'
            )
        );

        // Option for a source file which contains hostnames which will be imported/updated.
        $definition->addOption(
            new InputOption(
                'file', null, InputOption::VALUE_REQUIRED, 'File which contains a list of hostnames.'
            )
        );

        // Option for a comma separated list of hostnames which will be imported/updated.
        $definition->addOption(
            new InputOption(
                'hosts', null, InputOption::VALUE_REQUIRED, 'Comma separated list of Hostnames which will be imported.'
            )
        );

        // Option for the switch if snmp devices should be imported or not
        $definition->addOption(
            new InputOption(
                'snmpDevices', null, InputOption::VALUE_NONE, 'Switch to snmp device import.'
            )
        );

        // Option for a comma separated list of categories which will be used for the import
        $definition->addOption(
            new InputOption(
                'categories', null, InputOption::VALUE_REQUIRED, 'Categories which will be imported. Possible Values: drive,ui,sound,application,memory,model,graphic,net,stor,operating_system,cpu'
            )
        );

        // Option which determines the logging level
        $definition->addOption(
            new InputOption(
                'logging', null, InputOption::VALUE_REQUIRED, 'Activate file logging. 1 = Normal Log; 2 = Debug Log', 0
            )
        );

        // Option which lists all possible object types
        $definition->addOption(
            new InputOption(
                'listObjectTypes', null, InputOption::VALUE_NONE, 'Lists all possible object types'
            )
        );

        // Option which lists all possible categories
        $definition->addOption(
            new InputOption(
                'listCategories', null, InputOption::VALUE_NONE, 'Lists all possible categories'
            )
        );

        $definition->addOption(new InputOption(
            'usage',
            null,
            InputOption::VALUE_NONE
        ));

        return $definition;
    }

    /**
     * Checks if a command can have a config file via --config
     *
     * @return bool
     */
    public function isConfigurable()
    {
        return true;
    }

    /**
     * Returns an array of command usages
     *
     * @return string[]
     */
    public function getCommandUsages()
    {
        return [];
    }

    /**
     * @param OutputInterface $output
     * @param isys_cmdb_dao   $dao
     *
     * @author   Van Quyen Hoang <qhoang@i-doit.com>
     */
    private function listObjectTypes(OutputInterface $output, isys_cmdb_dao $dao)
    {
        $output->writeln("<info>Object-Types:</info>");
        $result = $dao->get_types();
        while ($objectTypeData = $result->get_row())
        {
            $output->writeln('  ' . $objectTypeData['isys_obj_type__id'] . ': ' . $objectTypeData['isys_obj_type__const']);
        }
    }

    /**
     * @param OutputInterface $output
     * @param isys_cmdb_dao   $dao
     *
     * @author   Van Quyen Hoang <qhoang@i-doit.com>
     */
    private function listCategories(OutputInterface $output, isys_cmdb_dao $dao)
    {
        $output->writeln("<info>Allowed categories:</info>");
        foreach($this->categories AS $categoryId => $unused)
        {
            if($categoryId == 'operating_system')
            {
                $output->writeln('  operating_system: ' . _L('LC__CATG__OPERATING_SYSTEM'));
                continue;
            }
            $categoryTitle = $dao->get_catg_name_by_id_as_string($categoryId);
            $output->writeln('  ' . $unused . ': ' . _L($categoryTitle));
        }
    }

    /**
     * Prints out the usage of the import handler
     *
     * @param OutputInterface $output
     */
    private function usage(OutputInterface $output, $level = null)
    {
        switch ($level)
        {
            case 2:
                // no hosts selected
                $output->writeln("Please select one or more hosts to be imported.");
                break;
            case 3:
                $output->writeln("<error>Could not connect to OCS Database.</error>");
                $output->writeln("<info>Please check your OCS database configuration.</info>");
                break;
            case 1:
                $output->writeln("<error>Missing database!</error>");
                $output->writeln("<info>Please set a default OCS database or use the option --databaseSchema.</info>");
            default:

                $style = new OutputFormatterStyle('yellow', 'black', []);
                $dao = new isys_cmdb_dao($this->container->database);
                $output->getFormatter()
                    ->setStyle('yellow', $style);

                $output->writeln("<yellow>Usage</yellow>:");
                $output->writeln(" import-ocs [--snmpDevices] [--databaseSchema] [--objectType] [--file|--hosts] [--categories] [--logging]");
                $output->writeln('');
                $output->writeln("Example for importing objects from an OCS inventory database with specified hosts, categories and activated logging lvl 2:");
                $output->writeln("import-ocs --databaseSchema=ocsweb --hosts=device1,device2 --logging=2 --objectType=C__OBJTYPE__CLIENT --categories=model,cpu,ui");
                $output->writeln('');
                $output->writeln("  --databaseSchema: Retrieves the OCS configuration from i-doit via schema name which will be used as import source.");
                $output->writeln("  --hosts: Comma separated list of hosts which will be searched and imported from the OCS database.");
                $output->writeln("  --logging: Specifies the log level of the import.");
                $output->writeln("  --objectType: All newly imported devices which could not be automatically identified are being created with the specified object type. Default from the configuration will be used if not specified.");
                $output->writeln("  --categories: Comma separated list of categories which will be imported.");
                $output->writeln('');

                // Output all possible object types
                $this->listObjectTypes($output, $dao);

                $output->writeln('');

                // Output all possible categories
                $this->listCategories($output, $dao);


                break;
        }
    } // function

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $style = new OutputFormatterStyle('red', 'black', []);
        $output->getFormatter()
            ->setStyle('red', $style);

        $output->writeln("------------------------------------------------");
        $output->writeln("i-do<red>it</red> OCS Import");
        $output->writeln("------------------------------------------------");

        $output->writeln("OCS-Handler initialized (" . date("Y-m-d H:i:s") . ")");

        $this->categories = [
            "operating_system"           => "operating_system",
            C__CATG__CPU                 => "cpu",
            C__CATG__MEMORY              => "memory",
            C__CATG__APPLICATION         => "application",
            C__CATG__NETWORK             => "net",
            C__CATG__STORAGE             => "stor",
            C__CATG__DRIVE               => "drive",
            C__CATG__GRAPHIC             => "graphic",
            C__CATG__SOUND               => "sound",
            C__CATG__MODEL               => "model",
            C__CATG__UNIVERSAL_INTERFACE => "ui"
        ];

        if ($input->getOption('usage'))
        {
            $this->usage($output);

            return;
        }

        /* Process import */
        try
        {

            $ocsDatabaseId = $this->getOcsDatabaseId($input);

            if (!$ocsDatabaseId)
            {
                $this->usage($output, 1);

                return;
            }

            $this->ocsDatabaseSettings = (new isys_component_dao_ocs($this->container->database))->getOCSDB($ocsDatabaseId);

            if (!is_array($this->ocsDatabaseSettings))
            {
                $this->usage($output, 3);

                return;
            }

            $this->process($input, $output);

            return true;
        }
        catch (Exception $e)
        {
            error($e->getMessage());
        }

        return false;
    }

    public function process(InputInterface $input, OutputInterface $output)
    {
        /**
         * Import start time, used to identify the updated objects within this import
         */
        $startTime = microtime(true);

        /**
         * Disconnect the onAfterCategoryEntrySave event to not always reindex the object in every category
         * This is extremely important!
         *
         * An Index is done for all objects at the end of the request, if enabled via checkbox.
         */
        Signals::instance()
            ->disconnectOnAfterCategoryEntrySave();

        $l_config_obj_types = [
            C__OBJTYPE__SERVER  => (isys_tenantsettings::get('ocs.prefix.server') ?: null),
            C__OBJTYPE__CLIENT  => (isys_tenantsettings::get('ocs.prefix.client') ?: null),
            C__OBJTYPE__SWITCH  => (isys_tenantsettings::get('ocs.prefix.switch') ?: null),
            C__OBJTYPE__ROUTER  => (isys_tenantsettings::get('ocs.prefix.router') ?: null),
            C__OBJTYPE__PRINTER => (isys_tenantsettings::get('ocs.prefix.printer') ?: null),
        ];

        $l_regApp         = isys_tenantsettings::get('ocs.application');
        $l_regAppAssign   = isys_tenantsettings::get('ocs.application.assignment');
        $l_logb_active    = isys_tenantsettings::get('ocs.logbook.active');
        $l_cmdb_status_id = isys_tenantsettings::get('ocs.default_status', C__CMDB_STATUS__IN_OPERATION);

        $l_daoCMDB  = new isys_cmdb_dao($this->container->database);
        $l_dao_logb = new isys_module_logbook();

        if (empty($this->ocsDatabaseSettings["isys_ocs_db__host"]))
        {
            $output->writeln("<error>You have to configure the OCS connector first</error>");
            return;
        } // if

        $l_ocsdb = isys_component_database::get_database(
            "mysqli",
            $this->ocsDatabaseSettings["isys_ocs_db__host"],
            $this->ocsDatabaseSettings["isys_ocs_db__port"],
            $this->ocsDatabaseSettings["isys_ocs_db__user"],
            isys_helper_crypt::decrypt($this->ocsDatabaseSettings["isys_ocs_db__pass"]),
            $this->ocsDatabaseSettings["isys_ocs_db__schema"]
        );

        $l_daoOCS = new isys_component_dao_ocs($l_ocsdb);

        $l_gui               = false;
        $l_overwrite_ip_port = false;
        $l_categories        = [];
        $l_log_lvl           = 0;
        $l_objtype_arr       = $l_objtype_snmp_arr = $l_snmpIDs = $l_hardwareIDs = [];

        if (!empty($_SERVER['HTTP_HOST']))
        {
            $l_cached = $this->categories;
            unset($this->categories);
            foreach ($l_cached AS $l_key => $l_val)
            {
                $this->categories[] = $l_val;
            }
            // Import one ocs object
            if (isys_glob_get_param("hardwareID"))
            {
                $l_snmp = html_entity_decode(isys_glob_get_param('snmp'));
                if ($l_snmp && $l_snmp !== 'false')
                {
                    $l_snmpIDs[] = html_entity_decode(isys_glob_get_param("hardwareID"));
                }
                else
                {
                    $l_hardwareIDs[] = html_entity_decode(isys_glob_get_param("hardwareID"));
                }

                $l_objType = html_entity_decode(isys_glob_get_param("objTypeID"));
            }
            else
            {
                $l_hardwareIDs      = html_entity_decode(isys_glob_get_param("id"));
                $l_snmpIDs          = html_entity_decode(isys_glob_get_param("id_snmp"));
                $l_objtype_arr      = html_entity_decode(isys_glob_get_param("objtypes"));
                $l_objtype_snmp_arr = html_entity_decode(isys_glob_get_param("objtypes_snmp"));

                $l_objType = isys_tenantsettings::get('ocs.default.objtype');
            } // if

            // Default Object type as string
            $l_default_objtype = _L($l_daoCMDB->get_objtype_name_by_id_as_string($l_objType));
            $l_log_lvl         = html_entity_decode(isys_glob_get_param("ocs_logging"));

            $l_categories = html_entity_decode(isys_glob_get_param("category"));

            if (!is_array($l_objtype_snmp_arr) && isys_format_json::is_json($l_objtype_snmp_arr))
            {
                $l_objtype_snmp_arr = isys_format_json::decode($l_objtype_snmp_arr);
            }

            if (!is_array($l_objtype_arr) && isys_format_json::is_json($l_objtype_arr))
            {
                $l_objtype_arr = isys_format_json::decode($l_objtype_arr);
            }

            if (isys_format_json::is_json($l_categories))
            {
                $l_categories = isys_format_json::decode($l_categories);
            }

            if (!is_array($l_hardwareIDs) && isys_format_json::is_json($l_hardwareIDs))
            {
                $l_hardwareIDs = isys_format_json::decode($l_hardwareIDs);
            }

            if (!is_array($l_snmpIDs) && isys_format_json::is_json($l_snmpIDs))
            {
                $l_snmpIDs = isys_format_json::decode($l_snmpIDs);
            }

            // @deprecated
            $l_overwrite_ip_port = (bool) html_entity_decode(isys_glob_get_param("overwrite_ip_port"));

            $l_gui = true;
        }
        else
        {

            $l_file              = $input->getOption('file') ?: null;
            $l_hosts             = $input->getOption("hosts") ?: null;
            $l_import_categories = $input->getOption('categories') ?: null;
            $l_snmp              = (bool) $input->getOption("snmpDevices");
            $l_log_lvl           = $input->getOption('logging') ?: 0;
            $l_standardObjType   = $input->getOption('objectType') ?: null;
            $l_overwrite_ip_port = (bool) $input->getOption('ipPortOverwrite');
            $l_hardware          = true;
            $l_arHosts           = $l_temp = [];

            $listObjecttypes = $input->getOption('listObjectTypes');
            $listCategories = $input->getOption('listCategories');

            if($listObjecttypes)
            {
                $this->listObjectTypes($output, $l_daoCMDB);
                return true;
            }

            if($listCategories)
            {
                $this->listCategories($output, $l_daoCMDB);
                return true;
            }

            if ($l_file !== null && $l_hosts !== null)
            {
                $output->writeln('<error>Only one source can be chosen.</error>');

                return false;
            }

            if ($l_file !== null)
            {
                if (!file_exists($l_file))
                {
                    $output->writeln('<error>Input file ' . $l_file . '</error> not found!!');

                    return false;
                }

                $l_arHosts = explode("\n", file_get_contents($l_file));
            }
            elseif ($l_hosts !== null)
            {
                $l_arHosts = explode(',', $l_hosts);
            }

            if (count($l_arHosts) > 0)
            {
                foreach ($l_arHosts as $l_hostname)
                {
                    $l_hostname = trim($l_hostname);
                    if ($l_hostname == "")
                    {
                        continue;
                    }
                    $l_temp[] = $l_hostname;
                }

                if ($l_hardware)
                {
                    $l_resHW = $l_daoOCS->getHardwareIDs($l_temp);

                    while ($l_row = $l_resHW->get_row())
                    {
                        $l_hardwareIDs[] = $l_row["ID"];
                    }
                }
                if ($l_snmp)
                {
                    $l_resHW = $l_daoOCS->getHardwareSnmpIDs($l_temp);

                    while ($l_row = $l_resHW->get_row())
                    {
                        $l_snmpIDs[] = $l_row["ID"];
                    }
                }
            }
            elseif ($l_file === null && $l_hosts === null)
            {
                // Import all Devices from OCS
                if ($l_hardware)
                {
                    $l_res = $l_daoOCS->getHardware();

                    while ($l_row = $l_res->get_row())
                    {
                        $l_hardwareIDs[] = $l_row["ID"];
                    }
                }
                if ($l_snmp)
                {
                    $l_resHW = $l_daoOCS->getHardwareSnmp();

                    $l_already_set = [];
                    while ($l_row = $l_resHW->get_row())
                    {
                        if (isset($l_already_set[$l_row['NAME']])) continue;

                        $l_already_set[$l_row['NAME']] = true;
                        $l_snmpIDs[]                   = $l_row["ID"];
                    }
                }
            }
            else
            {
                // No objects found in the OCS database
                $output->writeln('<info>No objects found.</info>');

                return false;
            }

            if ($l_import_categories)
            {
                $l_import_categories = explode(',', $l_import_categories);
                $l_cached_arr        = array_flip($this->categories);

                foreach ($l_cached_arr AS $l_key => $l_val)
                {
                    if (in_array($l_key, $l_import_categories))
                    {
                        $l_categories[] = $l_val;
                    }
                }

            }
            else
            {
                $l_categories = array_flip($this->categories);
            }

            /* Is an standard object type is setted */
            if ($l_standardObjType !== null)
            {
                if (is_numeric($l_standardObjType))
                {
                    $l_objType = $l_standardObjType;
                }
                elseif (is_string($l_standardObjType))
                {

                    /* Retrieve object type by constant */
                    $l_objType = $l_daoCMDB->get_object_type(null, $l_standardObjType, null);

                    /* Set $l_objType */
                    if (is_array($l_objType))
                    {
                        $l_objType = $l_objType['isys_obj_type__id'];
                    }
                }
            }
            else
            {
                /* Retrieve standard object type by i-doit registry */
                $l_objType = isys_tenantsettings::get('ocs.default.objtype');
            }

            $l_default_objtype = _L($l_daoCMDB->get_objtype_name_by_id_as_string($l_objType));
        }

        $l_logging = true;
        $l_loglevel = null;

        switch ($l_log_lvl)
        {
            case 2:
                $l_loglevel = isys_log::C__ALL;
                break;
            case 1:
                $l_loglevel = isys_log::C__ALL & ~isys_log::C__DEBUG;
                break;
            default:
                $l_logging = false;
                break;
        }

        if (count($l_hardwareIDs) == 0 && count($l_snmpIDs) == 0)
        {
            $this->usage($output, 2);

            return false;
        }

        // initialize objects once
        $l_daoCableCon = isys_cmdb_dao_cable_connection::instance($this->container->database);
        $ocsHardware   = new Hardware();

        $this->m_log = IsysLogWrapper::instance();

        if ($l_logging)
        {
            $this->m_log->setOutput(new StreamOutput(fopen('log/import_ocs_' . date('Y-m-d_H_i_s') . '.log', 'a')));
            $this->m_log
                ->set_log_level($l_loglevel)
                ->set_verbose_level(isys_log::C__FATAL | isys_log::C__ERROR | isys_log::C__WARNING | isys_log::C__NOTICE);
        }
        else
        {
            $this->m_log->setOutput(new NullOutput());
        }

        /**
         * Typehinting:
         *
         * @var  $l_daoGl              isys_cmdb_dao_category_g_global
         * @var  $l_daoNet_s           isys_cmdb_dao_category_s_net
         * @var  $l_daoApp             isys_cmdb_dao_category_g_application
         * @var  $l_daoOS              isys_cmdb_dao_category_g_operating_system
         * @var  $l_daoApp_s           isys_cmdb_dao_category_s_application
         * @var  $l_daoPort            isys_cmdb_dao_category_g_network_port
         * @var  $l_daoIP              isys_cmdb_dao_category_g_ip
         */
        $l_daoGl         = isys_cmdb_dao_category_g_global::instance($this->container->database);
        $l_daoNet_s      = isys_cmdb_dao_category_s_net::instance($this->container->database);
        $l_daoApp        = isys_cmdb_dao_category_g_application::instance($this->container->database);
        $l_daoOS         = isys_cmdb_dao_category_g_operating_system::instance($this->container->database);
        $l_daoApp_s      = isys_cmdb_dao_category_s_application::instance($this->container->database);
        $l_daoPort       = isys_cmdb_dao_category_g_network_port::instance($this->container->database);
        $l_daoIP         = isys_cmdb_dao_category_g_ip::instance($this->container->database);
        $l_dao_interface = isys_cmdb_dao_category_g_network_interface::instance($this->container->database);

        $l_add_cpu             = false;
        $l_add_model           = false;
        $l_add_memory          = false;
        $l_add_application     = false;
        $l_add_graphic         = false;
        $l_add_sound           = false;
        $l_add_storage         = false;
        $l_add_drive           = false;
        $l_add_net             = false;
        $l_add_ui              = false;
        $l_add_os              = false;
        $l_add_virtual_machine = false;

        $l_category_selection_as_string = '';
        if (array_search('operating_system', $l_categories) !== false)
        {
            $l_add_os                       = true;
            $l_category_selection_as_string .= 'Operating System, ';
        }

        if (array_search(C__CATG__CPU, $l_categories) !== false || (array_key_exists(C__CATG__CPU, $l_categories) !== false && !$l_gui))
        {
            $l_add_cpu                      = true;
            $l_daoCPU                       = isys_cmdb_dao_category_g_cpu::instance($this->container->database);
            $l_category_selection_as_string .= 'CPU, ';
        }
        if (array_search(C__CATG__MEMORY, $l_categories) !== false || (array_key_exists(C__CATG__MEMORY, $l_categories) !== false && !$l_gui))
        {
            $l_add_memory                   = true;
            $l_daoMemory                    = isys_cmdb_dao_category_g_memory::instance($this->container->database);
            $l_category_selection_as_string .= 'Memory, ';
        }

        if (array_search(C__CATG__APPLICATION, $l_categories) !== false || (array_key_exists(C__CATG__APPLICATION, $l_categories) !== false && !$l_gui))
        {
            $l_add_application              = true;
            $l_relation_dao                 = isys_cmdb_dao_category_g_relation::instance($this->container->database);
            $l_relation_data                = $l_relation_dao->get_relation_type(C__RELATION_TYPE__SOFTWARE, null, true);
            $l_category_selection_as_string .= 'Software assignment, ';
        }

        if (array_search(C__CATG__NETWORK, $l_categories) !== false || (array_key_exists(C__CATG__NETWORK, $l_categories) !== false && !$l_gui))
        {
            $l_add_net                      = true;
            $l_dao_power_consumer           = isys_cmdb_dao_category_g_power_consumer::instance($this->container->database);
            $l_category_selection_as_string .= 'Network, ';
        }

        if (array_search(C__CATG__STORAGE, $l_categories) !== false || (array_key_exists(C__CATG__STORAGE, $l_categories) !== false && !$l_gui))
        {
            $l_add_storage                  = true;
            $l_daoStor                      = isys_cmdb_dao_category_g_stor::instance($this->container->database);
            $l_category_selection_as_string .= 'Devices, ';
        }

        if (array_search(C__CATG__GRAPHIC, $l_categories) !== false || (array_key_exists(C__CATG__GRAPHIC, $l_categories) !== false && !$l_gui))
        {
            $l_add_graphic                  = true;
            $l_dao_graphic                  = isys_cmdb_dao_category_g_graphic::instance($this->container->database);
            $l_category_selection_as_string .= 'Graphic card, ';
        }

        if (array_search(C__CATG__SOUND, $l_categories) !== false || (array_key_exists(C__CATG__SOUND, $l_categories) !== false && !$l_gui))
        {
            $l_add_sound                    = true;
            $l_dao_sound                    = isys_cmdb_dao_category_g_sound::instance($this->container->database);
            $l_category_selection_as_string .= 'Sound card, ';
        }

        if (array_search(C__CATG__MODEL, $l_categories) !== false || (array_key_exists(C__CATG__MODEL, $l_categories) !== false && !$l_gui))
        {
            $l_add_model                    = true;
            $l_daoModel                     = isys_cmdb_dao_category_g_model::instance($this->container->database);
            $l_daoStackMember               = isys_cmdb_dao_category_g_stack_member::instance($this->container->database);
            $l_category_selection_as_string .= 'Model, ';
        }

        if (array_search(C__CATG__UNIVERSAL_INTERFACE, $l_categories) !== false || (array_key_exists(C__CATG__UNIVERSAL_INTERFACE, $l_categories) !== false && !$l_gui))
        {
            $l_add_ui                       = true;
            $l_daoUI                        = isys_cmdb_dao_category_g_ui::instance($this->container->database);
            $l_category_selection_as_string .= 'Interface, ';
        }

        if (array_search(C__CATG__DRIVE, $l_categories) !== false || (array_key_exists(C__CATG__DRIVE, $l_categories) !== false && !$l_gui))
        {
            $l_add_drive                    = true;
            $l_dao_drive                    = isys_cmdb_dao_category_g_drive::instance($this->container->database);
            $l_category_selection_as_string .= 'Drives, ';
        }

        if (array_search(C__CATG__VIRTUAL_MACHINE, $l_categories) !== false || (array_key_exists(C__CATG__VIRTUAL_MACHINE, $l_categories) !== false && !$l_gui))
        {
            $l_add_virtual_machine = true;
            $l_dao_vm              = isys_cmdb_dao_category_g_virtual_machine::instance($this->container->database);
            $l_dao_gs              = isys_cmdb_dao_category_g_guest_systems::instance($this->container->database);
        }

        if ($l_category_selection_as_string != '')
        {
            $this->m_log->info('Following categories are selected for the import: ' . $l_category_selection_as_string);
        }
        else
        {
            $this->m_log->warning('No categories selected for the import.');
        } // if

        try
        {
            $this->m_log->info('Preparing environment data for the import.');
            $l_daoCMDB->begin_update();

            $l_capacityUnitMB         = $l_daoCMDB->retrieve("SELECT isys_memory_unit__id FROM isys_memory_unit WHERE isys_memory_unit__const = 'C__MEMORY_UNIT__MB'")
                ->get_row_value('isys_memory_unit__id');
            $l_frequency_unit         = $l_daoCMDB->retrieve(
                "SELECT isys_frequency_unit__id FROM isys_frequency_unit WHERE isys_frequency_unit__const = 'C__FREQUENCY_UNIT__GHZ'"
            )
                ->get_row_value('isys_frequency_unit__id');
            $l_model_default_manufact = $l_daoCMDB->retrieve(
                "SELECT isys_model_manufacturer__title FROM isys_model_manufacturer WHERE isys_model_manufacturer__const = 'C__MODEL_NOT_SPECIFIED'"
            )
                ->get_row_value('isys_model_manufacturer__title');

            $l_app_manufacturer = [];
            if ($l_add_application)
            {
                $l_res = $l_daoCMDB->retrieve("SELECT isys_application_manufacturer__id, isys_application_manufacturer__title FROM isys_application_manufacturer");
                while ($l_row = $l_res->get_row())
                {
                    $l_app_manufacturer[$l_row['isys_application_manufacturer__id']] = trim(_L($l_row['isys_application_manufacturer__title']));
                } // while
            } // if

            $l_vm_types = [];
            if ($l_add_virtual_machine)
            {
                $l_res = $l_daoCMDB->retrieve("SELECT isys_vm_type__id, isys_vm_type__title FROM isys_vm_type");
                while ($l_row = $l_res->get_row())
                {
                    $l_vm_types[$l_row['isys_vm_type__id']] = trim(_L($l_row['isys_vm_type__title']));
                } // while
            } // if

            if ($l_add_net)
            {
                $l_resNet_s = $l_daoNet_s->get_data();
                while ($l_rowNetS = $l_resNet_s->get_row())
                {
                    $l_net_address = null;
                    if ($l_rowNetS['isys_cats_net_list__address'] === null || $l_rowNetS['isys_cats_net_list__address'] == '')
                    {
                        $l_net_address = substr($l_rowNetS['isys_obj__title'], 0, strpos($l_rowNetS['isys_obj__title'], '/'));
                        if (!Ip::validate_net_ip($l_net_address))
                        {
                            continue;
                        }
                    }
                    else
                    {
                        $l_net_address = $l_rowNetS['isys_cats_net_list__address'];
                    }
                    if ($l_net_address !== null)
                    {
                        $l_net_arr[$l_net_address] = [
                            'row_data' => $l_rowNetS,
                        ];
                    }
                } // while
            } // if

            $l_conTypeTitle = $l_daoCMDB->retrieve("SELECT isys_ui_con_type__title FROM isys_ui_con_type WHERE isys_ui_con_type__const = 'C__UI_CON_TYPE__OTHER'")
                ->get_row_value('isys_ui_con_type__title');

            if ($l_capacityUnitMB == null)
            {
                $this->m_log->debug("Internal error: ID for capacity unit MB could not be retrieved");
                // throw new Exception("Internal error: ID for capacity unit MB could not be retrieved");
                return;
            }

            if (count($l_hardwareIDs) > 0)
            {
                $output->writeln('<info>Starting Import.</info>');
                $this->m_log->info('Device count: ' . count($l_hardwareIDs));
                foreach ($l_hardwareIDs AS $l_position => $l_hardwareID)
                {
                    $l_objID          = false;
                    $l_object_updated = false;

                    $l_hw_data           = $this->get_hardware_info(
                        $l_daoOCS,
                        $l_hardwareID,
                        $l_add_model,
                        $l_add_memory,
                        $l_add_application,
                        $l_add_graphic,
                        $l_add_sound,
                        $l_add_storage,
                        $l_add_net,
                        $l_add_ui,
                        $l_add_drive,
                        $l_add_virtual_machine
                    );
                    $l_inventory         = $l_hw_data['inventory'];
                    $l_inventory["NAME"] = trim($l_inventory["NAME"]);

                    $l_thisObjTypeID = null;

                    if ($l_inventory == null)
                    {
                        $output->writeln("Object with ID " . $l_hardwareID . " does not exist");
                        $this->m_log->debug("Object with ID \"" . $l_hardwareID . "\" does not exist. Skipping to next device.");
                        continue;
                    }

                    $output->writeln("Processing device: \"" . $l_inventory["NAME"] . "\".");
                    $this->m_log->info("Processing device: \"" . $l_inventory["NAME"] . "\".");

                    // Object matching variables
                    $l_macaddresses = [];
                    $l_serialnumber = null;
                    $l_objectTitle  = $l_inventory["NAME"];

                    // New object, or update existing?
                    if (count($l_inventory['macaddr']))
                    {
                        $this->m_log->debug("MAC-addresses found for " . $l_inventory["NAME"] . ".");
                        $l_macaddresses = $l_inventory['macaddr'];
                    }
                    else
                    {
                        $output->writeln("No MAC-Addresses found for \"" . $l_inventory["NAME"] . "\".");
                        $this->m_log->debug("No MAC-Addresses found for \"" . $l_inventory["NAME"] . "\".");
                    } // if

                    if ($l_inventory['SSN'] != '')
                    {
                        $this->m_log->debug("Serial found for \"" . $l_inventory["NAME"] . "\".");
                        $l_serialnumber = $l_inventory['SSN'];
                    }
                    elseif ($l_inventory["SSN"] == '')
                    {
                        $output->writeln("No Serial found for \"" . $l_inventory["NAME"] . "\".");
                        $this->m_log->debug("No Serial found for \"" . $l_inventory["NAME"] . "\".");
                    } // if

                    if (strpos($l_inventory["NAME"], '.') !== false)
                    {
                        $output->writeln("Possible FQDN found \"" . $l_inventory["NAME"] . "\".");
                        $this->m_log->debug("Possible FQDN found \"" . $l_inventory["NAME"] . "\".");
                    }
                    else
                    {
                        $output->writeln("No possible FQDN found for \"" . $l_inventory["NAME"] . "\".");
                        $this->m_log->debug("No possible FQDN found for \"" . $l_inventory["NAME"] . "\".");
                    } // if

                    $this->m_log->debug("Checking object type.");
                    $l_objectTypeMatch = true;
                    if (isset($l_objtype_arr[$l_position]) && $l_objtype_arr[$l_position] > 0)
                    {
                        $l_thisObjTypeID = $l_objtype_arr[$l_position];
                        $this->m_log->debug(
                            "Object type is set in the dialog. Using selected Object type " . _L(
                                $l_daoCMDB->get_objtype_name_by_id_as_string($l_thisObjTypeID)
                            ) . " for \"" . $l_inventory["NAME"] . "\"."
                        );
                    }
                    else
                    {
                        foreach ($l_config_obj_types AS $l_conf_objtype_id => $l_prefix)
                        {
                            if ($l_thisObjTypeID === null)
                            {
                                $l_prefix_arr = null;
                                if (strpos($l_prefix, ',') !== false)
                                {
                                    $l_prefix_arr = explode(',', $l_prefix);
                                } // if

                                if (is_array($l_prefix_arr))
                                {
                                    foreach ($l_prefix_arr AS $l_sub_prefix)
                                    {
                                        if ($ocsHardware->check_pattern_for_objtype($l_sub_prefix, $l_inventory["TAG"], $l_inventory["NAME"]))
                                        {
                                            $l_thisObjTypeID = $l_conf_objtype_id;
                                            break;
                                        } // if
                                    } // foreach
                                }
                                else
                                {
                                    if ($ocsHardware->check_pattern_for_objtype($l_prefix, $l_inventory["TAG"], $l_inventory["NAME"]))
                                    {
                                        $l_thisObjTypeID = $l_conf_objtype_id;
                                    } // if
                                } // if
                            }
                            else
                            {
                                break;
                            } // if
                        } // foreach

                        // Use Default Object type
                        if ($l_thisObjTypeID === null)
                        {
                            if ($l_objType > 0)
                            {
                                $output->writeln(
                                    "Could not determine object type from configuration. Using default object type: " . $l_default_objtype . "."
                                );
                                $this->m_log->debug(
                                    "Could not determine object type from configuration. Using default object type \"" . $l_default_objtype . "\" for \"" .
                                    $l_inventory["NAME"] . "\"."
                                );
                                $l_thisObjTypeID   = $l_objType;
                                $l_objectTypeMatch = false;
                            }
                            else
                            {
                                $output->writeln("No default object type has been defined.");
                            } // if
                        } // if
                    }

                    $l_objtype_const = null;

                    if ($l_thisObjTypeID > 0)
                    {
                        if (isset($this->m_objtypes[$l_thisObjTypeID]))
                        {
                            $l_objtype_const = $this->m_objtypes[$l_thisObjTypeID];
                        }
                        else
                        {
                            $this->m_objtypes[$l_thisObjTypeID] = $l_objtype_const = $l_daoCMDB->get_object_type($l_thisObjTypeID)['isys_obj_type__const'];
                        } // if
                    } // if

                    // Get object id by object matching
                    $l_objID = $this->get_object_id_by_matching($l_serialnumber, $l_macaddresses, $l_objectTitle, $l_objtype_const, null, null, $output);

                    // first check
                    if ($l_objID)
                    {
                        $this->m_log->info("Object found. Updating object-ID: " . $l_objID);
                        $l_object_updated = true;
                        // Update existing object
                        $l_row           = $l_daoCMDB->get_object_by_id($l_objID)
                            ->get_row();
                        $l_thisObjTypeID = $l_row["isys_obj__isys_obj_type__id"];

                        if ($l_thisObjTypeID > 0 && in_array($l_thisObjTypeID, $this->m_objtype_blacklist))
                        {
                            $l_error_msg = "Error: Import object '" . $l_inventory["NAME"] . "' (#" . $l_objID . ") is of blacklisted object type '" . _L(
                                    $l_daoCMDB->get_objtype_name_by_id_as_string(intval($l_row['isys_obj_type__id'])) . "'"
                                );

                            $output->writeln($l_error_msg);
                            $this->m_log->debug($l_error_msg);
                            continue;
                        }
                        $l_update_msg = "Updating existing object " . $l_inventory["NAME"] . " (" . _L(
                                $l_daoCMDB->get_objtype_name_by_id_as_string(intval($l_row['isys_obj_type__id']))
                            ) . ")";
                        $output->writeln($l_update_msg);

                        $l_update_title = '';
                        if ($l_row['isys_obj__title'] !== $l_inventory['NAME'])
                        {
                            $this->m_log->debug("Object title is differnt: " . $l_row['isys_obj__title'] . " (i-doit) !== " . $l_inventory["NAME"] . " (OCS).");
                            if (is_array($l_daoGl->validate(['title' => $l_inventory['NAME']])) || isys_tenantsettings::get('cmdb.unique.object-title'))
                            {
                                $l_title        = $l_daoCMDB->generate_unique_obj_title($l_inventory['NAME']);
                                $l_update_title = "isys_obj__title = " . $l_daoCMDB->convert_sql_text($l_title) . ", ";
                            }
                            else
                            {
                                $l_update_title = "isys_obj__title = " . $l_daoCMDB->convert_sql_text($l_inventory['NAME']) . ", ";
                            } // if
                        } // if

                        $l_update = "UPDATE isys_obj SET " . $l_update_title . "isys_obj__hostname = " . $l_daoCMDB->convert_sql_text(
                                $l_inventory["NAME"]
                            ) . ", " . "isys_obj__updated     =    NOW(), " . "isys_obj__updated_by  =    '" . $this->container->session->get_current_username() . "', " .
                            "isys_obj__imported    =    NOW() ";

                        if (isset($l_objtype_arr[$l_position]) && $l_objtype_arr[$l_position] > 0)
                        {
                            $this->m_log->debug("Updating object with object type: " . $l_daoCMDB->get_objtype_name_by_id_as_string($l_objtype_arr[$l_position]) . ".");
                            $l_update .= ", isys_obj__isys_obj_type__id = " . $l_daoCMDB->convert_sql_id($l_objtype_arr[$l_position]) . " ";
                        } // if

                        /**
                         * Main object data
                         */
                        $l_daoCMDB->update(
                            $l_update . "WHERE isys_obj__id = " . $l_daoCMDB->convert_sql_id($l_row["isys_obj__id"]) . ";"
                        );
                    }
                    else
                    {
                        if ($l_thisObjTypeID === null)
                        {
                            $output->writeln("Object type could not be identified and no default object type is set in the configuration. Skipping device!!");
                            continue;
                        }

                        $output->writeln("Creating new object " . $l_inventory["NAME"] . " (" . _L($l_daoCMDB->get_objtype_name_by_id_as_string($l_thisObjTypeID)) . ")");
                        $this->m_log->info(
                            "Object not found. Creating new object " . $l_inventory["NAME"] . " (" . _L($l_daoCMDB->get_objtype_name_by_id_as_string($l_thisObjTypeID)) . ")"
                        );

                        $l_objID = $l_daoCMDB->insert_new_obj(
                            $l_thisObjTypeID,
                            false,
                            $l_inventory["NAME"],
                            null,
                            C__RECORD_STATUS__NORMAL,
                            $l_inventory["NAME"],
                            null,
                            true,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            $l_cmdb_status_id
                        );

                        isys_event_manager::getInstance()
                            ->triggerCMDBEvent("C__LOGBOOK_EVENT__OBJECT_CREATED", "-object imported from OCS-", $l_objID, $l_thisObjTypeID);
                    }

                    /*
                     * Clear categories hostaddress and port
                     */
                    if ($l_overwrite_ip_port)
                    {
                        $l_daoCMDB->clear_data($l_objID, 'isys_catg_ip_list', true);
                        $l_daoCMDB->clear_data($l_objID, 'isys_catg_port_list', true);
                        $output->writeln("Categories hostaddress and port cleared.", true);
                        $this->m_log->debug("Categories hostaddress and port cleared.");
                    } // if

                    /**
                     * Model
                     */
                    if (isset($l_hw_data['model']) && isset($l_daoModel))
                    {
                        $output->writeln("Processing model");
                        $this->m_log->info("Processing model");

                        $ocsHardware->handleHardwareModel($this->container, $l_daoModel, $l_dao_logb, $l_objID, $l_thisObjTypeID, $l_hw_data['model'], $l_logb_active);
                    }

                    /**
                     * Processors
                     */
                    if ($l_add_cpu && isset($l_daoCPU))
                    {
                        $output->writeln("Processing CPUs");
                        $this->m_log->info("Processing CPUs");

                        $ocsHardware->handleHardwareCpu($this->container, $l_daoCPU, $l_dao_logb, $l_objID, $l_thisObjTypeID, $l_inventory, $l_frequency_unit, $l_logb_active);
                    }

                    /**
                     * Memory
                     */
                    if (isset($l_hw_data['memory']) && isset($l_daoMemory))
                    {
                        $output->writeln("Processing memory");
                        $this->m_log->info("Processing memory");

                        $ocsHardware->handleHardwareMemory(
                            $this->container,
                            $l_daoMemory,
                            $l_dao_logb,
                            $l_objID,
                            $l_thisObjTypeID,
                            $l_hw_data['memory'],
                            $l_capacityUnitMB,
                            $l_logb_active
                        );
                    }

                    /**
                     * Operating system
                     */
                    if ($l_add_os && isset($l_daoOS))
                    {
                        $output->writeln("Processing Operating system");
                        $this->m_log->info("Processing Operating system");

                        $ocsHardware->handleHardwareOperatingSystem($this->container, $l_daoOS, $l_dao_logb, $l_objID, $l_thisObjTypeID, $l_inventory, $l_logb_active);
                    } // if

                    /**
                     * Applications
                     */
                    if (isset($l_hw_data['application']) && isset($l_daoApp) && isset($l_relation_dao))
                    {
                        $output->writeln("Processing applications");
                        $this->m_log->info("Processing applications");

                        $ocsHardware->handleHardwareApplications(
                            $this->container,
                            $l_daoApp,
                            $l_daoApp_s,
                            $l_relation_dao,
                            $l_dao_logb,
                            $l_objID,
                            $l_thisObjTypeID,
                            $l_hw_data['application'],
                            $l_app_manufacturer,
                            $l_relation_data,
                            $l_inventory["NAME"],
                            $l_logb_active
                        );
                    } // if

                    /**
                     * Graphics adapter
                     */
                    if (isset($l_hw_data['graphic']) && isset($l_dao_graphic))
                    {
                        $output->writeln("Processing graphics adapter");
                        $this->m_log->info("Processing graphics adapter");

                        $ocsHardware->handleHardwareGraphic(
                            $this->container,
                            $l_dao_graphic,
                            $l_dao_logb,
                            $l_objID,
                            $l_thisObjTypeID,
                            $l_hw_data['graphic'],
                            $l_capacityUnitMB,
                            $l_logb_active
                        );
                    }

                    /**
                     * Sound adapter
                     */
                    if (isset($l_hw_data['sound']) && isset($l_dao_sound))
                    {
                        $output->writeln("Processing sound adapter");
                        $this->m_log->info("Processing sound adapter");

                        $ocsHardware->handleHardwareSound($this->container, $l_dao_sound, $l_dao_logb, $l_objID, $l_thisObjTypeID, $l_hw_data['sound'], $l_logb_active);
                    }

                    /**
                     * Drives
                     */
                    if (isset($l_hw_data['drive']) && isset($l_dao_drive))
                    {
                        $output->writeln("Processing drives");
                        $this->m_log->info("Processing drives");

                        $ocsHardware->handleHardwareDrive($this->container, $l_dao_drive, $l_dao_logb, $l_objID, $l_thisObjTypeID, $l_hw_data['drive'], $l_logb_active);
                    } // if

                    /**
                     * Network
                     */
                    if (isset($l_hw_data['net']))
                    {
                        $output->writeln("Processing network");
                        $this->m_log->info("Processing network");
                        unset($l_check_ip, $l_check_net, $l_check_iface, $l_check_port, $l_already_imported_ips);

                        $ocsHardware->handleHardwareNet(
                            $this->container,
                            $l_daoIP,
                            $l_daoPort,
                            $l_dao_interface,
                            $l_dao_logb,
                            $l_objID,
                            $l_thisObjTypeID,
                            $l_hw_data['net'],
                            $l_inventory['NAME'],
                            $l_net_arr,
                            $l_logb_active
                        );
                    }

                    /**
                     * Universal interfaces
                     */
                    if (isset($l_hw_data['ui']) && isset($l_daoUI))
                    {
                        $output->writeln("Processing UI");
                        $this->m_log->info("Processing UI");

                        $ocsHardware->handleHardwareUi($this->container, $l_daoUI, $l_dao_logb, $l_objID, $l_thisObjTypeID, $l_hw_data['ui'], $l_conTypeTitle, $l_logb_active);
                    }

                    /**
                     * Storage
                     */
                    if (isset($l_hw_data['stor']) && isset($l_daoStor))
                    {
                        $output->writeln("Processing storage");
                        $this->m_log->info("Processing storage");

                        $ocsHardware->handleHardwareStorage(
                            $this->container,
                            $l_daoStor,
                            $l_dao_logb,
                            $l_objID,
                            $l_thisObjTypeID,
                            $l_hw_data['stor'],
                            $l_capacityUnitMB,
                            $l_logb_active
                        );
                    }

                    if ($l_object_updated === true)
                    {
                        $output->writeln($l_inventory["NAME"] . " succesfully updated\n\n");
                        $this->m_log->info("\"" . $l_inventory["NAME"] . "\" succesfully updated.");
                    }
                    else
                    {
                        $output->writeln($l_inventory["NAME"] . " succesfully imported\n\n");
                        $this->m_log->info("\"" . $l_inventory["NAME"] . "\" succesfully imported.");
                    } // if

                    $this->m_log->flush_log(true, false);
                }
            }

            /**
             * @todo refactor snmp devices like hardware devices
             * @TODO Memory in ocs database the value is puzzling
             */
            if (count($l_snmpIDs) > 0)
            {
                $this->m_log->info("Found " . count($l_snmpIDs) . " SNMP devices.");
                $this->m_log->info("Starting import for SNMP devices.");
                $l_object_ids = [];
                foreach ($l_snmpIDs AS $l_position => $l_snmp_id)
                {
                    if (!($l_hw_data = $this->get_snmp_info(
                        $l_daoOCS,
                        $l_snmp_id,
                        $l_add_memory,
                        $l_add_storage,
                        $l_add_net,
                        $l_add_cpu,
                        $l_add_ui,
                        $l_add_model,
                        $l_add_application,
                        $l_add_graphic,
                        $l_add_sound,
                        $l_add_drive,
                        $l_add_virtual_machine
                    )))
                    {
                        // Device has already been imported
                        continue;
                    } // if

                    $l_port_descriptions = $l_port_connections = [];

                    $l_thisObjTypeID   = null;
                    $l_object_updated  = false;
                    $l_serialnumber    = null;
                    $l_objectTitle     = null;
                    $l_macaddresses    = [];
                    $l_objectTypeMatch = true;

                    $l_inventory   = $l_hw_data['inventory'];
                    $l_objectTitle = $l_inventory["NAME"] = trim($l_inventory["NAME"]);

                    if ($l_inventory == null)
                    {
                        $output->writeln("Object with ID " . $l_snmp_id . " does not exist");
                        $this->m_log->debug("Object wit ID " . $l_snmp_id . " does not exist. Skipping to next device.");
                        continue;
                    } // if

                    if (isset($l_hw_data['model']) && count($l_hw_data['model']) === 1)
                    {
                        $l_inventory['SSN'] = $l_hw_data['model'][0]['SERIALNUMBER'];
                    } // if

                    if (isset($l_inventory['SSN']))
                    {
                        $this->m_log->debug("Serialnumber found for " . $l_inventory["NAME"] . ".");
                        $l_serialnumber = $l_inventory['SSN'];
                    }
                    else
                    {
                        if (count($l_hw_data['model']) > 1)
                        {
                            $output->writeln("Could not identify Serialnumber for device \"" . $l_inventory["NAME"] . "\".");
                            $this->m_log->debug("Could not identify Serialnumber for device \"" . $l_inventory["NAME"] . "\".");
                        }
                        else
                        {
                            $output->writeln("No Serialnumber found for \"" . $l_inventory["NAME"] . "\".");
                            $this->m_log->debug("No Serialnumber found for \"" . $l_inventory["NAME"] . "\".");
                        } // if
                    } // if

                    // New object, or update existing?
                    if (count($l_inventory['macaddr']))
                    {
                        $this->m_log->debug("MAC-addresses found for " . $l_inventory["NAME"] . ".");
                        $l_macaddresses = $l_inventory['macaddr'];
                    }
                    else
                    {
                        $output->writeln("No MAC-Addresses found for \"" . $l_inventory["NAME"] . "\".");
                        $this->m_log->debug("No MAC-Addresses found for \"" . $l_inventory["NAME"] . "\".");
                    } // if

                    if (strpos($l_inventory["NAME"], '.') !== false)
                    {
                        $output->writeln("Possible FQDN found \"" . $l_inventory["NAME"] . "\".");
                        $this->m_log->debug("Possible FQDN found \"" . $l_inventory["NAME"] . "\".");
                    }
                    else
                    {
                        $output->writeln("No possible FQDN found for \"" . $l_inventory["NAME"] . "\".");
                        $this->m_log->debug("No possible FQDN found for \"" . $l_inventory["NAME"] . "\".");
                    }

                    if (isset($l_objtype_snmp_arr[$l_position]) && $l_objtype_snmp_arr[$l_position] > 0)
                    {
                        $this->m_log->debug(
                            "Object type is set in dialog. Using \"" . $l_daoCMDB->get_objtype_name_by_id_as_string(
                                $l_objtype_snmp_arr[$l_position]
                            ) . "\" as object type."
                        );
                        $l_thisObjTypeID = $l_objtype_snmp_arr[$l_position];
                    }
                    else
                    {
                        foreach ($l_config_obj_types AS $l_conf_objtype_id => $l_prefix)
                        {
                            if ($l_thisObjTypeID === null)
                            {
                                $l_prefix_arr = null;
                                if (strpos($l_prefix, ',') !== false)
                                {
                                    $l_prefix_arr = explode(',', $l_prefix);
                                } // if

                                if (is_array($l_prefix_arr))
                                {
                                    foreach ($l_prefix_arr AS $l_sub_prefix)
                                    {
                                        if ($ocsHardware->check_pattern_for_objtype($l_sub_prefix, $l_inventory["TAG"], $l_inventory["NAME"]))
                                        {
                                            $l_thisObjTypeID = $l_conf_objtype_id;
                                            break;
                                        } // if
                                    } // foreach
                                }
                                else
                                {
                                    if ($ocsHardware->check_pattern_for_objtype($l_prefix, $l_inventory["TAG"], $l_inventory["NAME"]))
                                    {
                                        $l_thisObjTypeID = $l_conf_objtype_id;
                                    } // if
                                } // if
                            }
                            else
                            {
                                break;
                            } // if
                        } // foreach

                        if ($l_thisObjTypeID === null)
                        {
                            // Get Object type from ocs
                            switch ($l_inventory['OBJTYPE'])
                            {
                                case 'blade':
                                    $l_thisObjTypeID = C__OBJTYPE__BLADE_CHASSIS;
                                    break;
                                case 'printer':
                                    $l_thisObjTypeID = C__OBJTYPE__PRINTER;
                                    break;
                                case 'switch':
                                    $l_thisObjTypeID = C__OBJTYPE__SWITCH;
                                    break;
                                case 'router':
                                    $l_thisObjTypeID = C__OBJTYPE__ROUTER;
                                    break;
                                case 'server':
                                    $l_thisObjTypeID = C__OBJTYPE__SERVER;
                                    break;
                                default:
                                    $l_thisObjTypeID = $l_objType;
                                    break;
                            }
                            $output->writeln(
                                "Could not determine object type from configuration. Using object type " . $l_inventory['OBJTYPE'] . " with object type id " .
                                $l_thisObjTypeID . "."
                            );
                            $this->m_log->debug(
                                "Could not determine object type from configuration. Using object type " . $l_daoCMDB->get_objtype_name_by_id_as_string(
                                    $l_inventory['OBJTYPE']
                                ) . " with object type id " . $l_daoCMDB->get_objtype_name_by_id_as_string($l_thisObjTypeID) . "."
                            );

                            $l_objectTypeMatch = false;
                        }
                    }

                    $l_objtype_const = null;

                    if ($l_thisObjTypeID > 0)
                    {
                        if (isset($this->m_objtypes[$l_thisObjTypeID]))
                        {
                            $l_objtype_const = $this->m_objtypes[$l_thisObjTypeID];
                        }
                        else
                        {
                            $this->m_objtypes[$l_thisObjTypeID] = $l_objtype_const = $l_daoCMDB->get_object_type($l_thisObjTypeID)['isys_obj_type__const'];
                        } // if
                    } // if

                    // Get object id by object matching
                    $l_objID = $this->get_object_id_by_matching($l_serialnumber, $l_macaddresses, $l_objectTitle, $l_objtype_const, null, null, $output);

                    if ($l_objID)
                    {
                        $this->m_log->info("Object found. Updating object-ID: " . $l_objID);
                        $l_object_updated = true;
                        // Update existing object
                        $l_row           = $l_daoCMDB->get_object_by_id($l_objID)
                            ->get_row();
                        $l_thisObjTypeID = $l_row["isys_obj__isys_obj_type__id"];

                        $l_update_msg = "Updating existing object " . $l_inventory["NAME"] . " (" . _L(
                                $l_daoCMDB->get_objtype_name_by_id_as_string(intval($l_row['isys_obj_type__id']))
                            ) . ")";

                        $output->writeln($l_update_msg);
                        $this->m_log->debug($l_update_msg);
                        $l_update_title = '';

                        /**
                         * Main object data
                         */
                        if ($l_row['isys_obj__title'] !== $l_inventory['NAME'])
                        {
                            $this->m_log->debug("Object title is differnt: \"" . $l_row['isys_obj__title'] . "\" (i-doit) !== \"" . $l_inventory["NAME"] . "\" (OCS).");
                            if (is_array($l_daoGl->validate(['title' => $l_inventory['NAME']])) || isys_tenantsettings::get('cmdb.unique.object-title'))
                            {
                                $l_title        = $l_daoCMDB->generate_unique_obj_title($l_inventory['NAME']);
                                $l_update_title = "isys_obj__title = " . $l_daoCMDB->convert_sql_text($l_title) . ", ";
                            }
                            else
                            {
                                $l_update_title = "isys_obj__title = " . $l_daoCMDB->convert_sql_text($l_inventory['NAME']) . ", ";
                            } // if
                        } // if

                        $l_update = "UPDATE isys_obj SET " . $l_update_title . "isys_obj__hostname = " . $l_daoCMDB->convert_sql_text(
                                $l_inventory["NAME"]
                            ) . ", " . "isys_obj__updated     =    NOW(), " . "isys_obj__updated_by  =    '" . $this->container->session->get_current_username() . "', " .
                            "isys_obj__imported    =    NOW() ";

                        if ($l_row['isys_obj__description'] == '')
                        {
                            $l_update .= ", isys_obj__description = " . $l_daoCMDB->convert_sql_text($l_inventory['DESCRIPTION']) . " ";
                        } // if

                        if (isset($l_objtype_snmp_arr[$l_position]) && $l_objtype_snmp_arr[$l_position] > 0)
                        {
                            $this->m_log->debug("Updating object with object type: " . $l_daoCMDB->get_objtype_name_by_id_as_string($l_objtype_snmp_arr[$l_position]) . ".");
                            $l_update .= ", isys_obj__isys_obj_type__id = " . $l_daoCMDB->convert_sql_id($l_objtype_snmp_arr[$l_position]) . " ";
                        } // if

                        $l_daoCMDB->update(
                            $l_update . " WHERE isys_obj__id    =    " . $l_daoCMDB->convert_sql_text($l_row["isys_obj__id"]) . ";"
                        );
                    }
                    else
                    {
                        if ($l_thisObjTypeID === null)
                        {
                            $output->writeln("Object type could not be identified and no default object type is set in the configuration. Skipping device!!\n");
                            continue;
                        }

                        $output->writeln("Creating new object " . $l_inventory["NAME"] . " (" . _L($l_daoCMDB->get_objtype_name_by_id_as_string($l_thisObjTypeID)) . ")");
                        $this->m_log->info(
                            "Object not found. Creating new object " . $l_inventory["NAME"] . " (" . _L($l_daoCMDB->get_objtype_name_by_id_as_string($l_thisObjTypeID)) . ")"
                        );

                        $l_objID = $l_daoCMDB->insert_new_obj(
                            $l_thisObjTypeID,
                            false,
                            $l_inventory["NAME"],
                            null,
                            C__RECORD_STATUS__NORMAL,
                            $l_inventory["NAME"],
                            null,
                            true,
                            null,
                            null,
                            null,
                            null,
                            null,
                            null,
                            $l_cmdb_status_id
                        );

                        isys_event_manager::getInstance()
                            ->triggerCMDBEvent("C__LOGBOOK_EVENT__OBJECT_CREATED", "-object imported from OCS-", $l_objID, $l_thisObjTypeID);
                    }

                    /*
                     * Clear categories hostaddress and port
                     */
                    if ($l_overwrite_ip_port)
                    {
                        $l_daoCMDB->clear_data($l_objID, 'isys_catg_ip_list', true);
                        $l_daoCMDB->clear_data($l_objID, 'isys_catg_port_list', true);
                        $output->writeln("Categories hostaddress and port cleared.", true);
                        $this->m_log->info("Categories hostaddress and port cleared.");
                    } // if

                    $l_object_ids[] = $l_objID;

                    /**
                     * Model
                     */
                    if (isset($l_hw_data['model']))
                    {
                        $output->writeln("Processing model");
                        $this->m_log->info("Processing model");
                        $l_data = [];

                        $l_rowModel = null;
                        $l_row      = $l_hw_data['model'];

                        if (count($l_row) > 1)
                        {
                            // Its a chassis
                            $l_assigned_switch_stack = [];
                            foreach ($l_row AS $l_switch_stack)
                            {
                                $l_obj_id_switch_stack = $this->get_object_id_by_matching(
                                    $l_switch_stack['SERIALNUMBER'],
                                    [],
                                    $l_inventory["NAME"] . ' - ' . $l_switch_stack['TITLE'],
                                    null,
                                    (ModelSerial::getBit() + ObjectTitle::getBit()),
                                    2,
                                    $output
                                );
                                $l_import_mode         = isys_import_handler_cmdb::C__UPDATE;

                                if (!$l_obj_id_switch_stack)
                                {
                                    $l_obj_id_switch_stack = $l_daoCMDB->insert_new_obj(
                                        C__OBJTYPE__SWITCH,
                                        null,
                                        $l_inventory["NAME"] . ' - ' . $l_switch_stack['TITLE'],
                                        null,
                                        C__RECORD_STATUS__NORMAL
                                    );
                                    $l_import_mode         = isys_import_handler_cmdb::C__CREATE;
                                } // if

                                $l_objModelData = $l_daoModel->get_data(null, $l_obj_id_switch_stack)
                                    ->get_row() ?: null;

                                $l_data['manufacturer'] = $l_switch_stack["MANUFACTURER"] = (trim(
                                    $l_switch_stack["MANUFACTURER"]
                                ) == '' ? $l_model_default_manufact : trim($l_switch_stack["MANUFACTURER"]));
                                $l_data['serial']       = $l_switch_stack["SERIALNUMBER"];
                                $l_data['title']        = $l_switch_stack["DESCRIPTION"];
                                $l_data['firmware']     = ($l_switch_stack["FIRMVERSION"] != '') ? $l_switch_stack["FIRMVERSION"] : $l_switch_stack["FIRMVERSION_IOS"];
                                $l_data['data_id']      = $l_objModelData['isys_catg_model_list__id'] ?: null;

                                if ($l_logb_active)
                                {
                                    $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                        'manufacturer' => ['title_lang' => $l_switch_stack["MANUFACTURER"]],
                                        'title'        => ['title_lang' => $l_switch_stack["DESCRIPTION"]],
                                        'serial'       => ['value' => $l_switch_stack["SERIALNUMBER"]],
                                        'firmware'     => [
                                            'value' => (($l_switch_stack["FIRMVERSION"] != '') ? $l_switch_stack["FIRMVERSION"] : $l_switch_stack["FIRMVERSION_IOS"])
                                        ]
                                    ];
                                    $l_changes                                                  = $l_dao_logb->prepare_changes(
                                        $l_daoModel,
                                        $l_objModelData,
                                        $l_category_values
                                    );
                                    if (count($l_changes) > 0)
                                    {
                                        isys_event_manager::getInstance()
                                            ->triggerCMDBEvent(
                                                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                "-modified from OCS-",
                                                $l_obj_id_switch_stack,
                                                C__OBJTYPE__SWITCH,
                                                'LC__CMDB__CATG__MODEL',
                                                serialize($l_changes)
                                            );
                                    }
                                }
                                $l_import_data = $l_daoModel->parse_import_array($l_data);
                                $l_daoModel->sync($l_import_data, $l_obj_id_switch_stack, $l_import_mode);
                                $l_assigned_switch_stack[$l_obj_id_switch_stack] = $l_obj_id_switch_stack;
                            } // foreach

                            $l_stackMemberResult = $l_daoStackMember->get_connected_objects($l_objID);
                            if (count($l_stackMemberResult) > 0)
                            {
                                while ($l_stackMember = $l_stackMemberResult->get_row())
                                {
                                    if (isset($l_assigned_switch_stack[$l_stackMember['isys_catg_stack_member_list__stack_member']]))
                                    {
                                        unset($l_assigned_switch_stack[$l_stackMember['isys_catg_stack_member_list__stack_member']]);
                                    } // if
                                } // while
                            } // if

                            if (count($l_assigned_switch_stack) > 0)
                            {
                                foreach ($l_assigned_switch_stack AS $l_newStackMember)
                                {
                                    $l_daoStackMember->create_data(['isys_obj__id' => $l_objID, 'assigned_object' => $l_newStackMember, 'mode' => null]);
                                } // foreach
                            } // if
                        }
                        else
                        {
                            $l_res = $l_daoModel->get_data(null, $l_objID);
                            if ($l_res->num_rows() < 1)
                            {
                                $l_data['description'] = null;
                            }
                            else
                            {
                                $l_rowModel            = $l_res->get_row();
                                $l_data['description'] = $l_rowModel["isys_catg_model_list__description"];
                                $l_data['data_id']     = $l_rowModel['isys_catg_model_list__id'];
                            }
                            $l_data['manufacturer'] = $l_row[0]["MANUFACTURER"] = (trim($l_row[0]["MANUFACTURER"]) == '' ? $l_model_default_manufact : trim(
                                $l_row[0]["MANUFACTURER"]
                            ));
                            $l_data['title']        = $l_row[0]['DESCRIPTION'];
                            $l_data['serial']       = $l_row[0]["SERIALNUMBER"];
                            $l_data['firmware']     = ($l_row[0]["FIRMVERSION"] !== '') ? $l_row[0]["FIRMVERSION"] : $l_row[0]['FIRMVERSION_IOS'];

                            if ($l_logb_active)
                            {
                                $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                    'manufacturer' => ['title_lang' => $l_row[0]["MANUFACTURER"]],
                                    'title'        => ['title_lang' => $l_row[0]["DESCRIPTION"]],
                                    'serial'       => ['value' => $l_row[0]["SERIALNUMBER"]],
                                    'firmware'     => ['value' => (($l_row[0]["FIRMVERSION"] !== '') ? $l_row[0]["FIRMVERSION"] : $l_row[0]['FIRMVERSION_IOS'])]
                                ];
                                $l_changes                                                  = $l_dao_logb->prepare_changes($l_daoModel, $l_rowModel, $l_category_values);
                                if (count($l_changes) > 0)
                                {
                                    isys_event_manager::getInstance()
                                        ->triggerCMDBEvent(
                                            'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                            "-modified from OCS-",
                                            $l_objID,
                                            $l_thisObjTypeID,
                                            'LC__CMDB__CATG__MODEL',
                                            serialize($l_changes)
                                        );
                                }
                            }
                            $l_import_data = $l_daoModel->parse_import_array($l_data);
                            $l_entry_id    = $l_daoModel->sync(
                                $l_import_data,
                                $l_objID,
                                ((!empty($l_data['data_id'])) ? isys_import_handler_cmdb::C__UPDATE : isys_import_handler_cmdb::C__CREATE)
                            );

                            // Emit category signal (afterCategoryEntrySave).
                            isys_component_signalcollection::get_instance()
                                ->emit(
                                    "mod.cmdb.afterCategoryEntrySave",
                                    $l_daoModel,
                                    $l_entry_id,
                                    !!$l_entry_id,
                                    $l_objID,
                                    $l_import_data,
                                    isset($l_changes) ? $l_changes : []
                                );
                        }
                        unset($l_data);
                    }

                    if (isset($l_hw_data['cpu']))
                    {
                        $output->writeln("Processing CPUs");
                        $this->m_log->info("Processing CPUs");
                        $l_data = $l_check_data = [];

                        $l_res  = $l_daoCPU->get_data(null, $l_objID);
                        $l_cpus = $l_res->num_rows();

                        if ($l_cpus > 0)
                        {
                            // Get data in i-doit
                            while ($l_rowCPU = $l_res->get_row())
                            {
                                $l_check_data[]                                    = [
                                    'data_id'        => $l_rowCPU['isys_catg_cpu_list__id'],
                                    'title'          => $l_rowCPU["isys_catg_cpu_list__title"],
                                    'manufacturer'   => $l_rowCPU['isys_catg_cpu_manufacturer__title'],
                                    'type'           => $l_rowCPU['isys_catg_cpu_type__title'],
                                    'frequency'      => $l_rowCPU['isys_catg_cpu_list__frequency'],
                                    'frequency_unit' => $l_rowCPU['isys_catg_cpu_list__isys_frequency_unit__id'],
                                    'cores'          => $l_rowCPU['isys_catg_cpu_list__cores'],
                                    'description'    => $l_rowCPU['isys_catg_cpu_list__description']
                                ];
                                $l_rowCPU_arr[$l_rowCPU['isys_catg_cpu_list__id']] = $l_rowCPU;
                            }

                            if (count($l_hw_data['cpu']) > 0)
                            {
                                foreach ($l_hw_data['cpu'] AS $l_rowCPU)
                                {
                                    // Check data from ocs with data from i-doit
                                    foreach ($l_check_data AS $l_key => $l_val)
                                    {
                                        if ($l_val['type'] == $l_rowCPU['TYPE'] && $l_val['manufacturer'] == $l_rowCPU['MANUFACTURER'] &&
                                            $l_val['frequency'] == isys_convert::frequency(
                                                $l_rowCPU['SPEED'],
                                                $l_frequency_unit
                                            ))
                                        {
                                            unset($l_check_data[$l_key]);
                                            continue 2;
                                        }
                                    }

                                    // Raw array for preparing the import array
                                    $l_data = [
                                        'title'          => null,
                                        'manufacturer'   => ['title_lang' => $l_rowCPU['MANUFACTURER']],
                                        'type'           => ['title_lang' => $l_rowCPU['TYPE']],
                                        'frequency_unit' => $l_frequency_unit,
                                        'cores'          => 1,
                                        'frequency'      => $l_rowCPU['SPEED']
                                    ];

                                    if ($l_logb_active)
                                    {
                                        $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                            'manufacturer'   => ['title_lang' => $l_rowCPU['MANUFACTURER']],
                                            'frequency'      => ['value' => $l_rowCPU['SPEED']],
                                            'type'           => ['title_lang' => $l_rowCPU['TYPE']],
                                            'frequency_unit' => ['title_lang' => $l_rowCPU['UNIT']]
                                        ];

                                        $l_changes = $l_dao_logb->prepare_changes($l_daoCPU, null, $l_category_values);
                                        if (count($l_changes) > 0)
                                        {
                                            isys_event_manager::getInstance()
                                                ->triggerCMDBEvent(
                                                    'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                    "-modified from OCS-",
                                                    $l_objID,
                                                    $l_thisObjTypeID,
                                                    'LC__CMDB__CATG__CPU',
                                                    serialize($l_changes)
                                                );
                                        }
                                    }

                                    $l_import_data = $l_daoCPU->parse_import_array($l_data);
                                    $l_entry_id    = $l_daoCPU->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);

                                    // Emit category signal (afterCategoryEntrySave).
                                    isys_component_signalcollection::get_instance()
                                        ->emit(
                                            "mod.cmdb.afterCategoryEntrySave",
                                            $l_daoCPU,
                                            $l_entry_id,
                                            !!$l_entry_id,
                                            $l_objID,
                                            $l_import_data,
                                            isset($l_changes) ? $l_changes : []
                                        );
                                }
                            }
                            // Delete entries
                            if (count($l_check_data) > 0)
                            {
                                $ocsHardware->delete_entries_from_category(
                                    $l_check_data,
                                    $l_daoCPU,
                                    'isys_catg_cpu_list',
                                    $l_objID,
                                    $l_thisObjTypeID,
                                    'LC__CMDB__CATG__CPU',
                                    $l_logb_active
                                );
                            }
                        }
                        else
                        {
                            // Create
                            foreach ($l_hw_data['cpu'] AS $l_rowCPU)
                            {
                                $l_data = [
                                    'title'          => null,
                                    'manufacturer'   => $l_rowCPU['MANUFACTURER'],
                                    'type'           => $l_rowCPU['TYPE'],
                                    'frequency_unit' => $l_frequency_unit,
                                    'cores'          => 1,
                                    'frequency'      => $l_rowCPU['SPEED']
                                ];
                                if ($l_logb_active)
                                {
                                    $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                        'title'          => null,
                                        'frequency'      => ['value' => $l_rowCPU['SPEED']],
                                        'frequency_unit' => ['title_lang' => 'GHz'],
                                        'type'           => ['title_lang' => $l_rowCPU['TYPE']]
                                    ];

                                    $l_changes = $l_dao_logb->prepare_changes($l_daoCPU, null, $l_category_values);
                                    isys_event_manager::getInstance()
                                        ->triggerCMDBEvent(
                                            'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                            "-modified from OCS-",
                                            $l_objID,
                                            $l_thisObjTypeID,
                                            'LC__CMDB__CATG__CPU',
                                            serialize($l_changes)
                                        );
                                }

                                $l_import_data = $l_daoCPU->parse_import_array($l_data);
                                $l_entry_id    = $l_daoCPU->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);

                                // Emit category signal (afterCategoryEntrySave).
                                isys_component_signalcollection::get_instance()
                                    ->emit(
                                        "mod.cmdb.afterCategoryEntrySave",
                                        $l_daoCPU,
                                        $l_entry_id,
                                        !!$l_entry_id,
                                        $l_objID,
                                        $l_import_data,
                                        isset($l_changes) ? $l_changes : []
                                    );
                            }
                        }
                    }

                    /**
                     * Memory Skip because the value is strange
                     */
                    if (isset($l_hw_data['memory']) && 1 == 2)
                    {
                        $output->writeln("Processing memory");
                        $this->m_log->info("Processing memory");
                        $l_check_data = [];
                        $l_res        = $l_daoMemory->get_data(null, $l_objID);
                        $l_mem_amount = $l_res->num_rows();
                        if ($l_mem_amount > 0)
                        {

                            // Get data in i-doit
                            while ($l_rowMemory = $l_res->get_row())
                            {
                                $l_check_data[] = [
                                    'data_id'      => $l_rowMemory['isys_catg_memory_list__id'],
                                    'title'        => $l_rowMemory["isys_memory_title__title"],
                                    'manufacturer' => $l_rowMemory['isys_memory_manufacturer__title'],
                                    'type'         => $l_rowMemory['isys_memory_type__title'],
                                    'unit'         => $l_rowMemory['isys_catg_memory_list__isys_memory_unit__id'],
                                    'capacity'     => $l_rowMemory['isys_catg_memory_list__capacity'],
                                    'description'  => $l_rowMemory['isys_catg_memory_list__description']
                                ];
                            }

                            if (count($l_hw_data) > 0)
                            {
                                foreach ($l_hw_data['memory'] AS $l_rowMemory)
                                {

                                    // Check data from ocs with data from i-doit
                                    foreach ($l_check_data AS $l_key => $l_val)
                                    {
                                        if ($l_val['capacity'] == isys_convert::memory($l_rowMemory["CAPACITY"], $l_capacityUnitMB))
                                        {

                                            unset($l_check_data[$l_key]);
                                            continue 2;
                                        }
                                    }

                                    // Raw array for preparing the import array
                                    $l_data = [
                                        'title'        => null,
                                        'manufacturer' => null,
                                        'type'         => null,
                                        'unit'         => $l_capacityUnitMB,
                                        'capacity'     => $l_rowMemory['CAPACITY']
                                    ];

                                    if ($l_logb_active)
                                    {
                                        $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                            'unit'     => ['title_lang' => 'MB'],
                                            'capacity' => ['value' => $l_rowMemory["CAPACITY"]]
                                        ];

                                        $l_changes = $l_dao_logb->prepare_changes($l_daoMemory, null, $l_category_values);
                                        if (count($l_changes) > 0)
                                        {
                                            isys_event_manager::getInstance()
                                                ->triggerCMDBEvent(
                                                    'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                    "-modified from OCS-",
                                                    $l_objID,
                                                    $l_thisObjTypeID,
                                                    'LC__CMDB__CATG__MEMORY',
                                                    serialize($l_changes)
                                                );
                                        }
                                    }

                                    $l_import_data = $l_daoMemory->parse_import_array($l_data);
                                    $l_entry_id    = $l_daoMemory->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);

                                    // Emit category signal (afterCategoryEntrySave).
                                    isys_component_signalcollection::get_instance()
                                        ->emit(
                                            "mod.cmdb.afterCategoryEntrySave",
                                            $l_daoMemory,
                                            $l_entry_id,
                                            !!$l_entry_id,
                                            $l_objID,
                                            $l_import_data,
                                            isset($l_changes) ? $l_changes : []
                                        );
                                }
                            }
                            // Delete entries
                            if (count($l_check_data) > 0)
                            {
                                $ocsHardware->delete_entries_from_category(
                                    $l_check_data,
                                    $l_daoMemory,
                                    'isys_catg_memory_list',
                                    $l_objID,
                                    $l_thisObjTypeID,
                                    'LC__CMDB__CATG__MEMORY',
                                    $l_logb_active
                                );
                            }

                        }
                        else
                        {
                            // Create entries
                            foreach ($l_hw_data['memory'] AS $l_rowMemory)
                            {
                                $l_data = [
                                    'title'        => null,
                                    'manufacturer' => null,
                                    'type'         => null,
                                    'unit'         => $l_capacityUnitMB,
                                    'capacity'     => $l_rowMemory["CAPACITY"]
                                ];

                                if ($l_logb_active)
                                {
                                    $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                        'unit'     => ['title_lang' => 'MB'],
                                        'capacity' => ['value' => $l_rowMemory["CAPACITY"]]
                                    ];

                                    $l_changes = $l_dao_logb->prepare_changes($l_daoMemory, null, $l_category_values);
                                    if (count($l_changes) > 0)
                                    {
                                        isys_event_manager::getInstance()
                                            ->triggerCMDBEvent(
                                                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                "-modified from OCS-",
                                                $l_objID,
                                                $l_thisObjTypeID,
                                                'LC__CMDB__CATG__MEMORY',
                                                serialize($l_changes)
                                            );
                                    }
                                }

                                $l_import_data = $l_daoMemory->parse_import_array($l_data);
                                $l_entry_id    = $l_daoMemory->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);

                                // Emit category signal (afterCategoryEntrySave).
                                isys_component_signalcollection::get_instance()
                                    ->emit(
                                        "mod.cmdb.afterCategoryEntrySave",
                                        $l_daoMemory,
                                        $l_entry_id,
                                        !!$l_entry_id,
                                        $l_objID,
                                        $l_import_data,
                                        isset($l_changes) ? $l_changes : []
                                    );
                            }
                        }
                    }

                    /**
                     * Network
                     */
                    if (isset($l_hw_data['net']))
                    {
                        $output->writeln("Processing network");
                        $this->m_log->info("Processing network");
                        unset($l_check_ip, $l_check_net, $l_check_iface, $l_check_port, $l_already_imported_ips);
                        $l_already_imported_ports = $l_already_imported_ips = [];
                        // IP info
                        $l_query_ip = 'SELECT t1.isys_catg_ip_list__id, t2.isys_cats_net_ip_addresses_list__title, t3.* ' . 'FROM isys_catg_ip_list  AS t1 ' .
                            'INNER JOIN isys_cats_net_ip_addresses_list AS t2 ON t2.isys_cats_net_ip_addresses_list__id = t1.isys_catg_ip_list__isys_cats_net_ip_addresses_list__id ' .
                            'INNER JOIN isys_cats_net_list AS t3 ON t3.isys_cats_net_list__isys_obj__id = t2.isys_cats_net_ip_addresses_list__isys_obj__id ' .
                            'WHERE t1.isys_catg_ip_list__isys_obj__id = ' . $l_daoIP->convert_sql_id(
                                $l_objID
                            );
                        $l_res_ip   = $l_daoIP->retrieve($l_query_ip);
                        while ($l_row_ip = $l_res_ip->get_row())
                        {
                            $l_check_ip[]  = [
                                'data_id'   => $l_row_ip['isys_catg_ip_list__id'],
                                'title'     => $l_row_ip['isys_cats_net_ip_addresses_list__title'],
                                'net'       => $l_row_ip['isys_cats_net_list__isys_obj__id'],
                                'net_title' => $l_row_ip['isys_cats_net_list__address'],
                                'hostnmae'  => $l_row_ip['isys_catg_ip_list__hostname'],
                                'primary'   => (bool) $l_row_ip['isys_catg_ip_list__primary']
                            ];
                            $l_check_net[] = [
                                'data_id'    => $l_row_ip['isys_cats_net_list__id'],
                                'title'      => $l_row_ip['isys_cats_net_list__address'],
                                'mask'       => $l_row_ip['isys_cats_net_list__mask'],
                                'range_from' => $l_row_ip['isys_cats_net_list__address_range_from'],
                                'range_to'   => $l_row_ip['isys_cats_net_list__address_range_to'],
                            ];
                        }

                        // Port info
                        $l_query_port = 'SELECT * FROM isys_catg_port_list ' .
                            'INNER JOIN isys_catg_connector_list ON isys_catg_connector_list__id = isys_catg_port_list__isys_catg_connector_list__id ' .
                            'LEFT JOIN isys_catg_netp_list ON isys_catg_netp_list__id = isys_catg_port_list__isys_catg_netp_list__id ' .
                            'LEFT JOIN isys_port_type ON isys_port_type__id = isys_catg_port_list__isys_port_type__id ' .
                            'LEFT JOIN isys_port_speed ON isys_port_speed__id = isys_catg_port_list__isys_port_speed__id ' . 'WHERE isys_catg_port_list__isys_obj__id = ' .
                            $l_daoPort->convert_sql_id(
                                $l_objID
                            );
                        $l_res_port   = $l_daoPort->retrieve($l_query_port);
                        while ($l_row_port = $l_res_port->get_row())
                        {
                            $l_check_port[] = [
                                'data_id'            => $l_row_port['isys_catg_port_list__id'],
                                'title'              => $l_row_port['isys_catg_port_list__title'],
                                'mac'                => $l_row_port['isys_catg_port_list__mac'],
                                'speed'              => $l_row_port['isys_catg_port_list__port_speed_value'],
                                'speed_type'         => $l_row_port['isys_port_speed__id'],
                                'port_type'          => $l_row_port['isys_port_type__title'],
                                'active'             => $l_row_port['isys_catg_port_list__state_enabled'],
                                'assigned_connector' => $l_row_port['isys_catg_port_list__isys_catg_connector_list__id'],
                                'description'        => $l_row_port['isys_catg_port_list__description']
                            ];
                        }

                        // Interface info
                        $l_check_iface     = [];
                        $l_existing_ifaces = [];

                        $l_query_interface = 'SELECT isys_catg_netp_list__id, isys_catg_netp_list__title FROM isys_catg_netp_list ' .
                            'WHERE isys_catg_netp_list__isys_obj__id = ' . $l_dao_interface->convert_sql_id(
                                $l_objID
                            );
                        $l_res_iface       = $l_dao_interface->retrieve($l_query_interface);
                        while ($l_row_iface = $l_res_iface->get_row())
                        {
                            $l_check_iface[] = [
                                'data_id' => $l_row_iface['isys_catg_netp_list__id'],
                                'title'   => $l_row_iface['isys_catg_netp_list__title']
                            ];
                        } // while

                        $l_check_ps    = [];
                        $l_existing_ps = [];
                        $l_query_ps    = 'SELECT isys_catg_pc_list__id, isys_catg_pc_list__title FROM isys_catg_pc_list
							WHERE isys_catg_pc_list__isys_obj__id = ' . $l_dao_power_consumer->convert_sql_id($l_objID);
                        $l_res_ps      = $l_dao_power_consumer->retrieve($l_query_ps);
                        while ($l_row_ps = $l_res_ps->get_row())
                        {
                            $l_check_ps[] = [
                                'data_id' => $l_row_ps['isys_catg_pc_list__id'],
                                'title'   => $l_row_ps['isys_catg_pc_list__title']
                            ];
                        } // while

                        // Copy list of existing ifaces
                        $l_existing_ifaces = $l_check_iface;
                        $l_existing_ps     = $l_check_ps;

                        $l_counter    = 0;
                        $l_net_amount = count($l_hw_data['net']) - 1;

                        foreach ($l_hw_data['net'] AS $l_hw_key => $l_row)
                        {
                            $l_speed         = null;
                            $l_speed_type_id = null;
                            if (isset($l_row['SPEED']) && $l_hw_key !== 'interfaces' && $l_hw_key !== 'powersupplies')
                            {
                                preg_match('/[0-9]*\.[0-9]*|[0-9]*/', $l_row['SPEED'], $l_speed_arr);
                                $l_speed            = $l_speed_arr[0];
                                $l_speed_type       = ltrim(str_replace($l_speed, '', $l_row['SPEED']));
                                $l_speed_type_lower = strtolower($l_speed_type);

                                if ($l_speed_type_lower == 'mb/s' || $l_speed_type_lower == 'm' || $l_speed_type_lower == 'mb' || $l_speed_type_lower == 'mbps')
                                {
                                    $l_speed_type_id = C__PORT_SPEED__MBIT_S;
                                }
                                elseif ($l_speed_type_lower == 'gb/s' || $l_speed_type_lower == 'g' || $l_speed_type_lower == 'gb' || $l_speed_type_lower == 'gbps')
                                {
                                    $l_speed_type_id = C__PORT_SPEED__GBIT_S;
                                }
                            }

                            $l_address      = null;
                            $l_interface_id = null;
                            $l_port_id      = null;
                            $l_ip_id        = null;
                            $l_sync_ip      = true;
                            $l_sync_port    = true;
                            $l_sync_iface   = true;

                            // Trim iface title if needed
                            if (isset($l_row['SLOT']) && $l_hw_key !== 'interfaces' && $l_hw_key !== 'powersupplies')
                            {
                                $l_row['SLOT'] = trim($l_row['SLOT']);
                            } // if

                            if (count($l_existing_ifaces) > 0 && $l_hw_key === 'interfaces')
                            {
                                foreach ($l_row AS $l_interface_key => $l_interface_data)
                                {
                                    foreach ($l_existing_ifaces AS $l_key => $l_iface)
                                    {
                                        if (strcasecmp($l_iface['title'], $l_interface_data['REFERENCE']) == 0 && $l_iface['serial'] === $l_interface_data['SERIALNUMBER'])
                                        {
                                            $l_interface_id          = $l_existing_ifaces[$l_key]['data_id'];
                                            $l_row[$l_interface_key] = null;
                                            // Unset to prevent removing of iface later
                                            if (isset($l_check_iface[$l_key]))
                                            {
                                                unset($l_check_iface[$l_key]);
                                            } // if
                                            continue;
                                        } // if
                                    } // foreach
                                }
                            } // if
                            if (count($l_existing_ps) > 0 && $l_hw_key === 'powersupplies')
                            {
                                foreach ($l_row AS $l_ps_key => $l_ps_data)
                                {
                                    foreach ($l_existing_ps AS $l_key => $l_ps)
                                    {
                                        if (strcasecmp($l_ps['title'], $l_ps_data['TITLE']) == 0)
                                        {
                                            $l_ps_id          = $l_existing_ps[$l_key]['data_id'];
                                            $l_row[$l_ps_key] = null;
                                            // Unset to prevent removing of iface later
                                            if (isset($l_check_ps[$l_key]))
                                            {
                                                unset($l_check_ps[$l_key]);
                                            } // if
                                            continue;
                                        }
                                    }
                                }
                            }

                            if (count($l_check_port) > 0 && isset($l_row['TYPE']) && $l_hw_key !== 'interfaces' && $l_hw_key !== 'powersupplies')
                            {
                                foreach ($l_check_port AS $l_key => $l_port)
                                {
                                    if ($l_port['title'] == $l_row['SLOT'] && strcmp($l_port['mac'], $l_row['MACADDR']) == 0)
                                    {
                                        $l_port_id = $l_check_port[$l_key]['data_id'];
                                        unset($l_check_port[$l_key]);
                                        $l_arr                                                              = new SplFixedArray(2);
                                        $l_arr[0]                                                           = $l_port['data_id'];
                                        $l_arr[1]                                                           = $l_port['description'];
                                        $l_already_imported_ports[$l_row['SLOT'] . '_' . $l_row['MACADDR']] = $l_arr;
                                        $l_sync_port                                                        = false;

                                        if (empty($l_port['description']) && $l_port['description'] != $l_row['DESCRIPTION'] && !isset($l_port_descriptions[$l_port_id]) &&
                                            $l_port['description'] === '')
                                        {
                                            $l_port_descriptions[$l_port_id] = $l_row['DESCRIPTION'];
                                        } // if
                                        continue;
                                    }
                                }
                            }

                            if (count($l_check_ip) > 0 && isset($l_row['IPADDR']) && $l_hw_key !== 'interfaces' && $l_hw_key !== 'powersupplies')
                            {
                                foreach ($l_check_ip AS $l_key => $l_ip)
                                {
                                    if ($l_ip['title'] == $l_row['IPADDR'])
                                    {
                                        if (!!$l_row['PRIMARY'] === true)
                                        {
                                            // Check hostname
                                            if ($l_ip['hostname'] != substr($l_inventory['NAME'], 0, strpos($l_inventory['NAME'], '.')))
                                            {
                                                continue;
                                            } // if
                                        } // if

                                        $l_ip_id = $l_check_ip[$l_key]['data_id'];
                                        unset($l_check_ip[$l_key]);
                                        unset($l_check_net[$l_key]);
                                        $l_already_imported_ips[$l_ip_id] = $l_ip['title'];
                                        $l_sync_ip                        = false;
                                        continue;
                                    }
                                }
                            }

                            // Sync Power Consumer
                            if ($l_hw_key === 'powersupplies')
                            {
                                foreach ($l_row AS $l_ps_data)
                                {
                                    if ($l_ps_data === null) continue;

                                    $l_data = [
                                        'title'        => $l_ps_data['TITLE'],
                                        'manufacturer' => $l_ps_data['MANUFACTURER']
                                    ];

                                    if ($l_logb_active)
                                    {
                                        $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                            'title'        => ['value' => $l_ps_data['TITLE']],
                                            'manufacturer' => ['title_lang' => $l_ps_data["MANUFACTURER"]]
                                        ];
                                        $l_changes                                                  = $l_dao_logb->prepare_changes(
                                            $l_dao_power_consumer,
                                            null,
                                            $l_category_values
                                        );
                                        if (count($l_changes) > 0)
                                        {
                                            isys_event_manager::getInstance()
                                                ->triggerCMDBEvent(
                                                    'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                    "-modified from OCS-",
                                                    $l_objID,
                                                    $l_thisObjTypeID,
                                                    'LC__CMDB__CATG__NETWORK_TREE_CONFIG_INTERFACE',
                                                    serialize($l_changes)
                                                );
                                        }
                                    }

                                    if (!empty($l_data['manufacturer']))
                                    {
                                        $l_manufacturer = isys_import_handler::check_dialog('isys_pc_manufacturer', $l_data['manufacturer']);
                                    }
                                    else $l_manufacturer = null;

                                    $l_import_data = [
                                        'data_id'    => null,
                                        'properties' => [
                                            'title'              => [
                                                'value' => $l_data['title']
                                            ],
                                            'manufacturer'       => [
                                                'value' => $l_manufacturer
                                            ],
                                            'active'             => [
                                                'value' => null
                                            ],
                                            'model'              => [
                                                'value' => null
                                            ],
                                            'volt'               => [
                                                'value' => null
                                            ],
                                            'watt'               => [
                                                'value' => null
                                            ],
                                            'ampere'             => [
                                                'value' => null
                                            ],
                                            'btu'                => [
                                                'value' => null
                                            ],
                                            'assigned_connector' => [
                                                'value' => null
                                            ],
                                            'connector_sibling'  => [
                                                'value' => null
                                            ],
                                            'description'        => [
                                                'value' => null
                                            ]
                                        ]
                                    ];

                                    $l_pc_id = $l_dao_power_consumer->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);

                                    // Emit category signal (afterCategoryEntrySave).
                                    isys_component_signalcollection::get_instance()
                                        ->emit(
                                            "mod.cmdb.afterCategoryEntrySave",
                                            $l_dao_power_consumer,
                                            $l_pc_id,
                                            !!$l_pc_id,
                                            $l_objID,
                                            $l_import_data,
                                            isset($l_changes) ? $l_changes : []
                                        );

                                    // Add it to existing ifaces
                                    if (is_numeric($l_pc_id))
                                    {
                                        $l_existing_ps[] = [
                                            'data_id' => $l_pc_id,
                                            'title'   => $l_ps_data['TITLE']
                                        ];
                                    } // if
                                }
                            }

                            // Interface sync
                            if ($l_hw_key === 'interfaces')
                            {
                                foreach ($l_row AS $l_interface_data)
                                {
                                    if ($l_interface_data === null) continue;

                                    $l_data = [
                                        'title'        => $l_interface_data['REFERENCE'],
                                        'firmware'     => $l_interface_data['FIRMWARE'],
                                        'serial'       => $l_interface_data['SERIALNUMBER'],
                                        'manufacturer' => $l_interface_data['MANUFACTURER'],
                                        'description'  => $l_interface_data['DESCRIPTION']
                                    ];

                                    if ($l_logb_active)
                                    {
                                        $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                            'title' => ['value' => $l_interface_data['REFERENCE']]
                                        ];
                                        $l_changes                                                  = $l_dao_logb->prepare_changes(
                                            $l_dao_interface,
                                            null,
                                            $l_category_values
                                        );
                                        if (count($l_changes) > 0)
                                        {
                                            isys_event_manager::getInstance()
                                                ->triggerCMDBEvent(
                                                    'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                    "-modified from OCS-",
                                                    $l_objID,
                                                    $l_thisObjTypeID,
                                                    'LC__CMDB__CATG__NETWORK_TREE_CONFIG_INTERFACE',
                                                    serialize($l_changes)
                                                );
                                        }
                                    }

                                    $l_import_data = $l_dao_interface->parse_import_array($l_data);

                                    $l_interface_id = $l_dao_interface->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);

                                    // Emit category signal (afterCategoryEntrySave).
                                    isys_component_signalcollection::get_instance()
                                        ->emit(
                                            "mod.cmdb.afterCategoryEntrySave",
                                            $l_dao_interface,
                                            $l_interface_id,
                                            !!$l_interface_id,
                                            $l_objID,
                                            $l_import_data,
                                            isset($l_changes) ? $l_changes : []
                                        );
                                    // Add it to existing ifaces
                                    if (is_numeric($l_interface_id))
                                    {
                                        $l_existing_ifaces[] = [
                                            'data_id' => $l_interface_id,
                                            'title'   => $l_interface_data['REFERENCE']
                                        ];
                                    } // if
                                }
                                continue;
                            }

                            if (strtoupper($l_row['STATUS']) == 'UP')
                            {
                                $l_status = '1';
                            }
                            else $l_status = '0';

                            // Port sync
                            if ($l_sync_port && isset($l_row['TYPE']) && !isset($l_already_imported_ports[$l_row['SLOT'] . '_' . $l_row['MACADDR']]))
                            {
                                $l_data = [
                                    'title'       => $l_row['SLOT'],
                                    // 'Port ' . $l_hw_key,
                                    'port_type'   => $l_row['TYPE'],
                                    'speed'       => $l_speed,
                                    'speed_type'  => $l_speed_type_id,
                                    'mac'         => $l_row['MACADDR'],
                                    'active'      => $l_status,
                                    'description' => $l_row['DESCRIPTION']
                                ];

                                // Interface to port
                                if (isset($l_interface_id))
                                {
                                    $l_data['interface'] = $l_interface_id;
                                }

                                if (!$l_sync_ip)
                                {
                                    // add hostaddress
                                    $l_data['addresses'] = [
                                        $l_ip_id
                                    ];
                                }

                                if ($l_logb_active)
                                {
                                    $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                        'title'       => ['value' => $l_row['SLOT']],
                                        'port_type'   => ['title_lang' => $l_row['TYPE']],
                                        'speed'       => ['value' => $l_row['SPEED']],
                                        'speed_type'  => ['title_lang' => $l_speed_type],
                                        'mac'         => ['value' => $l_row['MACADDR']],
                                        'active'      => ['value' => ($l_status) ? _L('LC__UNIVERSAL__YES') : _L('LC__UNIVERSAL__NO')],
                                        'interface'   => ['value' => $l_row['SLOT']],
                                        'description' => ['value' => $l_row['DESCRIPTION']]
                                    ];

                                    $l_changes = $l_dao_logb->prepare_changes($l_daoPort, null, $l_category_values);

                                    if (count($l_changes) > 0)
                                    {
                                        isys_event_manager::getInstance()
                                            ->triggerCMDBEvent(
                                                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                "-modified from OCS-",
                                                $l_objID,
                                                $l_thisObjTypeID,
                                                'LC__CMDB__CATG__NETWORK_TREE_CONFIG_INTERFACE',
                                                serialize($l_changes)
                                            );
                                    }
                                }

                                $l_import_data                                                      = $l_daoPort->parse_import_array($l_data);
                                $l_port_id                                                          = $l_daoPort->sync(
                                    $l_import_data,
                                    $l_objID,
                                    isys_import_handler_cmdb::C__CREATE
                                );
                                $l_arr                                                              = new SplFixedArray(2);
                                $l_arr[0]                                                           = $l_port_id;
                                $l_arr[1]                                                           = $l_row['DESCRIPTION'];
                                $l_already_imported_ports[$l_row['SLOT'] . '_' . $l_row['MACADDR']] = $l_arr;

                                // Emit category signal (afterCategoryEntrySave).
                                isys_component_signalcollection::get_instance()
                                    ->emit(
                                        "mod.cmdb.afterCategoryEntrySave",
                                        $l_daoPort,
                                        $l_port_id,
                                        !!$l_port_id,
                                        $l_objID,
                                        $l_import_data,
                                        isset($l_changes) ? $l_changes : []
                                    );
                            } // if

                            // Cache Port connections this will be handled after the categories has been imported for this device
                            if (isset($l_already_imported_ports[$l_row['SLOT'] . '_' . $l_row['MACADDR']]))
                            {
                                if (!isset($l_port_connections[$l_already_imported_ports[$l_row['SLOT'] . '_' . $l_row['MACADDR']][0]]) && !empty($l_row['DEVICEADDRESS']) &&
                                    !empty($l_row['DEVICENAME']))
                                {
                                    $l_fixed_array                                                                              = new SplFixedArray(3);
                                    $l_fixed_array[0]                                                                           = $l_row['DEVICENAME'];
                                    $l_fixed_array[1]                                                                           = $l_row['DEVICEPORT'];
                                    $l_fixed_array[2]                                                                           = (isset($l_row['connected_to'])) ? $l_row['connected_to'] : null;
                                    $l_port_connections[$l_already_imported_ports[$l_row['SLOT'] . '_' . $l_row['MACADDR']][0]] = $l_fixed_array;
                                } // if

                                if (!isset($l_port_descriptions[$l_already_imported_ports[$l_row['SLOT'] . '_' . $l_row['MACADDR']][0]]) &&
                                    $l_row['DESCRIPTION'] != $l_already_imported_ports[$l_row['SLOT'] . '_' . $l_row['MACADDR']][1] && $l_row['DESCRIPTION'] != '')
                                {
                                    $l_port_descriptions[$l_already_imported_ports[$l_row['SLOT'] . '_' . $l_row['MACADDR']][0]] = $l_row['DESCRIPTION'];
                                } // if
                            } // if

                            // Ip sync
                            if ($l_sync_ip && isset($l_row['IPADDR']))
                            {

                                // ip type check ipv4, ipv6
                                if ($l_row['IPADDR'] != '' && $l_row['IPMASK'] != '')
                                {
                                    // Calculate net ip
                                    $l_subnet  = Ip::validate_net_ip($l_row['IPADDR'], $l_row['IPMASK'], null, true);
                                    $l_dhcp    = false;
                                    $l_address = $l_row['IPADDR'];
                                }
                                elseif (Ip::ip2long($l_row['IPDHCP']) > 0 && $l_row['IPSUBNET'] != '0.0.0.0')
                                {
                                    /*if(($l_pos = strpos($l_row['IPDHCP'],'.0')) > 0){
										$l_subnet = substr($l_row['IPDHCP'], 0, ($l_pos));
										$l_amount = substr_count($l_subnet, '.');
										while($l_amount < 3){
											$l_amount++;
											$l_subnet .= '.0';
										}
									} else{
										$l_ip_arr = explode('.', $l_row['IPDHCP']);
										$l_subnet = $l_ip_arr[0].'.'.$l_ip_arr[1].'.'.$l_ip_arr[2].'.0';
									}*/
                                    $l_subnet  = $l_row['IPSUBNET'];
                                    $l_dhcp    = true;
                                    $l_address = $l_row['IPDHCP'];
                                } // if

                                $l_primary = (int) $l_row['PRIMARY'];

                                // Secondary Check
                                if ($l_address !== null && !in_array($l_address, $l_already_imported_ips))
                                {
                                    if (Ip::validate_ip($l_subnet))
                                    {
                                        $l_net_type    = C__CATS_NET_TYPE__IPV4;
                                        $l_cidr_suffix = Ip::calc_cidr_suffix($l_row['IPMASK']);
                                        if ($l_dhcp)
                                        {
                                            $l_ip_type = C__CATP__IP__ASSIGN__DHCP;
                                        }
                                        else $l_ip_type = C__CATP__IP__ASSIGN__STATIC;
                                        $l_net_id = C__OBJ__NET_GLOBAL_IPV6;
                                    }
                                    elseif (Ip::validate_ipv6($l_subnet))
                                    {
                                        $l_net_type    = C__CATS_NET_TYPE__IPV6;
                                        $l_cidr_suffix = Ip::calc_cidr_suffix_ipv6($l_row['IPMASK']);
                                        if ($l_dhcp)
                                        {
                                            $l_ip_type = C__CMDB__CATG__IP__DHCPV6;
                                        }
                                        else $l_ip_type = C__CMDB__CATG__IP__STATIC;
                                        $l_net_id = C__OBJ__NET_GLOBAL_IPV6;
                                    }

                                    // Check Net
                                    if (isset($l_net_arr[$l_subnet]))
                                    {
                                        $l_net_id    = $l_net_arr[$l_subnet]['row_data']['isys_cats_net_list__isys_obj__id'];
                                        $l_net_title = $l_net_arr[$l_subnet]['row_data']['isys_obj__title'];

                                        if ($l_net_arr[$l_subnet]['row_data']['isys_cats_net_list__address_range_from'] === '')
                                        {
                                            // Update net because the range is not set
                                            $l_ip_range                                                                 = Ip::calc_ip_range(
                                                $l_subnet,
                                                $l_net_arr[$l_subnet]['row_data']['isys_cats_net_list__mask']
                                            );
                                            $l_from_long                                                                = Ip::ip2long($l_ip_range['from']);
                                            $l_to_long                                                                  = Ip::ip2long($l_ip_range['to']);
                                            $l_net_arr[$l_subnet]['row_data']['isys_cats_net_list__address_range_from'] = $l_ip_range['from'];
                                            $l_net_arr[$l_subnet]['row_data']['isys_cats_net_list__address_range_to']   = $l_ip_range['to'];
                                            $l_update                                                                   = 'UPDATE isys_cats_net_list SET isys_cats_net_list__address_range_from = ' .
                                                $l_daoCMDB->convert_sql_text(
                                                    $l_ip_range['from']
                                                ) . ',
												isys_cats_net_list__address_range_to = ' . $l_daoCMDB->convert_sql_text($l_ip_range['to']) . ',
												isys_cats_net_list__address_range_from_long = ' . $l_daoCMDB->convert_sql_text($l_from_long) . ',
												isys_cats_net_list__address_range_to_long = ' . $l_daoCMDB->convert_sql_text($l_to_long) . '
												WHERE isys_cats_net_list__isys_obj__id = ' . $l_daoCMDB->convert_sql_id($l_net_id);
                                            $l_daoCMDB->update($l_update);
                                        } // if
                                    }
                                    else
                                    {
                                        // Create net
                                        $l_gateway_arr = $l_daoCMDB->retrieve(
                                            'SELECT isys_catg_ip_list__id FROM isys_catg_ip_list ' .
                                            'INNER JOIN isys_cats_net_ip_addresses_list ON isys_catg_ip_list__isys_cats_net_ip_addresses_list__id = isys_cats_net_ip_addresses_list__id ' .
                                            'WHERE isys_cats_net_ip_addresses_list__title = ' . $l_daoCMDB->convert_sql_text(
                                                $l_row['IPGATEWAY']
                                            )
                                        )
                                            ->__to_array();
                                        if ($l_gateway_arr)
                                        {
                                            $l_gateway_id = $l_gateway_arr['isys_catg_ip_list__id'];
                                        }
                                        else
                                        {
                                            $l_gateway_id = null;
                                        }
                                        $l_net_id    = $l_daoCMDB->insert_new_obj(
                                            C__OBJTYPE__LAYER3_NET,
                                            false,
                                            $l_subnet . '/' . $l_cidr_suffix,
                                            null,
                                            C__RECORD_STATUS__NORMAL
                                        );
                                        $l_net_title = $l_subnet;
                                        $l_ip_range  = Ip::calc_ip_range($l_subnet, $l_row['IPMASK']);
                                        $l_daoNet_s->create(
                                            $l_net_id,
                                            C__RECORD_STATUS__NORMAL,
                                            $l_subnet,
                                            $l_net_type,
                                            $l_subnet,
                                            $l_row['IPMASK'],
                                            $l_gateway_id,
                                            false,
                                            $l_ip_range['from'],
                                            $l_ip_range['to'],
                                            null,
                                            null,
                                            '',
                                            $l_cidr_suffix
                                        );

                                        $l_net_arr[$l_subnet] = [
                                            'row_data' => [
                                                'isys_cats_net_list__isys_obj__id' => $l_net_id,
                                                'isys_obj__title'                  => $l_subnet,
                                            ]
                                        ];
                                    }

                                    $l_data = [
                                        'net_type' => $l_net_type,
                                        'primary'  => $l_primary,
                                        'active'   => '1',
                                        'net'      => $l_net_id,
                                        'hostname' => ($l_primary) ? (strpos($l_inventory['NAME'], '.') !== false ? substr(
                                            $l_inventory['NAME'],
                                            0,
                                            strpos($l_inventory['NAME'], '.')
                                        ) : $l_inventory['NAME']) : ''
                                    ];

                                    if (strpos($l_inventory['NAME'], '.') !== false && $l_primary)
                                    {
                                        $l_data['dns_domain'] = isys_cmdb_dao_dialog::instance($this->container->database)
                                            ->check_dialog('isys_net_dns_domain', substr($l_inventory['NAME'], strpos($l_inventory['NAME'], '.') + 1));
                                    }
                                    else
                                    {
                                        $l_data['dns_domain'] = null;
                                    } // if

                                    switch ($l_net_type)
                                    {
                                        case C__CATS_NET_TYPE__IPV4:
                                            $l_data['ipv4_address']    = $l_address;
                                            $l_data['ipv4_assignment'] = $l_ip_type;
                                            break;
                                        case C__CATS_NET_TYPE__IPV6:
                                            $l_data['ipv6_address']    = $l_address;
                                            $l_data['ipv6_assignment'] = $l_ip_type;
                                            break;
                                    }

                                    // add port
                                    if ($l_port_id > 0)
                                    {
                                        $l_data['assigned_port'] = $l_port_id;
                                    }
                                    else
                                    {
                                        $l_data['assigned_port'] = null;
                                    }

                                    if ($l_logb_active)
                                    {
                                        switch ($l_net_type)
                                        {
                                            case C__CATS_NET_TYPE__IPV4:
                                                $l_ip_assignment                                            = $l_daoCMDB->get_dialog('isys_ip_assignment', $l_ip_type)
                                                    ->get_row();
                                                $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                                    'ipv4_address'    => ['value' => $l_row["IPADDR"]],
                                                    'ipv4_assignment' => ['title_lang' => $l_ip_assignment['isys_ip_assignment__title']],
                                                    'net_type'        => ['title_lang' => 'IPv4']
                                                ];
                                                break;
                                            case C__CATS_NET_TYPE__IPV6:
                                                $l_ip_assignment                                            = $l_daoCMDB->get_dialog('isys_ipv6_assignment', $l_ip_type)
                                                    ->get_row();
                                                $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                                    'ipv6_address'    => ['value' => $l_row["IPADDR"]],
                                                    'ipv6_assignment' => ['title_lang' => $l_ip_assignment['isys_ip_assignment__title']],
                                                    'net_type'        => ['title_lang' => 'IPv6'],
                                                ];
                                                break;
                                        }

                                        $l_category_values[isys_import_handler_cmdb::C__PROPERTIES]['hostname'] = [
                                            'value' => (($l_primary) ? substr(
                                                $l_inventory['NAME'],
                                                0,
                                                strpos($l_inventory['NAME'], '.')
                                            ) : null)
                                        ];
                                        $l_category_values[isys_import_handler_cmdb::C__PROPERTIES]['net']      = ['value' => $l_net_id];
                                        $l_category_values[isys_import_handler_cmdb::C__PROPERTIES]['active']   = ['value' => 1];
                                        $l_category_values[isys_import_handler_cmdb::C__PROPERTIES]['primary']  = ['value' => $l_primary];

                                        $l_changes = $l_dao_logb->prepare_changes($l_daoIP, null, $l_category_values);

                                        if (count($l_changes) > 0)
                                        {
                                            isys_event_manager::getInstance()
                                                ->triggerCMDBEvent(
                                                    'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                    "-modified from OCS-",
                                                    $l_objID,
                                                    $l_thisObjTypeID,
                                                    'LC__CATG__IP_ADDRESS',
                                                    serialize($l_changes)
                                                );
                                            unset($l_changes);
                                        }
                                    }

                                    $l_import_data                         = $l_daoIP->parse_import_array($l_data);
                                    $l_ip_data_id                          = $l_daoIP->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);
                                    $l_already_imported_ips[$l_ip_data_id] = $l_address;

                                    // Emit category signal (afterCategoryEntrySave).
                                    isys_component_signalcollection::get_instance()
                                        ->emit(
                                            "mod.cmdb.afterCategoryEntrySave",
                                            $l_daoIP,
                                            $l_ip_data_id,
                                            !!$l_ip_data_id,
                                            $l_objID,
                                            $l_import_data,
                                            isset($l_changes) ? $l_changes : []
                                        );
                                }
                            }
                            $l_counter++;
                        }
                        if (count($l_check_iface) > 0)
                        {
                            $ocsHardware->delete_entries_from_category(
                                $l_check_iface,
                                $l_dao_interface,
                                'isys_catg_netp_list',
                                $l_objID,
                                $l_thisObjTypeID,
                                'LC__CMDB__CATG__NETWORK_TREE_CONFIG_INTERFACE',
                                $l_logb_active
                            );
                        }

                        if (count($l_check_port) > 0)
                        {
                            foreach ($l_check_port AS $l_val)
                            {
                                $l_cableConID = $l_daoCableCon->get_cable_connection_id_by_connector_id($l_val["assigned_connector"]);
                                $l_daoCableCon->delete_cable_connection($l_cableConID);
                                $l_daoCableCon->delete_connector($l_val["assigned_connector"]);
                                $l_daoPort->delete_entry($l_val['data_id'], 'isys_catg_port_list');

                                if ($l_logb_active)
                                {
                                    isys_event_manager::getInstance()
                                        ->triggerCMDBEvent(
                                            'C__LOGBOOK_EVENT__CATEGORY_PURGED',
                                            "-modified from OCS-",
                                            $l_objID,
                                            $l_thisObjTypeID,
                                            'LC__CATD__PORT',
                                            null
                                        );
                                }
                            }
                        }

                        if (count($l_check_ip) > 0)
                        {
                            $ocsHardware->delete_entries_from_category(
                                $l_check_ip,
                                $l_daoIP,
                                'isys_catg_ip_list',
                                $l_objID,
                                $l_thisObjTypeID,
                                'LC__CATG__IP_ADDRESS',
                                $l_logb_active
                            );
                        }
                    }

                    /**
                     * Universal interfaces
                     */
                    if (isset($l_hw_data['ui']))
                    {
                        $output->writeln("Processing UI");
                        $this->m_log->info("Processing UI");
                        $l_check_data = [];

                        $l_res_ui    = $l_daoUI->retrieve(
                            'SELECT isys_catg_ui_list__id, isys_catg_ui_list__title, isys_catg_ui_list__isys_catg_connector_list__id, ' . 'isys_ui_plugtype__title ' .
                            'FROM isys_catg_ui_list ' .
                            'INNER JOIN isys_catg_connector_list ON isys_catg_connector_list__id = isys_catg_ui_list__isys_catg_connector_list__id ' .
                            'LEFT JOIN isys_ui_plugtype ON isys_ui_plugtype__id = isys_catg_ui_list__isys_ui_plugtype__id ' . 'WHERE isys_catg_ui_list__isys_obj__id = ' .
                            $l_daoUI->convert_sql_id(
                                $l_objID
                            )
                        );
                        $l_ui_amount = $l_res_ui->num_rows();

                        if ($l_ui_amount > 0)
                        {
                            // data from i-doit
                            while ($l_row_ui = $l_res_ui->get_row())
                            {
                                $l_check_data[] = [
                                    'data_id'            => $l_row_ui['isys_catg_ui_list__id'],
                                    'title'              => $l_row_ui['isys_catg_ui_list__title'],
                                    'plug'               => $l_row_ui['isys_ui_plugtype__title'],
                                    'assigned_connector' => $l_row_ui['isys_catg_ui_list__isys_catg_connector_list__id'],
                                    'description'        => $l_row_ui['isys_catg_ui_list__description']
                                ];
                            }

                            if (count($l_hw_data['ui']) > 0)
                            {
                                foreach ($l_hw_data['ui'] AS $l_row)
                                {
                                    // Check if data already exists in i-doit
                                    foreach ($l_check_data AS $l_key => $l_value)
                                    {
                                        if ($l_value['title'] == $l_row["NAME"] && $l_value['plug'] == $l_row['TYPE'])
                                        {
                                            unset($l_check_data[$l_key]);
                                            continue 2;
                                        }
                                    }

                                    // Create new data
                                    $l_data = [
                                        'title'       => $l_row["NAME"],
                                        'plug'        => $l_row["TYPE"],
                                        'type'        => $l_conTypeTitle,
                                        'description' => $l_row['DESCRIPTION'],
                                    ];

                                    if ($l_logb_active)
                                    {
                                        $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                            'title' => ['value' => $l_row["NAME"]],
                                            'plug'  => ['title_lang' => $l_row["TYPE"]],
                                            'type'  => ['title_lang' => $l_conTypeTitle],
                                        ];

                                        $l_changes = $l_dao_logb->prepare_changes($l_daoUI, null, $l_category_values);

                                        if (count($l_changes) > 0)
                                        {
                                            isys_event_manager::getInstance()
                                                ->triggerCMDBEvent(
                                                    'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                    "-modified from OCS-",
                                                    $l_objID,
                                                    $l_thisObjTypeID,
                                                    'LC__CMDB__CATG__UNIVERSAL_INTERFACE',
                                                    serialize($l_changes)
                                                );
                                        }
                                    }

                                    $l_import_data = $l_daoUI->parse_import_array($l_data);
                                    $l_entry_id    = $l_daoUI->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);

                                    // Emit category signal (afterCategoryEntrySave).
                                    isys_component_signalcollection::get_instance()
                                        ->emit(
                                            "mod.cmdb.afterCategoryEntrySave",
                                            $l_daoUI,
                                            $l_entry_id,
                                            !!$l_entry_id,
                                            $l_objID,
                                            $l_import_data,
                                            isset($l_changes) ? $l_changes : []
                                        );
                                }
                            }

                            foreach ($l_check_data AS $l_val)
                            {
                                $l_cableConID = $l_daoCableCon->get_cable_connection_id_by_connector_id($l_val["assigned_connector"]);
                                $l_daoCableCon->delete_cable_connection($l_cableConID);
                                $l_daoCableCon->delete_connector($l_val["assigned_connector"]);
                                $l_daoUI->delete_entry($l_val['data_id'], 'isys_catg_ui_list');

                                if ($l_logb_active)
                                {
                                    isys_event_manager::getInstance()
                                        ->triggerCMDBEvent(
                                            'C__LOGBOOK_EVENT__CATEGORY_PURGED',
                                            "-modified from OCS-",
                                            $l_objID,
                                            $l_thisObjTypeID,
                                            'LC__CMDB__CATG__UNIVERSAL_INTERFACE',
                                            null
                                        );
                                }
                            }

                        }
                        else
                        {
                            // create
                            foreach ($l_hw_data['ui'] AS $l_row)
                            {

                                $l_data = [
                                    'title'       => $l_row["NAME"],
                                    'plug'        => $l_row["TYPE"],
                                    'type'        => $l_conTypeTitle,
                                    'description' => $l_row['DESCRIPTION'],
                                ];

                                if ($l_logb_active)
                                {
                                    $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                        'title' => ['value' => $l_row["NAME"]],
                                        'plug'  => ['title_lang' => $l_row["TYPE"]],
                                        'type'  => ['title_lang' => $l_conTypeTitle],
                                    ];

                                    $l_changes = $l_dao_logb->prepare_changes($l_daoUI, null, $l_category_values);

                                    if (count($l_changes) > 0)
                                    {
                                        isys_event_manager::getInstance()
                                            ->triggerCMDBEvent(
                                                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                "-modified from OCS-",
                                                $l_objID,
                                                $l_thisObjTypeID,
                                                'LC__CMDB__CATG__UNIVERSAL_INTERFACE',
                                                serialize($l_changes)
                                            );
                                    }
                                }

                                $l_import_data = $l_daoUI->parse_import_array($l_data);
                                $l_entry_id    = $l_daoUI->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);

                                // Emit category signal (afterCategoryEntrySave).
                                isys_component_signalcollection::get_instance()
                                    ->emit(
                                        "mod.cmdb.afterCategoryEntrySave",
                                        $l_daoUI->get_category_id(),
                                        $l_entry_id,
                                        !!$l_entry_id,
                                        $l_objID,
                                        $l_import_data,
                                        isset($l_changes) ? $l_changes : []
                                    );
                            }
                        }
                    }

                    /**
                     * Storage
                     */
                    if (isset($l_hw_data['stor']))
                    {
                        $output->writeln("Processing storage");
                        $this->m_log->info("Processing storage");
                        $l_check_data = [];

                        $l_res_stor = $l_daoStor->retrieve(
                            'SELECT isys_catg_stor_list__title, isys_stor_manufacturer__title, isys_stor_model__title, ' .
                            'isys_catg_stor_list__capacity, isys_catg_stor_list__id ' . 'FROM isys_catg_stor_list ' .
                            'LEFT JOIN isys_stor_manufacturer ON isys_stor_manufacturer__id = isys_catg_stor_list__isys_stor_manufacturer__id ' .
                            'LEFT JOIN isys_stor_model ON isys_stor_model__id = isys_catg_stor_list__isys_stor_model__id ' . 'WHERE isys_catg_stor_list__isys_obj__id = ' .
                            $l_daoStor->convert_sql_id(
                                $l_objID
                            )
                        );

                        $l_stor_amount = $l_res_stor->num_rows();

                        if ($l_stor_amount > 0)
                        {
                            // Check, Delete, Create
                            while ($l_row = $l_res_stor->get_row())
                            {
                                $l_check_data[] = [
                                    'data_id'      => $l_row['isys_catg_stor_list__id'],
                                    'title'        => $l_row['isys_catg_stor_list__title'],
                                    'manufacturer' => $l_row['isys_stor_manufacturer__title'],
                                    'model'        => $l_row['isys_stor_model__title'],
                                    'serial'       => $l_row['isys_catg_stor_list__serial'],
                                    'capacity'     => $l_row['isys_catg_stor_list__capacity']
                                ];
                            }

                            if (count($l_hw_data['stor']) > 0)
                            {
                                foreach ($l_hw_data['stor'] AS $l_row)
                                {
                                    // Check if data already exists in i-doit
                                    foreach ($l_check_data AS $l_key => $l_value)
                                    {
                                        if ($l_value['title'] == $l_row["NAME"] && $l_value['serial'] == $l_row['SERIALNUMBER'] &&
                                            $l_value['manufacturer'] == $l_row['MANUFACTURER'] && $l_value['model'] == $l_row['MODEL'] &&
                                            $l_value['capacity'] == isys_convert::memory(
                                                $l_row['DISKSIZE'],
                                                $l_capacityUnitMB
                                            ))
                                        {
                                            unset($l_check_data[$l_key]);
                                            continue 2;
                                        }
                                    }

                                    if ($l_row["TYPE"] == null || $l_row["TYPE"] == "")
                                    {
                                        $l_deviceType = $ocsHardware->parseStorageType($l_row["DESCRIPTION"]);
                                    }
                                    else
                                    {
                                        $l_deviceType = $ocsHardware->parseStorageType($l_row["TYPE"]);
                                    }

                                    // Create new data
                                    $l_data = [
                                        'title'        => $l_row['NAME'],
                                        'manufacturer' => $l_row['MANUFACTURER'],
                                        'model'        => $l_row['MODEL'],
                                        'serial'       => $l_row['SERIALNUMBER'],
                                        'capacity'     => $l_row['DISKSIZE'],
                                        'unit'         => $l_capacityUnitMB,
                                        'description'  => $l_row['DESCRIPTION'],
                                        'type'         => $l_deviceType
                                    ];

                                    if ($l_logb_active)
                                    {
                                        $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                            'title'        => ['value' => $l_row["NAME"]],
                                            'capacity'     => ['value' => $l_row["DISKSIZE"]],
                                            'unit'         => ['title_lang' => 'MB'],
                                            'manufacturer' => ['title_lang' => $l_row["MANUFACTURER"]],
                                            'model'        => ['title_lang' => $l_row["MODEL"]],
                                            'serial'       => ['value' => $l_row['SERIALNUMBER']]
                                        ];

                                        $l_changes = $l_dao_logb->prepare_changes($l_daoStor, null, $l_category_values);

                                        if (count($l_changes) > 0)
                                        {
                                            isys_event_manager::getInstance()
                                                ->triggerCMDBEvent(
                                                    'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                    "-modified from OCS-",
                                                    $l_objID,
                                                    $l_thisObjTypeID,
                                                    'LC__UNIVERSAL__DEVICES',
                                                    serialize($l_changes)
                                                );
                                        }
                                    }

                                    $l_import_data = $l_daoStor->parse_import_array($l_data);
                                    $l_entry_id    = $l_daoStor->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);

                                    // Emit category signal (afterCategoryEntrySave).
                                    isys_component_signalcollection::get_instance()
                                        ->emit(
                                            "mod.cmdb.afterCategoryEntrySave",
                                            $l_daoStor,
                                            $l_entry_id,
                                            !!$l_entry_id,
                                            $l_objID,
                                            $l_import_data,
                                            isset($l_changes) ? $l_changes : []
                                        );
                                }
                            }

                            if (count($l_check_data) > 0)
                            {
                                $ocsHardware->delete_entries_from_category(
                                    $l_check_data,
                                    $l_daoStor,
                                    'isys_catg_stor_list',
                                    $l_objID,
                                    $l_thisObjTypeID,
                                    'LC__UNIVERSAL__DEVICES',
                                    $l_logb_active
                                );
                            } // if
                        }
                        else
                        {
                            // create
                            foreach ($l_hw_data['stor'] AS $l_row)
                            {

                                if ($l_row["TYPE"] == null || $l_row["TYPE"] == "")
                                {
                                    $l_deviceType = $ocsHardware->parseStorageType($l_row["DESCRIPTION"]);
                                }
                                else
                                {
                                    $l_deviceType = $ocsHardware->parseStorageType($l_row["TYPE"]);
                                }

                                // Create new data
                                $l_data = [
                                    'title'        => $l_row['NAME'],
                                    'manufacturer' => $l_row['MANUFACTURER'],
                                    'model'        => $l_row['MODEL'],
                                    'capacity'     => $l_row['DISKSIZE'],
                                    'unit'         => $l_capacityUnitMB,
                                    'description'  => $l_row['DESCRIPTION'],
                                    'type'         => $l_deviceType
                                ];

                                if ($l_logb_active)
                                {
                                    $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = [
                                        'title'        => ['value' => $l_row["NAME"]],
                                        'capacity'     => ['value' => $l_row["DISKSIZE"]],
                                        'unit'         => ['title_lang' => 'MB'],
                                        'manufacturer' => ['title_lang' => $l_row["MANUFACTURER"]],
                                        'model'        => ['title_lang' => $l_row["MODEL"]]
                                    ];

                                    $l_changes = $l_dao_logb->prepare_changes($l_daoStor, null, $l_category_values);

                                    if (count($l_changes) > 0)
                                    {
                                        isys_event_manager::getInstance()
                                            ->triggerCMDBEvent(
                                                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                "-modified from OCS-",
                                                $l_objID,
                                                $l_thisObjTypeID,
                                                'LC__UNIVERSAL__DEVICES',
                                                serialize($l_changes)
                                            );
                                    } // if
                                } // if

                                $l_import_data = $l_daoStor->parse_import_array($l_data);
                                $l_entry_id    = $l_daoStor->sync($l_import_data, $l_objID, isys_import_handler_cmdb::C__CREATE);

                                // Emit category signal (afterCategoryEntrySave).
                                isys_component_signalcollection::get_instance()
                                    ->emit(
                                        "mod.cmdb.afterCategoryEntrySave",
                                        $l_daoStor,
                                        $l_entry_id,
                                        !!$l_entry_id,
                                        $l_objID,
                                        $l_import_data,
                                        isset($l_changes) ? $l_changes : []
                                    );
                            }
                        }
                    }

                    if (isset($l_hw_data['application']))
                    {
                        $output->writeln("Processing applications");
                        $this->m_log->info("Processing applications");
                        $l_check_data = $l_double_assigned = [];
                        $l_res_app    = $l_daoCMDB->retrieve(
                            "SELECT isys_obj__title,
							isys_catg_application_list__id,
							isys_obj__id,
							isys_obj__isys_obj_type__id,
							isys_catg_application_list__isys_catg_application_type__id,
							isys_catg_application_list__isys_catg_application_priority__id,
							isys_catg_application_list__isys_catg_version_list__id,
							isys_catg_version_list__title " . "FROM isys_catg_application_list " .
                            "INNER JOIN isys_connection ON isys_connection__id = isys_catg_application_list__isys_connection__id " .
                            "INNER JOIN isys_obj ON isys_connection__isys_obj__id = isys_obj__id " .
                            "INNER JOIN isys_obj_type ON isys_obj__isys_obj_type__id = isys_obj_type__id " .
                            "LEFT JOIN isys_catg_version_list ON isys_catg_application_list__isys_catg_version_list__id = isys_catg_version_list__id " .
                            "WHERE isys_catg_application_list__isys_obj__id = " . $l_daoCMDB->convert_sql_id(
                                $l_objID
                            ) . " " . "AND (isys_obj_type__id = " . $l_daoCMDB->convert_sql_id(C__OBJTYPE__APPLICATION) . " OR
							isys_obj_type__id = " . $l_daoCMDB->convert_sql_id(C__OBJTYPE__OPERATING_SYSTEM) . ");"
                        );

                        while ($l_rowApp = $l_res_app->get_row())
                        {
                            if (isset($l_check_data[$l_rowApp['isys_obj__id']]))
                            {
                                $l_double_assigned[] = [
                                    'data_id' => $l_rowApp['isys_catg_application_list__id']
                                ];
                            } // if

                            $l_check_data[$l_rowApp['isys_obj__id']] = [
                                'data_id'       => $l_rowApp['isys_catg_application_list__id'],
                                'obj_type'      => $l_rowApp['isys_obj__isys_obj_type__id'],
                                'type'          => $l_rowApp['isys_catg_application_list__isys_catg_application_type__id'],
                                'priority'      => $l_rowApp['isys_catg_application_list__isys_catg_application_priority__id'],
                                'version'       => $l_rowApp['isys_catg_application_list__isys_catg_version_list__id'],
                                'version_title' => $l_rowApp['isys_catg_version_list__title']
                            ];
                        }
                        $l_swIDs = $l_already_updated = [];
                        // Assign Application
                        foreach ($l_hw_data['application'] AS $l_row)
                        {
                            $l_swID = false;

                            $l_app_objtype = ($l_row['COMMENTS'] === 'IOS') ? C__OBJTYPE__OPERATING_SYSTEM : C__OBJTYPE__APPLICATION;

                            $l_row['VERSION']  = trim($l_row['VERSION']);
                            $l_row['COMMENTS'] = trim($l_row['COMMENTS']);
                            $l_row['NAME']     = trim($l_row['NAME']);

                            $l_resSW = $l_daoCMDB->retrieve(
                                "SELECT isys_obj__id, isys_cats_application_list.* " . "FROM isys_obj " .
                                "LEFT JOIN isys_cats_application_list ON isys_obj__id = isys_cats_application_list__isys_obj__id " . "WHERE isys_obj__title = " .
                                $l_daoCMDB->convert_sql_text(
                                    $l_row["NAME"]
                                ) . " " . "AND isys_obj__isys_obj_type__id = " . $l_daoCMDB->convert_sql_id($l_app_objtype) . ";"
                            );

                            if ($l_resSW->num_rows() > 0)
                            {
                                // Application object exists
                                $l_app_data = $l_resSW->get_row();
                                $l_swID     = $l_app_data['isys_obj__id'];
                                if ($l_app_data['isys_cats_application_list__id'] > 0 && !in_array($l_app_data['isys_cats_application_list__id'], $l_already_updated))
                                {
                                    $l_changed_data  = [];
                                    $l_specific_data = [
                                        'data_id'          => $l_app_data['isys_cats_application_list__id'],
                                        'specification'    => $l_app_data['isys_cats_application_list__specification'],
                                        'installation'     => $l_app_data['isys_cats_application_list__isys_installation_type__id'],
                                        'registration_key' => $l_app_data['isys_cats_application_list__registration_key'],
                                        'manufacturer'     => $l_app_data['isys_cats_application_list__isys_application_manufacturer__id'],
                                        'install_path'     => $l_app_data['isys_cats_application_list__install_path'],
                                        'description'      => $l_app_data['isys_cats_application_list__description']
                                    ];

                                    if ($l_row['COMMENTS'] != '' && $l_row['COMMENTS'] != $l_app_data['isys_cats_application_list__description'])
                                    {
                                        $l_changed_data['isys_cmdb_dao_category_s_application::description'] = [
                                            'from' => $l_specific_data['description'],
                                            'to'   => $l_row['COMMENTS']
                                        ];
                                        $l_specific_data['description']                                      = $l_row['COMMENTS'];
                                    } // if

                                    // Update specific category of application
                                    if (count($l_changed_data) > 0)
                                    {
                                        $l_daoApp_s->save(
                                            $l_specific_data['data_id'],
                                            C__RECORD_STATUS__NORMAL,
                                            $l_specific_data['specification'],
                                            $l_specific_data['manufacturer'],
                                            null,
                                            $l_specific_data['description'],
                                            $l_specific_data['installation'],
                                            $l_specific_data['registration_key'],
                                            $l_specific_data['install_path']
                                        );

                                        if ($l_logb_active)
                                        {
                                            isys_event_manager::getInstance()
                                                ->triggerCMDBEvent(
                                                    'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                    "-modified from OCS-",
                                                    $l_swID,
                                                    C__OBJTYPE__APPLICATION,
                                                    'LC__CMDB__CATS__APPLICATION',
                                                    serialize($l_changed_data)
                                                );
                                        } // if
                                    } // if
                                    $l_already_updated[] = $l_app_data['isys_cats_application_list__id'];
                                }
                                elseif (!$l_app_data['isys_cats_application_list__id'])
                                {

                                    $l_app_data['isys_cats_application_list__id'] = $l_daoApp_s->create(
                                        $l_swID,
                                        C__RECORD_STATUS__NORMAL,
                                        null,
                                        null,
                                        null,
                                        $l_row['COMMENTS'],
                                        null,
                                        null,
                                        null
                                    );
                                    $l_already_updated[]                          = $l_app_data['isys_cats_application_list__id'];
                                }
                            }
                            else if ($l_regApp == "0")
                            {
                                // Creat new application object
                                $l_swID = $l_daoCMDB->insert_new_obj($l_app_objtype, false, $l_row["NAME"], null, C__RECORD_STATUS__NORMAL);
                                $l_daoApp_s->create($l_swID, C__RECORD_STATUS__NORMAL, null, null, null, $l_row["COMMENTS"], null, null, null);
                                isys_event_manager::getInstance()
                                    ->triggerCMDBEvent("C__LOGBOOK_EVENT__OBJECT_CREATED", "-object imported from OCS-", $l_swID, $l_app_objtype);
                            }

                            $l_version_id = null;

                            // Check, if the found application version has already been created.
                            if ($l_swID && !empty($l_row["VERSION"]))
                            {
                                // Check, if the version has been created.
                                $l_app_version_sql = 'SELECT isys_catg_version_list__id FROM isys_catg_version_list
			                        WHERE isys_catg_version_list__isys_obj__id = ' . $l_daoCMDB->convert_sql_id($l_swID) . '
			                        AND isys_catg_version_list__title LIKE ' . $l_daoCMDB->convert_sql_text($l_row["VERSION"]) . ' LIMIT 1;';

                                $l_res = $l_daoCMDB->retrieve($l_app_version_sql);

                                if (count($l_res))
                                {
                                    $l_version_id = $l_res->get_row_value('isys_catg_version_list__id');
                                }
                                else
                                {
                                    $l_version_id = isys_cmdb_dao_category_g_version::instance($this->container->database)
                                        ->create($l_swID, C__RECORD_STATUS__NORMAL, $l_row["VERSION"]);
                                } // if
                            } // if

                            if ($l_swID && !in_array($l_swID, $l_swIDs))
                            {
                                $l_swIDs[] = $l_swID;
                                if (count($l_check_data) > 0 && isset($l_check_data[$l_swID]))
                                {
                                    if ($l_check_data[$l_swID]['obj_type'] == C__OBJTYPE__OPERATING_SYSTEM)
                                    {
                                        if ($l_check_data[$l_swID]['priority'] !== C__CATG__APPLICATION_PRIORITY__PRIMARY)
                                        {
                                            // Update operating system
                                            $l_update = 'UPDATE isys_catg_application_list SET isys_catg_application_list__isys_catg_application_priority__id = ' .
                                                $l_daoApp->convert_sql_id(
                                                    C__CATG__APPLICATION_PRIORITY__PRIMARY
                                                ) . ' WHERE isys_catg_application_list__id = ' . $l_daoApp->convert_sql_id($l_check_data[$l_swID]['data_id']);
                                            $l_daoApp->update($l_update);

                                            $l_model_data = $l_daoApp->retrieve(
                                                'SELECT isys_catg_model_list__firmware AS firmware, isys_catg_model_list__id AS id FROM isys_catg_model_list WHERE isys_catg_model_list__isys_obj__id = ' .
                                                $l_daoApp->convert_sql_id(
                                                    $l_objID
                                                )
                                            )
                                                ->get_row();
                                            if (empty($l_model_data['firmware']))
                                            {
                                                $l_update = 'UPDATE isys_catg_model_list SET isys_catg_model_list__firmware = ' . $l_daoApp->convert_sql_text(
                                                        $l_row['NAME']
                                                    ) . ' WHERE isys_catg_model_list__id = ' . $l_daoApp->convert_sql_id($l_model_data['id']);
                                                $l_daoApp->update($l_update);
                                            } // if
                                        } // if
                                    }

                                    if ((int) $l_check_data[$l_swID]['version'] !== (int) $l_version_id)
                                    {
                                        // Update version
                                        $l_update = 'UPDATE isys_catg_application_list SET isys_catg_application_list__isys_catg_version_list__id = ' .
                                            $l_daoApp->convert_sql_id(
                                                $l_version_id
                                            ) . ' WHERE isys_catg_application_list__id = ' . $l_daoApp->convert_sql_id($l_check_data[$l_swID]['data_id']);
                                        $l_daoApp->update($l_update);

                                        if ($l_logb_active)
                                        {
                                            $l_changed_data['isys_cmdb_dao_category_g_application::application']      = [
                                                'from' => $l_row["NAME"],
                                                'to'   => $l_row["NAME"]
                                            ];
                                            $l_changed_data['isys_cmdb_dao_category_g_application::assigned_version'] = [
                                                'from' => $l_check_data[$l_swID]['version_title'],
                                                'to'   => $l_row['VERSION']
                                            ];
                                            isys_event_manager::getInstance()
                                                ->triggerCMDBEvent(
                                                    'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                    "-modified from OCS-",
                                                    $l_objID,
                                                    $l_thisObjTypeID,
                                                    'LC__CMDB__CATG__APPLICATION',
                                                    serialize($l_changed_data)
                                                );
                                        } // if
                                    } // if

                                    // Application found
                                    unset($l_check_data[$l_swID]);
                                    continue;
                                } /*else{
									$l_data = array(
										'application' => $l_swID
									);
								}*/

                                if ($l_logb_active)
                                {
                                    $l_category_values[isys_import_handler_cmdb::C__PROPERTIES] = ['application' => ['value' => $l_swID]];

                                    $l_changes = $l_dao_logb->prepare_changes($l_daoApp, null, $l_category_values);

                                    if (count($l_changes) > 0)
                                    {
                                        isys_event_manager::getInstance()
                                            ->triggerCMDBEvent(
                                                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                                "-modified from OCS-",
                                                $l_objID,
                                                $l_thisObjTypeID,
                                                'LC__CMDB__CATG__APPLICATION',
                                                serialize($l_changes)
                                            );
                                    }
                                }

                                //$l_daoApp->sync($l_daoApp->parse_import_array($l_data), $l_objID, isys_import_handler_cmdb::C__CREATE);

                                // First create relation
                                $l_relation_obj = $l_relation_dao->create_object(
                                    $l_relation_dao->format_relation_name(
                                        $l_inventory["NAME"],
                                        $l_row['NAME'],
                                        $l_relation_data["isys_relation_type__master"]
                                    ),
                                    C__OBJTYPE__RELATION,
                                    C__RECORD_STATUS__NORMAL
                                );

                                $l_sql = "INSERT INTO isys_catg_relation_list " . "SET " . "isys_catg_relation_list__isys_obj__id = " . $l_daoCMDB->convert_sql_id(
                                        $l_relation_obj
                                    ) . ", " . "isys_catg_relation_list__isys_obj__id__master = " . $l_daoCMDB->convert_sql_id(
                                        $l_objID
                                    ) . ", " . "isys_catg_relation_list__isys_obj__id__slave = " . $l_daoCMDB->convert_sql_id(
                                        $l_swID
                                    ) . ", " . "isys_catg_relation_list__isys_relation_type__id = '" . C__RELATION_TYPE__SOFTWARE . "', " .
                                    "isys_catg_relation_list__isys_weighting__id = '" . C__WEIGHTING__5 . "', " . "isys_catg_relation_list__status = '" .
                                    C__RECORD_STATUS__NORMAL . "' " . ";";

                                if ($l_daoApp->update($l_sql))
                                {
                                    $l_relation_id = $l_daoApp->get_last_insert_id();

                                    // Secondly insert new application entry with relation id
                                    $l_sql = "INSERT INTO isys_catg_application_list SET
										isys_catg_application_list__isys_connection__id = " . $l_daoCMDB->convert_sql_id(
                                            isys_cmdb_dao_connection::instance($this->container->database)
                                                ->add_connection($l_swID)
                                        ) . ",
										isys_catg_application_list__status = '" . C__RECORD_STATUS__NORMAL . "',
										isys_catg_application_list__isys_catg_relation_list__id = " . $l_daoApp->convert_sql_id($l_relation_id) . ",
										isys_catg_application_list__isys_catg_version_list__id = " . $l_daoApp->convert_sql_id($l_version_id) . ",
										isys_catg_application_list__isys_obj__id = " . $l_daoApp->convert_sql_id($l_objID) . " ";

                                    if ($l_app_objtype == C__OBJTYPE__OPERATING_SYSTEM)
                                    {
                                        $l_sql .= ", isys_catg_application_list__isys_catg_application_type__id = " . $l_daoApp->convert_sql_id(
                                                C__CATG__APPLICATION_TYPE__OPERATING_SYSTEM
                                            ) . ", " . "isys_catg_application_list__isys_catg_application_priority__id = " . $l_daoApp->convert_sql_id(
                                                C__CATG__APPLICATION_PRIORITY__PRIMARY
                                            ) . ";";

                                        $l_model_data = $l_daoApp->retrieve(
                                            'SELECT isys_catg_model_list__firmware AS firmware, isys_catg_model_list__id AS id FROM isys_catg_model_list WHERE isys_catg_model_list__isys_obj__id = ' .
                                            $l_daoApp->convert_sql_id(
                                                $l_objID
                                            )
                                        )
                                            ->get_row();
                                        if (empty($l_model_data['firmware']))
                                        {
                                            $l_update = 'UPDATE isys_catg_model_list SET isys_catg_model_list__firmware = ' . $l_daoApp->convert_sql_text(
                                                    $l_row['NAME']
                                                ) . ' WHERE isys_catg_model_list__id = ' . $l_daoApp->convert_sql_id($l_model_data['id']);
                                            $l_daoApp->update($l_update);
                                        } // if
                                    } // if

                                    $l_daoApp->update($l_sql) && $l_daoApp->apply_update();
                                }
                            }
                        }

                        // Detach Applications
                        if ($l_regAppAssign == "1")
                        {
                            if (count($l_check_data) > 0)
                            {
                                $ocsHardware->delete_entries_from_category(
                                    $l_check_data,
                                    $l_daoApp,
                                    'isys_catg_application_list',
                                    $l_objID,
                                    $l_thisObjTypeID,
                                    'LC__CMDB__CATG__APPLICATION',
                                    $l_logb_active
                                );
                            }
                            if (count($l_double_assigned) > 0)
                            {
                                $ocsHardware->delete_entries_from_category(
                                    $l_double_assigned,
                                    $l_daoApp,
                                    'isys_catg_application_list',
                                    $l_objID,
                                    $l_thisObjTypeID,
                                    'LC__CMDB__CATG__APPLICATION',
                                    $l_logb_active
                                );
                            }
                        }
                    }

                    // Import Connection between ports
                    if (count($l_port_connections) > 0)
                    {
                        $output->writeln("Updating port connections.", true);
                        $this->m_log->info("Updating port connections.");
                        foreach ($l_port_connections AS $l_port_id => $l_connected_to)
                        {
                            $l_device_name = $l_connected_to[0];
                            $l_port        = $l_connected_to[1];
                            $l_mac         = $l_connected_to[2];
                            //list($l_device_name, $l_port, $l_mac) = explode('|', $l_connected_to);

                            //$l_connected_obj_id = $l_daoCMDB->get_object_by_hostname_serial_mac(null, null, $l_mac, $l_device_name);

                            $l_connected_obj_id = $this->get_object_id_by_matching(null, [$l_mac], $l_device_name, null, (Mac::getBit() + ObjectTitle::getBit()), 2, $output);

                            $l_cable_id = null;

                            if ($l_connected_obj_id)
                            {
                                $l_main_sql = 'SELECT isys_catg_port_list__id AS port_id, isys_catg_port_list__isys_catg_connector_list__id AS con_id, isys_catg_connector_list__isys_cable_connection__id AS cable_con
                                    FROM isys_catg_port_list
                                    INNER JOIN isys_catg_connector_list ON isys_catg_connector_list__id = isys_catg_port_list__isys_catg_connector_list__id
                                    WHERE ';
                                $l_sql      = $l_main_sql . ' isys_catg_port_list__isys_obj__id = ' . $l_daoPort->convert_sql_id($l_connected_obj_id);
                                if ($l_port)
                                {
                                    $l_sql .= ' AND isys_catg_port_list__title = ' . $l_daoPort->convert_sql_text($l_port);
                                } // if

                                if ($l_mac)
                                {
                                    $l_sql .= ' AND isys_catg_port_list__mac = ' . $l_daoPort->convert_sql_text($l_mac);
                                } // if
                                $l_connected_port = $l_daoPort->retrieve($l_sql)
                                    ->get_row();

                                $l_sql       = $l_main_sql . ' isys_catg_port_list__id = ' . $l_daoPort->convert_sql_id($l_port_id);
                                $l_main_port = $l_daoPort->retrieve($l_sql)
                                    ->get_row();

                                // Check if port is not assigned
                                if ($l_connected_port['cable_con'] != $l_main_port['cable_con'] || $l_connected_port['cable_con'] === null ||
                                    $l_main_port['cable_con'] === null)
                                {
                                    if ($l_main_port['cable_con'] !== null)
                                    {
                                        $l_cable_id = $l_daoCableCon->get_cable_object_id_by_connection_id($l_main_port['cable_con']);
                                        $l_daoCableCon->delete_cable_connection($l_main_port['cable_con']);
                                    } // if
                                    if ($l_connected_port['cable_con'] !== null)
                                    {
                                        if (!$l_cable_id)
                                        {
                                            $l_cable_id = $l_daoCableCon->get_cable_object_id_by_connection_id($l_main_port['cable_con']);
                                        }
                                        $l_daoCableCon->delete_cable_connection($l_connected_port['cable_con']);
                                    } // if

                                    if (!$l_cable_id)
                                    {
                                        $l_cable_id = isys_cmdb_dao_cable_connection::add_cable();
                                    } // if

                                    $l_daoPort->connection_save($l_main_port['con_id'], $l_connected_port['con_id'], $l_cable_id);
                                } // if
                            } // if
                        } // foreach
                    } // if

                    if (count($l_port_descriptions) > 0)
                    {
                        $output->writeln("Updating port descriptions.", true);
                        $this->m_log->info("Updating port descriptions.");
                        foreach ($l_port_descriptions AS $l_port_id => $l_description)
                        {
                            $l_update = 'UPDATE isys_catg_port_list SET isys_catg_port_list__description = ' . $l_daoCMDB->convert_sql_text(
                                    $l_description
                                ) . ' WHERE isys_catg_port_list__id = ' . $l_daoCMDB->convert_sql_id($l_port_id);
                            $l_daoCMDB->update($l_update);
                        } // foreach
                        $l_daoCMDB->apply_update();
                    } // if

                    if ($l_object_updated === true)
                    {
                        $output->writeln($l_inventory["NAME"] . " succesfully updated\n\n");
                        $this->m_log->info("\"" . $l_inventory["NAME"] . "\" succesfully updated.");
                    }
                    else
                    {
                        $output->writeln($l_inventory["NAME"] . " succesfully imported\n\n");
                        $this->m_log->info("\"" . $l_inventory["NAME"] . "\" succesfully imported.");
                    } // if

                    $this->m_log->flush_log(true, false);
                } // foreach
            } // if

            $l_daoCMDB->apply_update();
            $output->writeln("Import successful");
            $this->m_log->info("Import successful");

            /**
             * Create index for imported/updated objects, based on the start time of this import
             *
             * @todo prepare a list of imported categories so that not every category is indexed again
             */
            isys_component_signalcollection::get_instance()
                ->emit('mod.cmdb.afterLegacyImport', $startTime);
        }
        catch (Exception $e)
        {
            $l_daoCMDB->cancel_update();
            $output->writeln("Import failed");
            $this->m_log->info("Import failed");
            throw $e;
        }

        $this->m_log->flush_log();
    }

    /**
     * @author Van Quyen Hoang <qhoang@i-doit.org>
     * @return int|boolean
     */
    private function getOcsDatabaseId(InputInterface $input)
    {
        if (!empty($_SERVER['HTTP_HOST']))
        {
            return isys_glob_get_param('selected_ocsdb');
        }
        else
        {
            if (($databaseSchema = $input->getOption('databaseSchema')) !== null)
            {
                if (is_numeric($databaseSchema))
                {
                    return $databaseSchema;
                }
                elseif (is_string($databaseSchema))
                {
                    return ((new isys_component_dao_ocs($this->container->database))->get_ocs_db_id_by_schema($databaseSchema) ?: null);
                }
            }
            else
            {
                return isys_tenantsettings::get('ocs.default.db', null);
            }
        }
        return false;
    }

    /**
     * Retrieve object id by matching Serial, Mac, Object title and optional Object type
     *
     * @param string $p_serial
     * @param array  $p_macaddresses
     * @param string $p_objectTitle
     * @param int    $p_objecttype
     *
     * @return bool|int
     * @author   Van Quyen Hoang <qhoang@i-doit.com>
     */
    private function get_object_id_by_matching($p_serial = null, $p_macaddresses = [], $p_objectTitle = null, $p_objecttype = null, $p_bits = null, $p_minMatch = null, OutputInterface $output = null)
    {
        $l_match         = null;
        $l_matchKeywords = [];

        if ($p_bits !== null && $p_minMatch !== null)
        {
            $l_ciMatcherKey = $p_bits . '_' . $p_minMatch;
            if (!isset($this->ciMatcher[$l_ciMatcherKey]))
            {
                $l_virtualConfig = new MatchConfig($this->container);
                $l_virtualConfig->setBits($p_bits)
                    ->setMinMatch($p_minMatch);
                $this->ciMatcher[$l_ciMatcherKey] = new CiMatcher($l_virtualConfig);
                $this->m_log->debug('Using virtual object matcher for object matching.');
                $output->writeln('Using virtual object matcher for object matching.');
            } // if
        }
        else
        {
            $l_ciMatcherKey = 'main';
            if ($this->ciMatcher[$l_ciMatcherKey] === null)
            {
                $l_matchConfig                    = MatchConfig::factory(isys_tenantsettings::get('ocs.object-matching', 1), $this->container);
                $this->ciMatcher[$l_ciMatcherKey] = new CiMatcher($l_matchConfig);
                $this->m_log->info('Using object match profile "' . $l_matchConfig->getTitle() . '" for object matching.');
                $output->writeln('Using object match profile "' . $l_matchConfig->getTitle() . '" for object matching.');
            } // if
        } // if

        /**
         * TypeHinting
         *
         * @var $l_ciMatcher CiMatcher
         */
        $l_ciMatcher = $this->ciMatcher[$l_ciMatcherKey];

        if ($p_objectTitle !== null)
        {
            // Object title
            $l_matchKeywords[] = new MatchKeyword(ObjectTitle::KEY, $p_objectTitle);

            if (strpos($p_objectTitle, '.') !== false)
            {
                // Possible FQDN
                $l_matchKeywords[] = new MatchKeyword(Fqdn::KEY, $p_objectTitle);

                // Hostname
                $l_matchKeywords[] = new MatchKeyword(Hostname::KEY, substr($p_objectTitle, 0, strpos($p_objectTitle, '.') + 1));
                $l_matchKeywords[] = new MatchKeyword(Hostname::KEY, $p_objectTitle);
            }
            else
            {
                // Hostname
                $l_matchKeywords[] = new MatchKeyword(Hostname::KEY, $p_objectTitle);
            }
        } // if

        if ($p_serial !== null)
        {
            $l_matchKeywords[] = new MatchKeyword(ModelSerial::KEY, $p_serial);
        } // if

        if (count($p_macaddresses))
        {
            foreach ($p_macaddresses AS $l_mac)
            {
                $l_matchKeywords[] = new MatchKeyword(Mac::KEY, $l_mac);
            } // foreach
        } // if

        if ($p_objecttype !== null)
        {
            $l_matchKeywords[] = new MatchKeyword(ObjectTypeConst::KEY, $p_objecttype);
        } // if

        if (count($l_matchKeywords))
        {
            $l_match = $l_ciMatcher->match($l_matchKeywords);
        } // if

        if ($l_match !== null)
        {
            $l_object_id = $l_match->getId();

            if ($l_object_id)
            {
                $this->m_log->info('Found Object "' . $p_objectTitle . '" with ID "' . $l_object_id . '" via object matching.');
                $output->writeln('Found Object "' . $p_objectTitle . '" with ID "' . $l_object_id . '".');
                if ($l_match->getMatchCount() > 1)
                {
                    $l_otherMatches = $l_match->getMatchResult();
                    $this->m_log->debug('Other matches found for Object "' . $p_objectTitle . '". Please check the following objects:');
                    foreach ($l_otherMatches AS $l_otherMatch)
                    {
                        $this->m_log->debug('- ' . $l_otherMatch->getTitle() . ' (' . $l_otherMatch->getId() . ')');
                    } // foreach
                } // if

                return $l_object_id;
            } // if
        } // if

        $this->m_log->info('Object "' . $p_objectTitle . '" not found.');
        $output->writeln('Object "' . $p_objectTitle . '" not found.');

        return false;
    } // function

    /**
     * Builds array with all information that is neede for the import
     *
     * @param isys_component_dao_ocs $p_dao
     * @param array                  $p_hardwareids
     * @param array                  $p_categories
     *
     * @return array
     * @author Van Quyen Hoang <qhoang@i-doit.org>
     */
    private function get_hardware_info($p_dao, $p_hardwareid = null, $p_add_model = false, $p_add_memory = false, $p_add_application = false, $p_add_graphic = false, $p_add_sound = false, $p_add_storage = false, $p_add_net = false, $p_add_ui = false, $p_add_drive = false, $p_add_vm = false)
    {
        $l_memory  = [];
        $l_app     = [];
        $l_graphic = [];
        $l_sound   = [];
        $l_drive   = [];
        $l_net     = [];
        $l_ui      = [];
        $l_stor    = [];

        $l_data = [
            'inventory' => $p_dao->getHardwareItem($p_hardwareid)
        ];

        $l_data['inventory']['macaddr'] = $p_dao->get_unique_mac_addresses($p_hardwareid);

        if ($p_add_model)
        {
            $l_data['model'] = $p_dao->getBios($p_hardwareid)
                ->__to_array();
        }

        if ($p_add_memory)
        {
            $l_res_ocs_memory = $p_dao->getMemory($p_hardwareid);
            while ($l_row = $l_res_ocs_memory->get_row())
            {
                $l_memory[] = $l_row;
            }
            $l_data['memory'] = $l_memory;
        }

        if ($p_add_application)
        {
            $l_res_software = $p_dao->getSoftware($p_hardwareid);
            while ($l_row = $l_res_software->get_row())
            {
                $l_app[] = $l_row;
            }
            $l_data['application'] = $l_app;
        }

        if ($p_add_graphic)
        {
            $l_res_ocs_graphic = $p_dao->getGraphicsAdapter($p_hardwareid);
            while ($l_row = $l_res_ocs_graphic->get_row())
            {
                $l_graphic[] = $l_row;
            }
            $l_data['graphic'] = $l_graphic;
        }

        if ($p_add_sound)
        {
            $l_res_ocs_sound = $p_dao->getSoundAdapter($p_hardwareid);
            while ($l_row = $l_res_ocs_sound->get_row())
            {
                $l_sound[] = $l_row;
            }
            $l_data['sound'] = $l_sound;
        }

        if ($p_add_drive)
        {
            $l_res_ocs_drive = $p_dao->getDrives($p_hardwareid);
            while ($l_row = $l_res_ocs_drive->get_row())
            {
                $l_drive[] = $l_row;
            }
            $l_data['drive'] = $l_drive;
        }

        if ($p_add_net)
        {
            $l_res_net = $p_dao->getNetworkAdapter($p_hardwareid);
            while ($l_row = $l_res_net->get_row())
            {
                $l_net[] = $l_row;
            }
            $l_data['net'] = $l_net;
        }

        if ($p_add_ui)
        {
            $l_res_ui = $p_dao->getPorts($p_hardwareid);
            while ($l_row = $l_res_ui->get_row())
            {
                $l_ui[] = $l_row;
            }
            $l_data['ui'] = $l_ui;
        }

        if ($p_add_storage)
        {
            $l_res_stor = $p_dao->getStorage($p_hardwareid);
            while ($l_row = $l_res_stor->get_row())
            {
                $l_stor[] = $l_row;
            }
            $l_data['stor'] = $l_stor;
        }

        if ($p_add_vm)
        {
            $l_res_vm = $p_dao->getVirtualMachines($p_hardwareid);
            while ($l_row = $l_res_vm->get_row())
            {
                $l_vm[] = $l_row;
            }
            $l_data['virtual_machine'] = $l_vm;
        }

        return $l_data;
    } // function

    /**
     * @param isys_component_dao_ocs $p_dao
     * @param null                   $p_snmp_id
     * @param bool                   $p_add_memory
     * @param bool                   $p_add_storage
     * @param bool                   $p_add_net
     * @param                        $p_add_cpu
     * @param bool                   $p_add_ui
     * @param bool                   $p_add_model
     * @param bool                   $p_add_application
     * @param bool                   $p_add_graphic
     * @param bool                   $p_add_sound
     * @param bool                   $p_add_drive
     * @param bool                   $p_add_vm
     *
     * @return array
     */
    private function get_snmp_info($p_dao, $p_snmp_id = null, $p_add_memory = false, $p_add_storage = false, $p_add_net = false, $p_add_cpu, $p_add_ui = false, $p_add_model = false, $p_add_application = false, $p_add_graphic = false, $p_add_sound = false, $p_add_drive = false, $p_add_vm = false)
    {
        $l_data = [];

        if ($p_dao->does_snmp_exist())
        {
            $l_memory  = [];
            $l_app     = [];
            $l_graphic = [];
            $l_sound   = [];
            $l_drive   = [];
            $l_net     = [];
            $l_ui      = [];
            $l_stor    = [];
            $l_cpu     = [];

            /**
             * @var $p_dao isys_component_dao_ocs
             */
            $l_data = [
                'inventory' => $p_dao->getHardwareItemBySNMP($p_snmp_id)
            ];

            if (isset($this->m_already_imported_snmp_devices[trim($l_data['inventory']['NAME'])]))
            {
                return false;
            } // if

            $this->m_already_imported_snmp_devices[trim($l_data['inventory']['NAME'])] = true;
            $l_data['inventory']['macaddr']                                            = $p_dao->get_unique_mac_addresses($p_snmp_id, true);

            if ($p_add_cpu)
            {
                $l_res_ocs_cpu = $p_dao->getCPU($p_snmp_id, true);
                while ($l_row = $l_res_ocs_cpu->get_row())
                {
                    $l_speed        = $l_row['SPEED'];
                    $l_row['SPEED'] = preg_replace('/[^0-9.]*/', '', $l_speed);
                    $l_row['UNIT']  = preg_replace('/[0-9.]*/', '', $l_speed);
                    $l_cpu[]        = $l_row;
                }
                $l_data['cpu'] = $l_cpu;
            }

            if ($p_add_memory)
            {
                $l_res_ocs_memory = $p_dao->getMemory($p_snmp_id, true);
                while ($l_row = $l_res_ocs_memory->get_row())
                {
                    $l_memory[] = $l_row;
                }
                $l_data['memory'] = $l_memory;
            }

            if ($p_add_net)
            {
                $l_res_net     = $p_dao->getNetworkAdapter($p_snmp_id, true);
                $l_subnetmasks = [];
                $l_counter     = 0;
                $l_addresses   = [];
                $l_primary_set = false;
                while ($l_row = $l_res_net->get_row())
                {
                    if ($l_row['DEVICENAME'] !== '' && $l_row['DEVICEPORT'] !== '')
                    {
                        $l_connected_to = $p_dao->getNetworkConnectedTo($l_row['DEVICENAME'], $l_row['DEVICEPORT']);
                        if (is_array($l_connected_to))
                        {
                            $l_row += $l_connected_to;
                        } // if
                    } // if

                    if ($l_row['IPADDR'] !== '')
                    {
                        $l_addresses[$l_row['IPADDR']] = true;
                        $l_row['PRIMARY']              = ($l_primary_set === false) ? true : false;
                        $l_primary_set                 = true;
                    }
                    $l_net[$l_counter]                                                      = $l_row;
                    $l_subnetmasks[Ip::validate_net_ip($l_row['IPADDR'], $l_row['IPMASK'])] = $l_row['IPMASK'];
                    $l_counter++;
                }
                if ($l_data['inventory']['IPADDR'] != '' && !isset($l_addresses[$l_data['inventory']['IPADDR']]))
                {
                    $l_subnetmask = '255.255.255.0';
                    if (count($l_subnetmasks) > 0)
                    {
                        $l_cache_net_ip_arr    = explode('.', $l_data['inventory']['IPADDR']);
                        $l_cache_net_ip_first  = $l_cache_net_ip_arr[0] . '.' . $l_cache_net_ip_arr[1] . '.' . $l_cache_net_ip_arr[2] . '.0';
                        $l_cache_net_ip_second = $l_cache_net_ip_arr[0] . '.' . $l_cache_net_ip_arr[1] . '.0.0';
                        $l_cache_net_ip_third  = $l_cache_net_ip_arr[0] . '.0.0.0';
                        if (isset($l_subnetmasks[$l_cache_net_ip_first]))
                        {
                            $l_subnetmask = $l_subnetmasks[$l_cache_net_ip_first];
                        }
                        elseif (isset($l_subnetmasks[$l_cache_net_ip_second]))
                        {
                            $l_subnetmask = $l_subnetmasks[$l_cache_net_ip_second];
                        }
                        elseif (isset($l_subnetmasks[$l_cache_net_ip_third]))
                        {
                            $l_subnetmask = $l_subnetmasks[$l_cache_net_ip_third];
                        }
                    }

                    $l_net[] = [
                        'IPADDR'   => $l_data['inventory']['IPADDR'],
                        'IPMASK'   => $l_subnetmask,
                        'IPSUBNET' => Ip::validate_net_ip($l_data['inventory']['IPADDR'], $l_subnetmask, null, true),
                        'STATUS'   => 'UP',
                        'PRIMARY'  => ($l_primary_set === false) ? true : false
                    ];
                }

                $l_res_interface = $p_dao->getSNMPNetworkInterfaces($p_snmp_id);
                while ($l_row = $l_res_interface->get_row())
                {
                    $l_net['interfaces'][] = $l_row;
                } // while

                $l_res_ps = $p_dao->getSNMPPowerSupplies($p_snmp_id);
                while ($l_row = $l_res_ps->get_row())
                {
                    $l_net['powersupplies'][] = $l_row;
                } // while

                $l_data['net'] = $l_net;
            }

            if ($p_add_storage)
            {
                $l_res_stor = $p_dao->getStorage($p_snmp_id, true);
                while ($l_row = $l_res_stor->get_row())
                {
                    $l_stor[] = $l_row;
                }
                $l_data['stor'] = $l_stor;
            }

            if ($p_add_ui)
            {
                $l_res_ui = $p_dao->getPorts($p_snmp_id, true);
                while ($l_row = $l_res_ui->get_row())
                {
                    $l_ui[] = $l_row;
                }
                $l_data['ui'] = $l_ui;
            }

            if ($p_add_model)
            {
                $l_data['model'] = $p_dao->getBios($p_snmp_id, true)
                    ->__as_array();
            }

            if ($p_add_application)
            {
                $l_res_software = $p_dao->getSoftware($p_snmp_id, true);
                while ($l_row = $l_res_software->get_row())
                {
                    $l_app[] = $l_row;
                }
                $l_data['application'] = $l_app;
            }
        }

        return $l_data;
    }
}
