<?php

/**
 * i-doit
 *
 * Base class for popups. All popups are located in the directory "src/classes/popups".
 * The classes in it have to be inherited from this class.
 *
 * @package     i-doit
 * @subpackage  Components
 * @author      Andre Woesten <awoesten@i-doit.de>
 * @version     0.9
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
abstract class isys_component_popup extends isys_component
{
    /**
     * Private configuration array.
     * @var array
     */
    private $m_config;

    /**
     * @var string
     */
    private $m_popupname;

    /**
     * @var isys_component_database
     */
    protected $db;

    /**
     * @var isys_component_template_language_manager
     */
    protected $lang;

    /**
     * Abstract method for handling module request.
     *
     * @param  isys_module_request $p_modreq
     *
     * @return isys_component_template
     */
    abstract public function &handle_module_request(isys_module_request $p_modreq);

    /**
     * Returns a configuration parameter.
     *
     * @param   string $p_var
     *
     * @return  array
     * @return  null
     */
    public function get_config($p_var)
    {
        if (array_key_exists($p_var, $this->m_config)) {
            return $this->m_config[$p_var];
        }

        return null;
    }

    /**
     * Formats the popup selection.
     *
     * @param   integer $p_id
     * @param   boolean $p_plain
     *
     * @return  string
     */
    public function format_selection($p_id, $p_plain = false)
    {
        return '';
    }

    /**
     * Gets popupname.
     *
     * @return  string
     */
    public function get_popupname()
    {
        return $this->m_popupname;
    }

    /**
     * Returns the whole configuration array.
     *
     * @return  array
     */
    public function get_config_array()
    {
        return $this->m_config;
    }

    /**
     * Sets a configuration parameters in the configuration array. Returns null if the key does not exist.
     *
     * Parameters => default value:
     *    "dependant"  => "yes"
     *    "height"     => 500
     *    "width"      => 500
     *    "menubar"    => "no"
     *    "resizable"  => "yes"
     *    "scrollbars" => "yes"
     *    "status"     => "no"
     *    "toolbar"    => "no"
     *    "location"   => "yes"
     *
     * @param   string $p_var
     * @param   string $p_val
     *
     * @return  mixed
     * @author  Niclas Potthast <npotthast@i-doit.org>
     */
    public function &set_config($p_var, $p_val)
    {
        if (array_key_exists($p_var, $this->m_config)) {
            $this->m_config[$p_var] = $p_val;

            return $this->m_config;
        }

        return null;
    }

    /**
     * Process the overlay page.
     *
     * @param  string     $p_url
     * @param  int|string $width     Set exact pixel size via (int) 640 or a relative via "60%".
     * @param  int|string $height    Set exact pixel size via (int) 480 or a relative via "40%".
     * @param  array      $params
     * @param  string     $popupReceiver
     * @param  integer    $minWidth  This will only be used, if a relative size is applied.
     * @param  integer    $minHeight This will only be used, if a relative size is applied.
     * @param  integer    $maxWidth  This will only be used, if a relative size is applied.
     * @param  integer    $maxHeight This will only be used, if a relative size is applied.
     *
     * @return string
     */
    public function process_overlay($p_url, $width = 950, $height = 550, $params = [], $popupReceiver = null, $minWidth = null, $minHeight = null, $maxWidth = null, $maxHeight = null)
    {
        $l_popup = str_replace("isys_popup_", "", get_class($this));
        $l_params = "{params:'" . base64_encode(isys_format_json::encode($params)) . "'}";

        return "get_popup('" . $l_popup . "', '" . $p_url . "', '" . $width . "', '" . $height . "', " . $l_params . ", " .
            ($popupReceiver ? "'" . $popupReceiver . "'" : 'null') . "," .
            ($minWidth ? "'" . $minWidth . "'" : 'null') . "," .
            ($minHeight ? "'" . $minHeight . "'" : 'null') . "," .
            ($maxWidth ? "'" . $maxWidth . "'" : 'null') . "," .
            ($maxHeight ? "'" . $maxHeight . "'" : 'null') . ");";
    }

    /**
     * Returns a javascript function to display the object browser.
     *
     * @param  array      $p_params
     * @param  int|string $width     Set exact pixel size via (int) 640 or a relative via "60%".
     * @param  int|string $height    Set exact pixel size via (int) 480 or a relative via "40%".
     * @param  integer    $minWidth  This will only be used, if a relative size is applied.
     * @param  integer    $minHeight This will only be used, if a relative size is applied.
     * @param  integer    $maxWidth  This will only be used, if a relative size is applied.
     * @param  integer    $maxHeight This will only be used, if a relative size is applied.
     *
     * @return string
     */
    public function get_js_handler($p_params, $width = 1100, $height = 650, $minWidth = null, $minHeight = null, $maxWidth = null, $maxHeight = null)
    {
        return $this->process_overlay('', $width, $height, $p_params, null, $minWidth, $minHeight, $maxWidth, $maxHeight);
    }

    /**
     * Creates a popup-JS-block and returns it.
     *
     * @param   string  $p_uri
     * @param   boolean $p_center
     *
     * @return  string
     * @author  Niclas Potthast <npotthast@i-doit.de>
     */
    public function process($p_uri, $p_center = false)
    {
        // Without a URL just return an empty string.
        if (empty($p_uri)) {
            return '';
        }

        // Build JS call.
        return "isys_popup_open('" . $p_uri . "', '" . $this->m_popupname . "', " . (($p_center) ? 1 : 0) . ", " .
            str_replace('"', "'", isys_format_json::encode($this->m_config)) . ");";
    }

    /**
     * Popup constructor.
     */
    public function __construct()
    {
        $this->db = isys_application::instance()->container->get('database');
        $this->lang = isys_application::instance()->container->get('language');

        $this->m_popupname = 'isysPopup' . rand(10, 50);

        // Default popup configuration.
        $this->m_config = [
            'dependant'  => 'yes',
            'width'      => 500,
            'height'     => 500,
            'menubar'    => 'no',
            'resizable'  => 'yes',
            'scrollbars' => 'yes',
            'status'     => 'no',
            'toolbar'    => 'no',
            'location'   => 'no',
        ];
    }
}