<?php

/**
 * i-doit
 *
 * DAO: global folder category for Check_MK.
 *
 * @package     Modules
 * @subpackage  Check_MK
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.0
 */
class isys_cmdb_dao_category_g_cmk extends isys_cmdb_dao_category_global
{
    /**
     * Category's name. Will be used for the identifier, constant, main table, and many more.
     *
     * @var  string
     */
    protected $m_category = 'cmk';

    /**
     * Category entry is purgable
     *
     * @var  boolean
     */
    protected $m_is_purgable = true;

    /**
     * Callback method for property host.
     *
     * @param   isys_request $p_request
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function callback_property_export_config(isys_request $p_request)
    {
        global $g_comp_database;

        $l_export_config = [];
        $l_export_config_res = isys_monitoring_dao_hosts::instance($g_comp_database)
            ->get_export_data();

        if (count($l_export_config_res)) {
            while ($l_row = $l_export_config_res->get_row()) {
                $l_export_config[$l_row['isys_monitoring_export_config__id']] = $l_row['isys_monitoring_export_config__title'];
            }
        }

        return $l_export_config;
    }

    /**
     * Callback method for property host.
     *
     * @param   isys_request $p_request
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function callback_property_hostaddress(isys_request $p_request)
    {
        global $g_comp_database;

        $l_return = [
            '-1' => isys_application::instance()->container->get('language')
                ->get('LC__CATG__IP__PRIMARY_IP_ADDRESS')
        ];

        $l_hostaddresses = isys_cmdb_dao_category_g_ip::instance($g_comp_database)
            ->get_data(null, $p_request->get_object_id());

        if (count($l_hostaddresses)) {
            while ($l_row = $l_hostaddresses->get_row()) {
                $l_return[$l_row['isys_catg_ip_list__id']] = $l_row['isys_cats_net_ip_addresses_list__title'];
            }
        }

        return $l_return;
    }

    /**
     * This method selects all export configurations, which are used at least once (Doesn't matter if NORMAL, ARCHIVED or DELETED).
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function get_used_export_paths()
    {
        return $this->retrieve('SELECT isys_monitoring_export_config.*
			FROM isys_monitoring_export_config
			INNER JOIN isys_catg_cmk_list ON isys_catg_cmk_list__isys_monitoring_export_config__id = isys_monitoring_export_config__id
			LEFT JOIN isys_obj ON isys_obj__id = isys_catg_cmk_list__isys_obj__id;');
    }

    /**
     * Simply retrieve all export configurations.
     *
     * @return  isys_component_dao_result
     * @throws  isys_exception_database
     */
    public function get_export_configurations()
    {
        $mapping = [
            'isys_monitoring_export_config__id AS id',
            'isys_monitoring_export_config__title AS title',
            'isys_monitoring_export_config__path AS path',
            'isys_monitoring_export_config__address AS address',
            'isys_monitoring_export_config__type AS type',
            'isys_monitoring_export_config__options AS options'
        ];

        return $this->retrieve('SELECT ' . implode(',', $mapping) . ' FROM isys_monitoring_export_config WHERE isys_monitoring_export_config__type = "check_mk";');
    }

    /**
     * Gets all connected object parents
     *
     * @param integer $dataId
     *
     * @return isys_component_dao_result
     * @throws isys_exception_database
     * @author  Van Quyen Hoang <qhoang@i-doit.org>
     */
    public function getExportParents($dataId)
    {
        return $this->retrieve('SELECT obj.*, isys_obj_type__const FROM isys_catg_cmk_list_2_isys_obj AS cmk2ob
          INNER JOIN isys_obj AS obj ON obj.isys_obj__id = cmk2ob.isys_obj__id
          INNER JOIN isys_obj_type ON isys_obj_type__id = obj.isys_obj__isys_obj_type__id 
          WHERE cmk2ob.isys_catg_cmk_list__id = ' . $this->convert_sql_id($dataId));
    }

    /**
     * Gets all connected object parents as array
     *
     * @param integer $dataId
     * @param bool    $fullData
     *
     * @return array
     * @throws isys_exception_database
     * @author  Van Quyen Hoang <qhoang@i-doit.org>
     */
    public function getExportParentsAsArray($dataId, $fullData = false)
    {
        $result = $this->getExportParents($dataId);
        $parents = [];
        if ($result->num_rows()) {
            while ($data = $result->get_row()) {
                $parents[] = ($fullData ? $data : $data['isys_obj__id']);
            }
        }

        return $parents;
    }

    /**
     * Callback method for the attribute p_strSelectedID for the object browser
     *
     * @param isys_request $request
     *
     * @return array
     * @throws isys_exception_database
     * @author  Van Quyen Hoang <qhoang@i-doit.org>
     */
    public function callbackExportParents(isys_request $request)
    {
        $dataId = ($request->get_row('isys_catg_cmk_list') ?: $request->get_category_data_id());
        if (!$dataId) {
            return [];
        }

        return isys_cmdb_dao_category_g_cmk::instance(isys_application::instance()->container->database)
            ->getExportParentsAsArray($dataId);
    }

    /**
     * Updates existing entity.
     *
     * @param integer $entryId
     * @param array   $data
     *
     * @return bool
     * @throws Exception
     * @throws isys_exception_dao
     * @author  Van Quyen Hoang <qhoang@i-doit.org>
     */
    public function save_data($entryId, $data)
    {
        $result = parent::save_data($entryId, $data);

        if ($result && !empty($data['parents'])) {
            if ($entryId) {
                // Delete all connections to the parents
                $this->detachParents($entryId);
                // Add all new parents
                $this->attachParents($entryId, $data['parents']);
            }
        }

        return $result;
    }

    /**
     * Creates new entity
     *
     * @param array $data
     *
     * @return bool|integer
     * @throws Exception
     * @throws isys_exception_dao
     * @author  Van Quyen Hoang <qhoang@i-doit.org>
     */
    public function create_data($data)
    {
        $dataId = parent::create_data($data);

        if ($dataId && !empty($data['parents'])) {
            // Add parents to entry
            $this->attachParents($dataId, $data['parents']);
        }

        return $dataId;
    }

    /**
     * Method for returning the properties.
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    protected function properties()
    {
        return [
            'active'              => array_replace_recursive(isys_cmdb_dao_category_pattern::dialog(), [
                C__PROPERTY__INFO   => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CATG__CMK__ACTIVE',
                    C__PROPERTY__INFO__DESCRIPTION => 'Active'
                ],
                C__PROPERTY__DATA   => [
                    C__PROPERTY__DATA__FIELD  => 'isys_catg_cmk_list__exportable',
                    C__PROPERTY__DATA__SELECT => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory('(CASE WHEN isys_catg_cmk_list__exportable = "1" THEN ' .
                        $this->convert_sql_text('LC__UNIVERSAL__YES') . '
                        	    ELSE ' . $this->convert_sql_text('LC__UNIVERSAL__NO') . ' END)', 'isys_catg_cmk_list'),
                    C__PROPERTY__DATA__JOIN   => [
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_cmk_list', 'LEFT', 'isys_catg_cmk_list__isys_obj__id', 'isys_obj__id')
                    ]
                ],
                C__PROPERTY__UI     => [
                    C__PROPERTY__UI__ID     => 'C__CATG__CMK__ACTIVE',
                    C__PROPERTY__UI__PARAMS => [
                        'p_arData'     => get_smarty_arr_YES_NO(),
                        'p_bDbFieldNN' => true
                    ]
                ],
                C__PROPERTY__FORMAT => [
                    C__PROPERTY__FORMAT__CALLBACK => [
                        'isys_export_helper',
                        'get_yes_or_no'
                    ]
                ]
            ]),
            'export_config'       => array_replace_recursive(isys_cmdb_dao_category_pattern::dialog(), [
                    C__PROPERTY__INFO => [
                        C__PROPERTY__INFO__TITLE       => 'LC__MONITORING__EXPORT__CONFIGURATION',
                        C__PROPERTY__INFO__DESCRIPTION => 'Export configuration'
                    ],
                    C__PROPERTY__DATA => [
                        C__PROPERTY__DATA__FIELD  => 'isys_catg_cmk_list__isys_monitoring_export_config__id',
                        C__PROPERTY__DATA__SELECT => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory('isys_monitoring_export_config__title',
                            'isys_monitoring_export_config'),
                        C__PROPERTY__DATA__JOIN   => [
                            idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_cmk_list', 'LEFT', 'isys_catg_cmk_list__isys_obj__id', 'isys_obj__id'),
                            idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_monitoring_export_config', 'LEFT',
                                'isys_catg_cmk_list__isys_monitoring_export_config__id', 'isys_monitoring_export_config__id')
                        ],
                        C__PROPERTY__DATA__REFERENCES => [
                            'isys_monitoring_export_config',
                            'isys_monitoring_export_config__id',
                        ]
                    ],
                    C__PROPERTY__UI   => [
                        C__PROPERTY__UI__ID     => 'C__CATG__CMK__EXPORT_CONFIG',
                        C__PROPERTY__UI__PARAMS => [
                            'p_strTable' => 'isys_monitoring_export_config'
                        ]
                    ]
                ]),
            'title'               => array_replace_recursive(isys_cmdb_dao_category_pattern::text(), [
                C__PROPERTY__INFO => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CATG__CMK__ALIAS',
                    C__PROPERTY__INFO__DESCRIPTION => 'Alias'
                ],
                C__PROPERTY__DATA => [
                    C__PROPERTY__DATA__FIELD => 'isys_catg_cmk_list__title'
                ],
                C__PROPERTY__UI   => [
                    C__PROPERTY__UI__ID => 'C__CATG__CMK__ALIAS'
                ]
            ]),
            'export_ip'           => array_replace_recursive(isys_cmdb_dao_category_pattern::dialog(), [
                C__PROPERTY__INFO => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CATG__CMK__EXPORT_IP',
                    C__PROPERTY__INFO__DESCRIPTION => 'Export hostaddress'
                ],
                C__PROPERTY__DATA => [
                    C__PROPERTY__DATA__FIELD  => 'isys_catg_cmk_list__export_ip',
                    C__PROPERTY__DATA__SELECT => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory('(CASE WHEN isys_catg_cmk_list__export_ip = \'1\' THEN ' .
                        $this->convert_sql_text('LC__UNIVERSAL__YES') . '
                        	    ELSE ' . $this->convert_sql_text('LC__UNIVERSAL__NO') . ' END)', 'isys_catg_cmk_list'),
                        C__PROPERTY__DATA__JOIN   => [
                            idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_cmk_list', 'LEFT', 'isys_catg_cmk_list__isys_obj__id', 'isys_obj__id')
                        ]
                    ],
                    C__PROPERTY__UI   => [
                        C__PROPERTY__UI__ID     => 'C__CATG__CMK__EXPORT_IP',
                        C__PROPERTY__UI__PARAMS => [
                            'p_arData'     => get_smarty_arr_YES_NO(),
                            'p_strClass'   => 'input-mini',
                            'p_bDbFieldNN' => 1
                        ]
                    ],
                    C__PROPERTY__FORMAT => [
                        C__PROPERTY__FORMAT__CALLBACK => ['isys_export_helper', 'get_yes_or_no']
                    ]
                ]),
            'hostaddress'         => array_replace_recursive(isys_cmdb_dao_category_pattern::dialog(), [
                C__PROPERTY__INFO   => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CATG__IP_ADDRESS',
                    C__PROPERTY__INFO__DESCRIPTION => 'Hostaddress'
                ],
                C__PROPERTY__DATA   => [
                    C__PROPERTY__DATA__FIELD  => 'isys_catg_cmk_list__isys_catg_ip_list__id',
                    C__PROPERTY__DATA__SELECT => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory('SELECT isys_cats_net_ip_addresses_list__title FROM isys_catg_cmk_list
                              INNER JOIN isys_catg_ip_list ON isys_catg_ip_list__id = isys_catg_cmk_list__isys_catg_ip_list__id
                              INNER JOIN isys_cats_net_ip_addresses_list ON isys_catg_ip_list__isys_cats_net_ip_addresses_list__id = isys_cats_net_ip_addresses_list__id',
                        'isys_catg_cmk_list', 'isys_catg_cmk_list__id', 'isys_catg_cmk_list__isys_obj__id'),
                    C__PROPERTY__DATA__JOIN   => [
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_cmk_list', 'LEFT', 'isys_catg_cmk_list__isys_obj__id', 'isys_obj__id'),
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_ip_list', 'LEFT', 'isys_catg_cmk_list__isys_catg_ip_list__id',
                            'isys_catg_ip_list__id'),
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_cats_net_ip_addresses_list', 'LEFT',
                            'isys_catg_ip_list__isys_cats_net_ip_addresses_list__id', 'isys_cats_net_ip_addresses_list__id')
                    ]
                ],
                C__PROPERTY__UI     => [
                    C__PROPERTY__UI__ID     => 'C__CATG__CMK__HOSTADDRESS',
                    C__PROPERTY__UI__PARAMS => [
                        'p_arData'     => new isys_callback([
                            'isys_cmdb_dao_category_g_cmk',
                            'callback_property_hostaddress'
                        ]),
                        'p_bSort'      => false,
                        'p_bDbFieldNN' => 1
                    ]
                ],
                C__PROPERTY__FORMAT => [
                    C__PROPERTY__FORMAT__CALLBACK => [
                        'isys_export_helper_check_mk',
                        'hostaddress'
                    ]
                ]
            ]),
            'host_name'           => array_replace_recursive(isys_cmdb_dao_category_pattern::text(), [
                C__PROPERTY__INFO     => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CATG__CMK__HOSTNAME',
                    C__PROPERTY__INFO__DESCRIPTION => 'Hostname'
                ],
                C__PROPERTY__DATA     => [
                    C__PROPERTY__DATA__FIELD => 'isys_catg_cmk_list__host_name'
                ],
                C__PROPERTY__UI       => [
                    C__PROPERTY__UI__ID     => 'C__CATG__CMK_HOST_NAME',
                    C__PROPERTY__UI__PARAMS => [
                        'p_bInfoIconSpacer' => 0,
                        'disableInputGroup' => true
                    ]
                ],
                C__PROPERTY__PROVIDES => [
                    C__PROPERTY__PROVIDES__IMPORT    => true,
                    C__PROPERTY__PROVIDES__EXPORT    => true,
                    C__PROPERTY__PROVIDES__REPORT    => false,
                    C__PROPERTY__PROVIDES__LIST      => false,
                    C__PROPERTY__PROVIDES__MULTIEDIT => false
                ]
            ]),
            'host_name_selection' => array_replace_recursive(isys_cmdb_dao_category_pattern::int(), [
                C__PROPERTY__INFO     => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CATG__CMK__HOSTNAME',
                    C__PROPERTY__INFO__DESCRIPTION => 'Hostname selection'
                ],
                C__PROPERTY__DATA     => [
                    C__PROPERTY__DATA__FIELD => 'isys_catg_cmk_list__host_name_selection'
                ],
                C__PROPERTY__UI       => [
                    C__PROPERTY__UI__ID => 'C__CATG__CMK__HOSTNAME'
                ],
                C__PROPERTY__PROVIDES => [
                    C__PROPERTY__PROVIDES__IMPORT    => true,
                    C__PROPERTY__PROVIDES__EXPORT    => true,
                    C__PROPERTY__PROVIDES__REPORT    => false,
                    C__PROPERTY__PROVIDES__LIST      => false,
                    C__PROPERTY__PROVIDES__MULTIEDIT => false
                ]
            ]),
            'folder'              => array_replace_recursive(isys_cmdb_dao_category_pattern::dialog_plus(), [
                C__PROPERTY__INFO => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CATG__CMK__WATO_FOLDER',
                    C__PROPERTY__INFO__DESCRIPTION => 'WATO folder'
                ],
                C__PROPERTY__DATA => [
                    C__PROPERTY__DATA__FIELD        => 'isys_catg_cmk_list__isys_wato_folder__id',
                    C__PROPERTY__DATA__SOURCE_TABLE => 'isys_wato_folder',
                    C__PROPERTY__DATA__REFERENCES   => [
                        'isys_wato_folder',
                        'isys_wato_folder__id'
                    ],
                    C__PROPERTY__DATA__SELECT       => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory('isys_wato_folder__title', 'isys_wato_folder'),
                    C__PROPERTY__DATA__JOIN         => [
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_cmk_list', 'LEFT', 'isys_catg_cmk_list__isys_obj__id', 'isys_obj__id'),
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_wato_folder', 'LEFT', 'isys_catg_cmk_list__isys_wato_folder__id',
                            'isys_wato_folder__id')
                    ]
                ],
                C__PROPERTY__UI   => [
                    C__PROPERTY__UI__ID     => 'C__CATG__CMK__FOLDER',
                    C__PROPERTY__UI__PARAMS => [
                        'p_strTable' => 'isys_wato_folder'
                    ]
                ]
            ]),
            'agent'               => array_replace_recursive(isys_cmdb_dao_category_pattern::dialog_plus(), [
                C__PROPERTY__INFO => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CATG__CMK__AGENT',
                    C__PROPERTY__INFO__DESCRIPTION => 'Agent'
                ],
                C__PROPERTY__DATA => [
                    C__PROPERTY__DATA__FIELD        => 'isys_catg_cmk_list__isys_agent__id',
                    C__PROPERTY__DATA__SOURCE_TABLE => 'isys_agent',
                    C__PROPERTY__DATA__REFERENCES   => [
                        'isys_agent',
                        'isys_agent__id'
                    ],
                    C__PROPERTY__DATA__SELECT       => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory('isys_agent__title', 'isys_agent_folder'),
                    C__PROPERTY__DATA__JOIN         => [
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_cmk_list', 'LEFT', 'isys_catg_cmk_list__isys_obj__id', 'isys_obj__id'),
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_agent', 'LEFT', 'isys_catg_cmk_list__isys_agent__id', 'isys_agent__id')
                    ]
                ],
                C__PROPERTY__UI   => [
                    C__PROPERTY__UI__ID     => 'C__CATG__CMK__AGENT',
                    C__PROPERTY__UI__PARAMS => [
                        'p_strTable' => 'isys_agent'
                    ]
                ]
            ]),
            'site'                => array_replace_recursive(isys_cmdb_dao_category_pattern::dialog_plus(), [
                C__PROPERTY__INFO => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CATG__CMK__SITE',
                    C__PROPERTY__INFO__DESCRIPTION => 'Site'
                ],
                C__PROPERTY__DATA => [
                    C__PROPERTY__DATA__FIELD        => 'isys_catg_cmk_list__isys_site__id',
                    C__PROPERTY__DATA__SOURCE_TABLE => 'isys_site',
                    C__PROPERTY__DATA__REFERENCES   => [
                        'isys_site',
                        'isys_site__id'
                    ],
                    C__PROPERTY__DATA__SELECT       => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory('isys_site__title', 'isys_site'),
                    C__PROPERTY__DATA__JOIN         => [
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_cmk_list', 'LEFT', 'isys_catg_cmk_list__isys_obj__id', 'isys_obj__id'),
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_site', 'LEFT', 'isys_catg_cmk_list__isys_site__id', 'isys_site__id')
                    ]
                ],
                C__PROPERTY__UI   => [
                    C__PROPERTY__UI__ID     => 'C__CATG__CMK__SITE',
                    C__PROPERTY__UI__PARAMS => [
                        'p_strTable' => 'isys_site'
                    ]
                ]
            ]),
            'parents'             => array_replace_recursive(isys_cmdb_dao_category_pattern::object_browser(), [
                C__PROPERTY__INFO     => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CATG__CMK__PARENTS',
                    C__PROPERTY__INFO__DESCRIPTION => 'Parents'
                ],
                C__PROPERTY__DATA     => [
                    C__PROPERTY__DATA__FIELD      => 'isys_catg_cmk_list__id',
                    C__PROPERTY__DATA__REFERENCES => [
                        'isys_catg_cmk_list_2_isys_obj',
                        'isys_catg_cmk_list__id'
                    ],
                    C__PROPERTY__DATA__SELECT     => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory('SELECT CONCAT(obj.isys_obj__title, \' {\', obj.isys_obj__id, \'}\')
                            FROM isys_catg_cmk_list cmk
                            INNER JOIN isys_catg_cmk_list_2_isys_obj AS cmk2obj ON cmk2obj.isys_catg_cmk_list__id = cmk.isys_catg_cmk_list__id
                            INNER JOIN isys_obj obj ON obj.isys_obj__id = cmk2obj.isys_obj__id', 'isys_catg_cmk_list', 'cmk.isys_catg_cmk_list__id',
                        'cmk.isys_catg_cmk_list__isys_obj__id', '', '', idoit\Module\Report\SqlQuery\Structure\SelectCondition::factory([]),
                        idoit\Module\Report\SqlQuery\Structure\SelectGroupBy::factory(['cmk.isys_catg_cmk_list__isys_obj__id'])),
                    C__PROPERTY__DATA__JOIN       => [
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_cmk_list', 'LEFT', 'isys_catg_cmk_list__isys_obj__id', 'isys_obj__id', '', 'cmk',
                            'cmk'),
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_cmk_list_2_isys_obj', 'LEFT', 'isys_catg_cmk_list__id', 'isys_catg_cmk_list__id',
                            'cmk', 'cmk2obj', 'cmk2obj'),
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_obj', 'LEFT', 'isys_obj__id', 'isys_obj__id', 'cmk2obj', 'obj', 'obj')
                    ]
                ],
                C__PROPERTY__UI       => [
                    C__PROPERTY__UI__ID     => 'C__CATG__CMK__PARENTS',
                    C__PROPERTY__UI__PARAMS => [
                        isys_popup_browser_object_ng::C__MULTISELECTION => true,
                        'p_strSelectedID'                               => new isys_callback([
                            'isys_cmdb_dao_category_g_cmk',
                            'callbackExportParents'
                        ])
                    ]
                ],
                C__PROPERTY__PROVIDES => [
                    C__PROPERTY__PROVIDES__SEARCH => false,
                    C__PROPERTY__PROVIDES__REPORT => false
                ],
                C__PROPERTY__FORMAT   => [
                    C__PROPERTY__FORMAT__CALLBACK => [
                        'isys_export_helper_check_mk',
                        'chmkExportParents'
                    ]
                ]
            ]),
            'description'         => array_replace_recursive(isys_cmdb_dao_category_pattern::commentary(), [
                C__PROPERTY__INFO => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CMDB__LOGBOOK__DESCRIPTION',
                    C__PROPERTY__INFO__DESCRIPTION => 'Description'
                ],
                C__PROPERTY__DATA => [
                    C__PROPERTY__DATA__FIELD => 'isys_catg_cmk_list__description'
                ],
                C__PROPERTY__UI   => [
                    C__PROPERTY__UI__ID => 'C__CMDB__CAT__COMMENTARY_' . C__CMDB__CATEGORY__TYPE_GLOBAL . C__CATG__CMK
                ]
            ])
        ];
    }

    /**
     * Creates the reference between check mk export parameter to the specified parents
     *
     * @param integer $dataId
     * @param array   $parents
     *
     * @return bool
     * @throws Exception
     * @throws isys_exception_dao
     * @author  Van Quyen Hoang <qhoang@i-doit.org>
     */
    public function attachParents($dataId, $parents)
    {
        if (is_string($parents)) {
            $parents = isys_format_json::decode($parents);
        }

        if (is_array($parents) && count($parents)) {
            $insert = sprintf('INSERT INTO isys_catg_cmk_list_2_isys_obj (isys_catg_cmk_list__id, isys_obj__id) VALUES %s;',
                implode(',', array_map(function ($item) use ($dataId) {
                    return '(' . $dataId . ', ' . $item . ')';
                }, $parents)));

            return $this->update($insert) && $this->apply_update();
        }

        return true;
    }

    /**
     * Removes all connected parents
     *
     * @param integer $dataId
     *
     * @return bool
     * @throws isys_exception_dao
     * @author  Van Quyen Hoang <qhoang@i-doit.org>
     */
    public function detachParents($dataId)
    {
        return ($this->update('DELETE FROM isys_catg_cmk_list_2_isys_obj WHERE isys_catg_cmk_list__id = ' . $this->convert_sql_id($dataId)) && $this->apply_update());
    }
}
