<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Validation\Logical;

use idoit\Module\Api\Exception\ValidationException;
use idoit\Module\Api\Validation\Validation;

/**
 * Class DialogValue
 *
 * @package Idoit\Module\Api\Validation\Logical
 */
class DialogValue extends Validation
{
    /**
     * Data included in dialog field
     *
     * @var array
     */
    protected $dialogData;

    /**
     * @param array $value
     *
     * @return DialogValue
     */
    public function setValue($value)
    {

        $this->value = is_array($value) ? $value : [$value];

        return $this;
    }

    /**
     * Validate value
     *
     * @return bool
     * @throws \idoit\Module\Api\Exception\ValidationException
     */
    public function validate()
    {
        // Ensure one consistent data type for processing
        $integerValueCount = 0;
        $stringValueCount = 0;

        $propertyType = $this->getPropertyType();
        $dialogValues = $this->getValue();
        $dialogData = $this->getDialogData();

        foreach ($dialogValues as $dialogValue) {
            if (is_string($dialogValue)) {
                $stringValueCount++;
            } else if (is_int($dialogValue)) {
                $integerValueCount++;
            }
        }

        // Check whether provided values are mixed
        if ($integerValueCount !== count($dialogValues) && $stringValueCount !== count($dialogValues)) {
            throw new ValidationException($this->getPropertyType(), 'Mixed value types detected. Please provide only string or integer values.');
        }

        /**
         * Validate provided values
         */

        // Integer value handling
        if ($integerValueCount === count($dialogValues)) {
            /**
             * This branch is relevant for all dialog, dialog+, dialogList and multiSelect
             */

            // Get difference between provided and available data
            $unknownValues = array_diff($dialogValues, array_keys($dialogData));

            // Check whether all provided Ids exists in dialog table
            if (count($unknownValues) !== 0) {
                throw new ValidationException($this->getPropertyType(),
                    'Id value(s) \'' . implode(',', $unknownValues) . '\' not exists in dialog table. Please check your provided selection.');

            }
        } else {
            // String value handling for dialog, dialogList and multiSelect
            if ($propertyType == C__PROPERTY__INFO__TYPE__DIALOG) {
                // Get difference between provided and available data
                $unknownValues = array_diff($dialogValues, $dialogData);

                // Check whether given values exists all in dialog table
                if (count($unknownValues) !== 0) {
                    throw new ValidationException($propertyType,
                        'Unable to resolve given value. Provided value(s) \'' . implode(',', $unknownValues) . '\' not exists in dialog table.');
                }
            }
        }

        return true;
    }

    /**
     * @return array
     */
    public function getDialogData()
    {
        return $this->dialogData;
    }

    /**
     * @param array $dialogData
     *
     * @return DialogValue
     */
    public function setDialogData(array $dialogData)
    {
        $this->dialogData = $dialogData;

        return $this;
    }
}