<?php

/**
 * i-doit
 *
 * API model
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@synetics.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_cmdb_object_type_groups extends isys_api_model_cmdb implements isys_api_model_interface
{

    /**
     * Data formatting used in format methods
     *
     * @var array
     */
    protected $m_mapping = [
        'isys_obj_type_group__id'          => 'id',
        'isys_obj_type_group__title'       => ['_L', 'title'],
        'isys_obj_type_group__description' => 'description',
        'isys_obj_type_group__const'       => 'const',
        'isys_obj_type_group__status'      => 'status'
    ];

    /**
     * Possible options and their parameters
     *
     * @var array
     */
    protected $m_options = [
        'read' => [
            'filter'   => [
                'type'        => 'array',
                'description' => 'Filter array',
                'optional'    => true
            ],
            'limit'    => [
                'type'        => 'int',
                'description' => 'Resultset limiting',
                'optional'    => true
            ],
            'sort'     => [
                'type'        => 'string',
                'description' => 'ASC or DESC',
                'optional'    => true
            ],
            'order_by' => [
                'type'        => 'string',
                'description' => 'Ordering by title, id or status',
                'optional'    => true
            ]
        ]
    ];

    /**
     * Validation
     *
     * @var array
     */
    protected $m_validation = [];

    /**
     * Fetches object type groups by filter.
     * [
     *    integer  $p_params['filter']['id']      (optional) Object type identifier or Constants
     *    array    $p_params['filter']['ids']     (optional) Object type identifiers or Constants
     *    string   $p_params['filter']['title']   (optional) Object type title
     *    array    $p_params['filter']['titles']  (optional) Object type titles
     *    boolean  $p_params['raw']               (optional) Formatting.
     *    string   $p_params['order_by']          (optional) Order by one of the supported filter arguments. Defaults to null that means result will be ordered by object identifiers.
     *    string   $p_params['sort']              (optional) Order result ascending ('ASC') or descending ('DESC').
     *    integer  $p_params['limit']             (optional) Limitation: where to start and number of elements, i.e. 0 or 0,10. Defaults to null that means no limitation.
     * ]
     *
     * @param array $p_params Parameters:
     *
     * @return array Objects types. Returns an empty array on error.
     * @author Selcuk Kekec <skekec@synetics.de>
     */
    public function read($p_params)
    {
        if (!array_key_exists('filter', $p_params) || !is_array($p_params['filter']))
        {
            $p_params['filter'] = [];
        }

        // Force limit to record status 'normal':
        $p_params['filter']['status'] = C__RECORD_STATUS__NORMAL;

        // Raw mode:
        $l_raw      = (bool) $p_params['raw'];
        $l_order_by = null;
        $l_sort     = null;
        $l_limit    = null;

        // Order by:
        if (isset($p_params['order_by']) && !empty($p_params['order_by']))
        {
            $l_order_by = $p_params['order_by'];
        }

        // Sort:
        if (isset($p_params['sort']) && !empty($p_params['sort']))
        {
            $l_sort = $p_params['sort'];
        }

        // Limitation:
        if (isset($p_params['limit']) && !empty($p_params['limit']))
        {
            $l_limit = $p_params['limit'];
        } // if

        // Data retrieval:
        $l_data = $this->m_dao->get_object_type_groups_by_properties(
            $p_params['filter'],
            $l_order_by,
            $l_sort,
            $l_limit
        );

        $l_return = [];

        // Data formatting:
        while ($l_row = $l_data->get_row())
        {
            $l_return[] = ($l_raw ? $l_row : $this->format_by_mapping($this->m_mapping, $l_row));
        }

        // Order by translated titles:
        if ($l_order_by === 'title' && $l_raw === false)
        {
            usort($l_return, [$this, 'sort_by_title']);
        }

        return $l_return;
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @return isys_api_model_cmdb Returns itself.
     * @throws isys_exception_api
     * @author Selcuk Kekec <skekec@synetics.de>
     */
    public function create($p_params)
    {
        throw new isys_exception_api('Creating is not possible here.');
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @return isys_api_model_cmdb Returns itself.
     * @throws isys_exception_api
     * @author Selcuk Kekec <skekec@synetics.de>
     */
    public function delete($p_params)
    {
        throw new isys_exception_api('Deleting is not possible here.');
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @return isys_api_model_cmdb Returns itself.
     * @throws isys_exception_api
     * @author Selcuk Kekec <skekec@synetics.de>
     */
    public function update($p_params)
    {
        throw new isys_exception_api('Updating is not possible here.');
    }

    /**
     * Constructor
     */
    public function __construct(isys_cmdb_dao &$p_dao)
    {
        $this->m_dao = $p_dao;
        parent::__construct();
    }
}