<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Controller;

use idoit\Module\Api\Logging\ApiLogger;
use idoit\Module\Api\Logging\LineFormatter;
use idoit\Module\Api\Request\RequestStore;
use Monolog\Handler\NullHandler;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;

/**
 * Class AbstractController
 *
 * @package idoit\Module\Api\Controller
 */
abstract class AbstractController
{
    /**
     * Enable / Disable authentication system
     *
     * @var bool
     */
    protected $authenticationSystem = false;

    /**
     * Logger instance
     *
     * @var Logger
     */
    protected $logger;

    /**
     * RequestStore instance
     *
     * @var RequestStore
     */
    protected $requestStore;

    /**
     * Run request store
     *
     * @return mixed
     */
    public abstract function run();

    /**
     * Get logger instance
     *
     * @return Logger
     */
    public function getLogger()
    {
        return $this->logger;
    }

    /**
     * Set logger instance
     *
     * @param Logger $logger
     *
     * @return AbstractController
     */
    public function setLogger($logger)
    {
        $this->logger = $logger;

        return $this;
    }

    /**
     * Enable authentication system
     *
     * @return AbstractController
     */
    public function enableAuthenticationSystem()
    {
        return $this->setAuthenticationSystem(true);
    }

    /**
     * Disable authentication system
     *
     * @return AbstractController
     */
    public function disableAuthenticationSystem()
    {
        return $this->setAuthenticationSystem(false);
    }

    /**
     * @return bool
     */
    public function getAuthenticationSystem()
    {
        return $this->authenticationSystem;
    }

    /**
     * @param bool $authenticationSystem
     *
     * @return AbstractController
     */
    private function setAuthenticationSystem($authenticationSystem)
    {
        global $g_config;

        // Enable authenticat
        $g_config['use_auth'] = $authenticationSystem ? 1 : 0;

        $this->authenticationSystem = $authenticationSystem;

        return $this;
    }

    /**
     * Get request store
     *
     * @return RequestStore
     */
    public function getRequestStore()
    {
        return $this->requestStore;
    }

    /**
     * Set RequestStore
     *
     * @param RequestStore $requestStore
     *
     * @return AbstractController
     */
    public function setRequestStore($requestStore)
    {
        $this->requestStore = $requestStore;

        return $this;
    }

    /**
     * Create logger instance
     *
     * @return Logger
     * @throws \Exception
     */
    private function createLoggerInstance()
    {
        // Create and configure api logger instance
        $logger = ApiLogger::instance('api.logger');

        // Get log level
        $logLevel = \isys_tenantsettings::get('api.log-level', Logger::WARNING);

        if ($logLevel == -1) {
            // Create null handler and register it as handler
            $logger->pushHandler(new NullHandler());
        } else {
            // Create and setup line formatter
            $formatter = new LineFormatter("[%datetime%] %level_name%: %message% %context%\n", 'Y-m-d H:i:s u');
            $formatter->ignoreEmptyContextAndExtra(true);
            $formatter->allowInlineLineBreaks(true);

            // Create and setup stream handler
            $streamHandler = new StreamHandler(\isys_application::instance()->app_path . '/log/api_' . date('Y-m-d') . '.log', $logLevel);
            $streamHandler->setFormatter($formatter);

            // Add file stream handler
            $logger->pushHandler($streamHandler);
        }

        return $logger;
    }

    /**
     * Get api version
     *
     * @return string
     */
    public function getApiVersion()
    {
        return (file_exists(__DIR__ . '/../../VERSION')) ? file_get_contents(__DIR__ . '/../../VERSION') : '0.0';
    }

    /**
     * AbstractController constructor.
     *
     * @throws \Exception
     */
    public function __construct()
    {
        // Create and setup logger instance
        $loggerInstance = $this->createLoggerInstance();

        // Set logger in controller context
        $this->setLogger($loggerInstance);
    }
}
