<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Dialog;

use isys_request;

/**
 * Class AdminBuilder
 *
 * @package idoit\Module\Api\Dialog
 */
class AdminBuilder
{
    /**
     * @var array|\isys_callback
     */
    protected $array;

    /**
     * @var string
     */
    protected $implementationClass;

    /**
     * @var array
     */
    protected $propertyDefinition;

    /**
     * @var isys_request
     */
    protected $request;

    /**
     * @var string
     */
    protected $table;

    /**
     * @return array|\isys_callback
     */
    public function getArray()
    {
        return $this->array;
    }

    /**
     * @param array|\isys_callback $array
     *
     * @return AdminBuilder
     */
    public function setArray($array)
    {
        $this->array = $array;

        return $this;
    }

    /**
     * @return string
     */
    public function getTable()
    {
        return $this->table;
    }

    /**
     * @param string $table
     *
     * @return AdminBuilder
     */
    public function setTable($table)
    {
        $this->table = $table;

        return $this;
    }

    /**
     * @return isys_request
     */
    public function getRequest()
    {
        return $this->request;
    }

    /**
     * @param isys_request $request
     *
     * @return AdminBuilder
     */
    public function setRequest($request)
    {
        $this->request = $request;

        return $this;
    }

    /**
     * @return array
     */
    public function getPropertyDefinition()
    {
        return $this->propertyDefinition;
    }

    /**
     * @param array $propertyDefinition
     *
     * @return AdminBuilder
     */
    public function setPropertyDefinition($propertyDefinition)
    {
        $this->propertyDefinition = $propertyDefinition;

        return $this;
    }

    /**
     * Build admin instance
     *
     * @return DialogAdmin
     * @throws \Exception
     */
    public function build()
    {
        /**
         * Build Implementation based on provided information
         */
        if ($this->getArray() instanceof \isys_callback) {
            if (!$this->getRequest()) {
                throw new \Exception('Unable to build Administration object without request.');
            }

            return call_user_func_array([$this->implementationClass, 'byCallback'], [$this->getArray(), $this->getRequest(), $this->getPropertyDefinition()]);
        } else if (is_array($this->getArray())) {
            return call_user_func_array([$this->implementationClass, 'byArray'], [$this->getArray(), $this->getPropertyDefinition()]);
        } else if (is_string($this->getTable())) {
            return call_user_func_array([$this->implementationClass, 'byTableName'], [$this->getTable(), $this->getPropertyDefinition()]);
        }

        throw new \Exception('Unable to instantiate Dialog administrator with given parameters.');
    }

    /**
     * AdminBuilder constructor.
     *
     * @param string            $propertyType
     * @param array|null        $propertyDefinition
     * @param isys_request|null $request
     *
     * @throws \Exception
     */
    public function __construct($propertyType, array $propertyDefinition = null, \isys_request $request = null)
    {
        // Determine needed implementation
        switch ($propertyType) {
            case C__PROPERTY__INFO__TYPE__DIALOG:
                $this->implementationClass = DialogAdmin::class;
                break;
            case C__PROPERTY__INFO__TYPE__DIALOG_LIST:
            case C__PROPERTY__INFO__TYPE__MULTISELECT:
            case C__PROPERTY__INFO__TYPE__DIALOG_PLUS:
                $this->implementationClass = DialogPlusAdmin::class;
                break;
            default:
                throw new \Exception('Unable to build Administration for given property type.');
        }

        if (!empty($propertyDefinition)) {
            $this->setPropertyDefinition($propertyDefinition)
                ->setArray($propertyDefinition[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_arData'])
                ->setTable($propertyDefinition[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_strTable']);
        }

        $this->setRequest($request);
    }
}