<?php

use idoit\Exception\InvalidCacheDataException;

/**
 * i-doit core classes
 *
 * @package     i-doit
 * @subpackage  Core
 * @author      Dennis Stücken <dstuecken@synetics.de>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_settings implements isys_settings_interface
{
    use isys_settings_trait;

    /**
     * Database component.
     *
     * @var  isys_component_dao_settings
     */
    protected static $m_dao;

    /**
     * Settings register.
     *
     * @var  array
     */
    protected static $m_definition = [
        'User interface'    => [
            'gui.wiki-url'             => [
                'title'       => 'Wiki URL',
                'type'        => 'text',
                'placeholder' => 'https://wikipedia.org/wiki/'
            ],
            'login.tenantlist.sortby'  => [
                'title'   => 'LC__SYSTEM_SETTINGS__TENANT_SORT_FUNCTION',
                'type'    => 'select',
                'options' => [
                    'isys_mandator__title' => 'LC__UNIVERSAL__TITLE',
                    'isys_mandator__sort'  => 'LC__SYSTEM_SETTINGS__TENANT_SORT_FUNCTION__CUSTOM'
                ]
            ]
        ],
        'Session'           => [
            'session.time' => [
                'title'       => 'Session timeout',
                'type'        => 'int',
                'placeholder' => 300,
                'default'     => 300,
                'description' => 'LC__CMDB__UNIT_OF_TIME__SECOND'
            ]
        ],
        'Single Sign On'    => [
            'session.sso.active'      => [
                'title'   => 'LC__UNIVERSAL__ACTIVE',
                'type'    => 'select',
                'options' => [
                    '0' => 'LC__UNIVERSAL__NO',
                    '1' => 'LC__UNIVERSAL__YES'
                ]
            ],
            'session.sso.mandator-id' => [
                'title' => 'LC__SYSTEM_SETTINGS__DEFAULT_MANDATOR',
                'type'  => 'select'
            ],
            'session.sso.use-domain-part' => [
                'title'   => 'LC__UNIVERSAL__USE_DOMAIN_PART',
                'type'    => 'select',
                'options' => [
                    '0' => 'LC__UNIVERSAL__NO',
                    '1' => 'LC__UNIVERSAL__YES'
                ]
            ],
        ],
        'System Parameters' => [
            'reports.browser-url'            => [
                'title'       => 'Report-Browser URL',
                'type'        => 'text',
                'hidden'      => true,
                'placeholder' => 'https://reports-ng.i-doit.org/'
            ],
            'cmdb.connector.suffix-schema'   => [
                'title'  => '',
                'type'   => 'select',
                'hidden' => true
            ],
            'system.timezone'                => [
                'title'       => 'LC__SYSTEM_SETTINGS__PHP_TIMEZONE',
                'type'        => 'text',
                'placeholder' => 'Europe/Berlin',
                'description' => '<a href="https://php.net/manual/timezones.php">https://php.net/manual/timezones.php</a>'
            ],
            'system.devmode'                 => [
                'title'   => 'Developer mode',
                'type'    => 'select',
                'options' => [
                    '0' => 'LC__UNIVERSAL__NO',
                    '1' => 'LC__UNIVERSAL__YES'
                ],
                'hidden'  => true
            ],
            'system.dir.file-upload'         => [
                'title'       => 'LC__SYSTEM_SETTINGS__FILE_UPLOAD_DIRECTORY',
                'placeholder' => '/path/to/i-doit/upload/files/',
                'type'        => 'text'
            ],
            'system.dir.image-upload'        => [
                'title'       => 'LC__SYSTEM_SETTINGS__IMAGE_UPLOAD_DIRECTORY',
                'placeholder' => '/path/to/i-doit/upload/images/',
                'type'        => 'text'
            ]
        ],
        'Proxy'             => [
            'proxy.active'   => [
                'title'   => 'LC__UNIVERSAL__ACTIVE',
                'type'    => 'select',
                'options' => [
                    '0' => 'LC__UNIVERSAL__NO',
                    '1' => 'LC__UNIVERSAL__YES'
                ]
            ],
            'proxy.host'     => [
                'title'       => 'LC__SYSTEM_SETTINGS__HOST_IP_ADDRESS',
                'type'        => 'text',
                'placeholder' => 'proxy.i-doit.com'
            ],
            'proxy.port'     => [
                'title'       => 'Port',
                'type'        => 'int',
                'placeholder' => 3128
            ],
            'proxy.username' => [
                'title' => 'LC__LOGIN__USERNAME',
                'type'  => 'text'
            ],
            'proxy.password' => [
                'title' => 'LC__LOGIN__PASSWORD',
                'type'  => 'password'
            ]
        ],
        'Security'          => [
            'system.security.csrf' => [
                'title'       => 'CSRF-Token',
                'description' => 'LC__SYSTEM_SETTINGS__SECURITY__CSRF_IN_LOGIN',
                'type'        => 'select',
                'options'     => [
                    '1' => 'LC__UNIVERSAL__YES',
                    '0' => 'LC__UNIVERSAL__NO'
                ],
                'default'     => '0'
            ]
        ],
        'Login'             => [
            'system.login.welcome-message' => [
                'title'       => 'LC__SYSTEM_SETTINGS__LOGIN_WELCOME_MESSAGE',
                'type'        => 'textarea',
                'placeholder' => 'LC__SYSTEM_SETTINGS__LOGIN_WELCOME_MESSAGE_DEFAULT'
            ]
        ]
    ];

    /**
     * Load cache.
     *
     * @static
     *
     * @param   isys_component_database $p_database
     *
     * @return  void
     */
    public static function initialize(isys_component_database $p_database)
    {
        isys_component_signalcollection::get_instance()
            ->connect('system.shutdown', [
                'isys_settings',
                'shutdown'
            ]);

        self::$m_dao = new isys_component_dao_settings($p_database);
        $l_cache_dir = self::get_cache_dir();

        // Generate cache and load settings.
        if ($l_cache_dir) {
            try {
                try {
                    if (!file_exists($l_cache_dir . self::$m_cachefile)) {
                        throw new InvalidCacheDataException();
                    }

                    self::load_cache($l_cache_dir);
                } catch (InvalidCacheDataException $e) {
                    self::regenerate();
                }
            } catch (Exception $e) {
                if (isys_application::instance()->container->has('logger')) {
                    isys_application::instance()->container->get('logger')->error('System settings cache error: ' . $e->getMessage());
                }

                // Load settings from database instead of cache.
                if (!is_array(self::$m_settings) || count(self::$m_settings) === 0) {
                    self::$m_settings = self::$m_dao->get_settings();
                }
            }
        }

        self::$m_initialized = true;
    }

    /**
     * (Re)generates cache. Loads the cache into self::$m_settings.
     *
     * @throws  Exception
     * @return  array
     */
    public static function regenerate()
    {
        try {
            self::$m_settings = self::$m_dao->get_settings();

            // Write settings cache.
            self::write(self::get_cache_dir() . self::$m_cachefile, self::$m_settings);
        } catch (Exception $e) {
            throw $e;
        }

        return self::$m_settings;
    }
}
