<?php

namespace idoit\Module\JDisc\Console\Command;

use Exception;
use idoit\Console\Command\AbstractCommand;
use isys_application;
use isys_helper_crypt;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputDefinition;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class JDiscCreateServerCommand extends AbstractCommand
{
    const NAME = 'jdisc-create-server';

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     * @throws Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $dao = isys_application::instance()->container->get('cmdb_dao');

        if (trim($input->getOption('jdisc-password')) === '') {
            throw new Exception('Please provide a password for your configuration by passing --jdisc-password');
        }

        $defaultServer = $dao->convert_sql_boolean(false);
        $title = $dao->convert_sql_text($input->getOption('title'));
        $jdiscHost = $dao->convert_sql_text($input->getOption('jdisc-host'));
        $jdiscPort = $dao->convert_sql_int($input->getOption('jdisc-port'));
        $jdiscDatabase = $dao->convert_sql_text($input->getOption('jdisc-database'));
        $jdiscUsername = $dao->convert_sql_text($input->getOption('jdisc-username'));
        $jdiscPassword = $dao->convert_sql_text(isys_helper_crypt::encrypt($input->getOption('jdisc-password')));
        $allowOlderImports = $dao->convert_sql_boolean($input->getOption('allow-older-imports'));

        if ($input->getOption('default')) {
            $output->writeln('The <info>default</info> option was passed! Setting other configurations to "non-default".');
            $sql = "UPDATE isys_jdisc_db SET isys_jdisc_db__default_server = 0;";

            $dao->update($sql);
            $dao->apply_update();

            $defaultServer = $dao->convert_sql_boolean(true);
        }

        $output->writeln("Creating new JDisc configuration with title <info>$title</info>...");
        $sql = "INSERT INTO isys_jdisc_db SET
            isys_jdisc_db__title = {$title},
            isys_jdisc_db__host = {$jdiscHost},
            isys_jdisc_db__port = {$jdiscPort},
            isys_jdisc_db__database = {$jdiscDatabase},
            isys_jdisc_db__username = {$jdiscUsername},
            isys_jdisc_db__password = {$jdiscPassword},
            isys_jdisc_db__version_check = {$allowOlderImports},
            isys_jdisc_db__default_server = {$defaultServer};";

        $dao->update($sql);
        $dao->apply_update();

        $id = (int)$dao->get_last_id_from_table('isys_jdisc_db');

        $output->writeln("Success! The newly created configuration was saved with ID <info>{$id}</info>");

        return Command::SUCCESS;
    }

    /**
     * @return InputDefinition
     */
    public function getCommandDefinition()
    {
        return new InputDefinition([
            new InputOption('default', null, InputOption::VALUE_NONE, 'Set as default server'),
            new InputOption('title', null, InputOption::VALUE_REQUIRED, 'Name for JDisc server configuration', 'JDisc Server'),
            new InputOption('jdisc-host', null, InputOption::VALUE_REQUIRED, 'JDisc host', 'localhost'),
            new InputOption('jdisc-port', null, InputOption::VALUE_REQUIRED, 'JDisc port', 25321),
            new InputOption('jdisc-database', null, InputOption::VALUE_REQUIRED, 'JDisc database', 'inventory'),
            new InputOption('jdisc-username', null, InputOption::VALUE_REQUIRED, 'JDisc username', 'postgresro'),
            new InputOption('jdisc-password', null, InputOption::VALUE_REQUIRED, 'JDisc password'),
            new InputOption('allow-older-imports', null, InputOption::VALUE_NONE, 'Allow import of older JDisc version <comment>[default: no]</comment>'),
        ]);
    }

    /**
     * @return string
     */
    public function getCommandDescription()
    {
        return 'Creates a JDisc server, based on given input';
    }

    /**
     * @return string
     */
    public function getCommandName()
    {
        return self::NAME;
    }

    /**
     * @return array|string[]
     */
    public function getCommandUsages()
    {
        return [
            'jdisc-create-server --jdisc-password <password>',
            'jdisc-create-server --jdisc-password <password> --default',
            'jdisc-create-server --jdisc-user <user> --jdisc-password <password> --allow-older-imports',
        ];
    }

    /**
     * @return false
     */
    public function isConfigurable()
    {
        return false;
    }
}
