<?php

namespace idoit\Module\Pro\Controller\AttributeSettings;

use idoit\Component\Property\Property;
use idoit\Module\Pro\Model\AttributeSettings;
use isys_application;
use isys_cmdb_dao_category_property;
use isys_component_database;
use isys_component_template_language_manager;
use isys_format_json;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

/**
 * Controller for 'attribute settings' configuration
 *
 * @package   Modules
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Configuration
{
    private isys_component_database $database;

    private isys_component_template_language_manager $language;

    private AttributeSettings $attributeSettingModel;

    /**
     * Array of blocked categories, mainly due to them being 'assignment' categories.
     */
    private const BLOCKED_CATEGORIES = [
        'C__CATG__CONTACT',
        'C__CATG__GUEST_SYSTEMS',
        'C__CATG__IMAGE',
        'C__CATG__RM_CONTROLLER_BACKWARD',
        'C__CATG__CLUSTER_MEMBERS', // @see ID-10486
        'C__CATG__CLUSTER_MEMBERSHIPS', // @see ID-10486
        // Specific categories.
        'C__CATS__APPLICATION_ASSIGNED_OBJ',
        'C__CATS__APPLICATION_DBMS_ASSIGNED_OBJ',
        'C__CATS__APPLICATION_SERVICE_ASSIGNED_OBJ',
        'C__CATS__CLUSTER_SERVICE_ASSIGNED_OBJ',
        'C__CATS__ENCLOSURE',
        'C__CATS__ORGANIZATION_PERSONS',
        'C__CATS__PERSON_ASSIGNED_GROUPS',
        'C__CATS__SAN_ZONING',
    ];

    /**
     * Array of blocked properties.
     */
    private const BLOCKED_PROPERTIES = [
        'C__CATG__CLUSTER::administration_service',           // @see ID-10486 Block unused properties (specific behaviour)
        'C__CATG__CLUSTER::cluster_members',                  // @see above
        'C__CATG__CLUSTER::cluster_member_count',             // @see above
        'C__CATG__CLUSTER::cluster_service',                  // @see above
        'C__CATG__CLUSTER_SERVICE::hostaddresses',            // @see ID-10486 Block reloaded properties (specific behaviour)
        'C__CATG__CLUSTER_SERVICE::drives',                   // @see above
        'C__CATG__CLUSTER_SERVICE::shares',                   // @see above
        'C__CATG__CLUSTER_SERVICE::assigned_database_schema', // @see above
        'C__CATG__OPERATING_SYSTEM::assigned_version_for_report',
        'C__CATG__SLA::monday_time',
        'C__CATG__SLA::tuesday_time',
        'C__CATG__SLA::wednesday_time',
        'C__CATG__SLA::thursday_time',
        'C__CATG__SLA::friday_time',
        'C__CATG__SLA::saturday_time',
        'C__CATG__SLA::sunday_time',
        // Specific categories.
        'C__CATS__NET_ZONE::range_from_long',
        'C__CATS__NET_ZONE::range_to_long',
    ];

    public function __construct()
    {
        $this->database = isys_application::instance()->container->get('database');
        $this->language = isys_application::instance()->container->get('language');
        $this->attributeSettingModel = AttributeSettings::instance($this->database);
    }

    /**
     * @param Request $request
     *
     * @return Response
     */
    public function loadConfiguration(Request $request): Response
    {
        return new JsonResponse([
            'success' => true,
            'data'    => [],
            'message' => ''
        ]);
    }

    /**
     * @param string   $propertyReference
     * @param Property $property
     *
     * @return bool
     */
    private function isSystemRelevant(string $propertyReference, Property $property): bool
    {
        $attributeOptions = $this->attributeSettingModel->getPropertyOptions($propertyReference, $property);

        return $attributeOptions['isUnhideable'] // If the property can not be hidden
            && !$attributeOptions['canBeValidated'] // And it can not be validated
            && ($attributeOptions['isReadonly'] || $attributeOptions['isVirtual']); // And the property is either virtual or 'read only'
    }

    /**
     * @param Request $request
     *
     * @return Response
     */
    public function createAttributeSetting(Request $request): Response
    {
        try {
            $properties = isys_format_json::decode($request->request->get('properties'));

            foreach ($properties as $propertyReference) {
                $this->attributeSettingModel->createSettingWithDefaults($propertyReference);
            }

            $response = [
                'success' => true,
                'data'    => null,
                'message' => ''
            ];
        } catch (Throwable $e) {
            $response = [
                'success' => false,
                'data'    => null,
                'message' => $e->getMessage()
            ];
        }

        return new JsonResponse($response);
    }

    /**
     * @param Request $request
     *
     * @return Response
     */
    public function deleteAttributeSetting(Request $request): Response
    {
        try {
            $response = [
                'success' => true,
                'data'    => $this->attributeSettingModel->delete(isys_format_json::decode($request->request->get('ids'))),
                'message' => ''
            ];
        } catch (Throwable $e) {
            $response = [
                'success' => false,
                'data'    => null,
                'message' => $e->getMessage()
            ];
        }

        return new JsonResponse($response);
    }

    /**
     * @param Request $request
     *
     * @return Response
     */
    public function loadAttributes(Request $request): Response
    {
        try {
            $categories = [];
            $configuredAttributes = [];
            $propertyDao = isys_cmdb_dao_category_property::instance($this->database);

            // Get all configured attributes:
            $result = $this->attributeSettingModel->getAll();

            while ($row = $result->get_row()) {
                $configuredAttributes[] = $row['propertyReference'];
            }

            $result = $propertyDao->retrieve_properties();

            while ($row = $result->get_row()) {
                $propertyReference = $row['const'] . '::' . $row['key'];

                if (in_array($row['const'], self::BLOCKED_CATEGORIES, true) || in_array($propertyReference, self::BLOCKED_PROPERTIES, true)) {
                    continue;
                }

                if (!class_exists($row['class']) || !is_a($row['class'], 'isys_cmdb_dao_category', true)) {
                    continue;
                }

                $categoryDao = $propertyDao->get_dao_instance($row['class'], ($row['catg_custom'] ?: null));

                $property = $categoryDao->get_property_by_key($row['key']);

                if ($property === null) {
                    continue;
                }

                if (is_array($property)) {
                    $property = Property::createInstanceFromArray($property);
                }

                if (!isset($categories[$row['const']])) {
                    $categories[$row['const']] = [
                        'constant'   => $row['const'],
                        'title'      => $this->language->get($categoryDao->getCategoryTitle()),
                        'properties' => []
                    ];
                }

                $categories[$row['const']]['properties'][] = [
                    'key'               => $row['key'],
                    'title'             => $this->language->get($row['title']),
                    'propertyReference' => $propertyReference,
                    'configured'        => in_array($propertyReference, $configuredAttributes, true),
                    'isSystemRelevant'  => $this->isSystemRelevant($propertyReference, $property)
                ];
            }

            uasort($categories, fn (array $a, array $b) => strcasecmp($a['title'], $b['title']));

            $response = [
                'success' => true,
                'data'    => array_values($categories),
                'message' => ''
            ];
        } catch (Throwable $e) {
            $response = [
                'success' => false,
                'data'    => null,
                'message' => $e->getMessage()
            ];
        }

        return new JsonResponse($response);
    }

    /**
     * @param Request $request
     * @param int     $id
     *
     * @return Response
     */
    public function editAttributeSetting(Request $request, int $id): Response
    {
        try {
            $response = [
                'success' => true,
                'data'    => $this->attributeSettingModel->edit($id, [
                    'visible' => 1 - $request->request->get('hidden'),
                    'visibleOverview' => 1 - $request->request->get('hiddenOverview'),
                    'mandatory' => $request->request->get('mandatory'),
                    'uniqueObject' => $request->request->get('uniqueObject'),
                    'uniqueObjectType' => $request->request->get('uniqueObjectType'),
                    'uniqueGlobal' => $request->request->get('uniqueGlobal'),
                    'validation' => $request->request->get('validation'),
                    'validationOption' => isys_format_json::decode($request->request->get('validationOption'))
                ]),
                'message' => ''
            ];
        } catch (Throwable $e) {
            $response = [
                'success' => false,
                'data'    => null,
                'message' => $e->getMessage()
            ];
        }

        return new JsonResponse($response);
    }
}
