<?php

use idoit\Component\Property\Type\DynamicProperty;

/**
 * i-doit
 *
 * DAO: global category for status plans.
 *
 * @package     i-doit
 * @subpackage  CMDB_Categories
 * @author      Dennis Stuecken <dstuecken@i-doit.de>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_cmdb_dao_category_g_planning extends isys_cmdb_dao_category_global
{
    /**
     * Category's name. Will be used for the identifier, constant, main table, and many more.
     *
     * @var  string
     */
    protected $m_category = 'planning';

    /**
     * Is category multi-valued or single-valued?
     *
     * @var  boolean
     */
    protected $m_multivalued = true;

    /**
     * Dynamic property handling for getting the "start" date.
     *
     * @param   array $p_row
     *
     * @return  string
     */
    public function dynamic_property_callback_start($p_row)
    {
        if (isset($p_row['isys_catg_planning_list__start']) && $p_row['isys_catg_planning_list__start'] > 0) {
            return isys_locale::get_instance()
                ->fmt_date($p_row['isys_catg_planning_list__start']);
        }

        return isys_tenantsettings::get('gui.empty_value', '-');
    }

    /**
     * Dynamic property handling for getting the "end" date.
     *
     * @param   array $p_row
     *
     * @return  string
     */
    public function dynamic_property_callback_end($p_row)
    {
        if (isset($p_row['isys_catg_planning_list__end']) && $p_row['isys_catg_planning_list__end'] > 0) {
            return isys_locale::get_instance()
                ->fmt_date($p_row['isys_catg_planning_list__end']);
        }

        return isys_tenantsettings::get('gui.empty_value', '-');
    }

    /**
     * Creates new entity.
     *
     * @param array $entryData
     *
     * @return mixed
     */
    public function create_data($entryData): mixed
    {
        $possibleStart = strtotime($entryData['start']);
        $possibleEnd = strtotime($entryData['end']);

        $entryData['start'] = min($possibleStart, $possibleEnd);
        $entryData['end'] = max($possibleStart, $possibleEnd);

        return parent::create_data($entryData);
    }

    /**
     * Method for retrieving the dynamic properties.
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     * @return  array
     */
    protected function dynamic_properties()
    {
        return [
            '_start' => new DynamicProperty(
                'LC__UNIVERSAL__VALIDITY_FROM',
                'isys_catg_planning_list__start',
                'isys_catg_planning_list',
                [
                    $this,
                    'dynamic_property_callback_start'
                ]
            ),
            '_end'  => new DynamicProperty(
                'LC__UNIVERSAL__VALIDITY_TO',
                'isys_catg_planning_list__end',
                'isys_catg_planning_list',
                [
                    $this,
                    'dynamic_property_callback_end'
                ]
            )
        ];
    }

    /**
     * Method for returning the properties.
     *
     * @return  array
     */
    protected function properties()
    {
        return [
            'cmdb_status' => array_replace_recursive(isys_cmdb_dao_category_pattern::dialog(), [
                C__PROPERTY__INFO     => [
                    C__PROPERTY__INFO__TITLE       => 'LC__UNIVERSAL__CMDB_STATUS',
                    C__PROPERTY__INFO__DESCRIPTION => 'CMDB status'
                ],
                C__PROPERTY__DATA     => [
                    C__PROPERTY__DATA__FIELD        => 'isys_catg_planning_list__isys_cmdb_status__id',
                    C__PROPERTY__DATA__SOURCE_TABLE => 'isys_cmdb_status',
                    C__PROPERTY__DATA__REFERENCES   => [
                        'isys_cmdb_status',
                        'isys_cmdb_status__id'
                    ],
                    C__PROPERTY__DATA__SELECT       => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory(
                        'SELECT isys_cmdb_status__title
                              FROM isys_catg_planning_list
                              INNER JOIN isys_cmdb_status ON isys_cmdb_status__id = isys_catg_planning_list__isys_cmdb_status__id',
                        'isys_catg_planning_list',
                        'isys_catg_planning_list__id',
                        'isys_catg_planning_list__isys_obj__id',
                        '',
                        '',
                        idoit\Module\Report\SqlQuery\Structure\SelectCondition::factory([]),
                        idoit\Module\Report\SqlQuery\Structure\SelectGroupBy::factory(['isys_catg_planning_list__isys_obj__id'])
                    ),
                    C__PROPERTY__DATA__JOIN         => [
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_planning_list', 'LEFT', 'isys_catg_planning_list__isys_obj__id', 'isys_obj__id'),
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory(
                            'isys_cmdb_status',
                            'LEFT',
                            'isys_catg_planning_list__isys_cmdb_status__id',
                            'isys_cmdb_status__id'
                        )
                    ]
                ],
                C__PROPERTY__UI       => [
                    C__PROPERTY__UI__ID     => 'C__CATG__PLANNING__STATUS',
                    C__PROPERTY__UI__PARAMS => [
                        'p_strTable' => 'isys_cmdb_status'
                    ]
                ],
                C__PROPERTY__PROVIDES => [
                    C__PROPERTY__PROVIDES__SEARCH => false
                ],
                C__PROPERTY__FORMAT   => [
                    C__PROPERTY__FORMAT__CALLBACK => [
                        'isys_export_helper',
                        'dialog'
                    ]
                ]
            ]),
            // @todo  Convert to "datetime" and add a dynamic property
            'start'       => array_replace_recursive(isys_cmdb_dao_category_pattern::date(), [
                C__PROPERTY__INFO     => [
                    C__PROPERTY__INFO__TITLE       => 'LC__UNIVERSAL__VALIDITY_FROM',
                    C__PROPERTY__INFO__DESCRIPTION => 'Validity period from'
                ],
                C__PROPERTY__DATA     => [
                    C__PROPERTY__DATA__FIELD  => 'isys_catg_planning_list__start',
                    C__PROPERTY__DATA__SELECT => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory(
                        'SELECT FROM_UNIXTIME(isys_catg_planning_list__start, \'%Y-%m-%d\') FROM isys_catg_planning_list',
                        'isys_catg_planning_list',
                        'isys_catg_planning_list__id',
                        'isys_catg_planning_list__isys_obj__id',
                        '',
                        '',
                        idoit\Module\Report\SqlQuery\Structure\SelectCondition::factory([]),
                        idoit\Module\Report\SqlQuery\Structure\SelectGroupBy::factory(['isys_catg_planning_list__isys_obj__id'])
                    ),
                    C__PROPERTY__DATA__JOIN   => [
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_planning_list', 'LEFT', 'isys_catg_planning_list__isys_obj__id', 'isys_obj__id')
                    ]
                ],
                C__PROPERTY__UI       => [
                    C__PROPERTY__UI__ID     => 'C__CATG__PLANNING__START',
                    C__PROPERTY__UI__PARAMS => [
                        'p_strClass' => 'input-mini'
                    ]
                ],
                C__PROPERTY__FORMAT   => [
                    C__PROPERTY__FORMAT__CALLBACK => [
                        'isys_export_helper',
                        'timestamp'
                    ]
                ],
                C__PROPERTY__PROVIDES => [
                    C__PROPERTY__PROVIDES__REPORT => false,
                    C__PROPERTY__PROVIDES__LIST   => true
                ]
            ]),
            // @todo  Convert to "datetime" and add a dynamic property
            'end'         => array_replace_recursive(isys_cmdb_dao_category_pattern::date(), [
                C__PROPERTY__INFO     => [
                    C__PROPERTY__INFO__TITLE       => 'LC__UNIVERSAL__VALIDITY_TO',
                    C__PROPERTY__INFO__DESCRIPTION => 'Validity period to'
                ],
                C__PROPERTY__DATA     => [
                    C__PROPERTY__DATA__FIELD  => 'isys_catg_planning_list__end',
                    C__PROPERTY__DATA__SELECT => idoit\Module\Report\SqlQuery\Structure\SelectSubSelect::factory(
                        'SELECT FROM_UNIXTIME(isys_catg_planning_list__end, \'%Y-%m-%d\') FROM isys_catg_planning_list',
                        'isys_catg_planning_list',
                        'isys_catg_planning_list__id',
                        'isys_catg_planning_list__isys_obj__id',
                        '',
                        '',
                        idoit\Module\Report\SqlQuery\Structure\SelectCondition::factory([]),
                        idoit\Module\Report\SqlQuery\Structure\SelectGroupBy::factory(['isys_catg_planning_list__isys_obj__id'])
                    ),
                    C__PROPERTY__DATA__JOIN   => [
                        idoit\Module\Report\SqlQuery\Structure\SelectJoin::factory('isys_catg_planning_list', 'LEFT', 'isys_catg_planning_list__isys_obj__id', 'isys_obj__id')
                    ]
                ],
                C__PROPERTY__UI       => [
                    C__PROPERTY__UI__ID     => 'C__CATG__PLANNING__END',
                    C__PROPERTY__UI__PARAMS => [
                        'p_strClass' => 'input-mini'
                    ]
                ],
                C__PROPERTY__FORMAT   => [
                    C__PROPERTY__FORMAT__CALLBACK => [
                        'isys_export_helper',
                        'timestamp'
                    ]
                ],
                C__PROPERTY__PROVIDES => [
                    C__PROPERTY__PROVIDES__REPORT => false,
                    C__PROPERTY__PROVIDES__LIST   => true
                ]
            ]),
            'description' => array_replace_recursive(isys_cmdb_dao_category_pattern::commentary(), [
                C__PROPERTY__INFO => [
                    C__PROPERTY__INFO__TITLE       => 'LC__CMDB__LOGBOOK__DESCRIPTION',
                    C__PROPERTY__INFO__DESCRIPTION => 'Description'
                ],
                C__PROPERTY__DATA => [
                    C__PROPERTY__DATA__FIELD => 'isys_catg_planning_list__description'
                ],
                C__PROPERTY__UI   => [
                    C__PROPERTY__UI__ID => 'C__CMDB__CAT__COMMENTARY_' . C__CMDB__CATEGORY__TYPE_GLOBAL . defined_or_default('C__CATG__PLANNING', 'C__CATG__PLANNING')
                ]
            ])
        ];
    }

    /**
     * Updates existing entity.
     *
     * @param integer $entryId
     * @param array   $entryData
     *
     * @return boolean
     */
    public function save_data($entryId, $entryData): bool
    {
        $possibleStart = strtotime($entryData['start']);
        $possibleEnd = strtotime($entryData['end']);

        $entryData['start'] = min($possibleStart, $possibleEnd);
        $entryData['end'] = max($possibleStart, $possibleEnd);

        return parent::save_data($entryId, $entryData);
    }
}
