<?php
/**
 *
 *
 * @package     i-doit
 * @subpackage
 * @author      Pavel Abduramanov <pabduramanov@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */

namespace idoit\Component\Table\Filter\Operation;

use idoit\Component\Table\Filter\Condition\ConditionInterface;
use idoit\Component\Table\Filter\Formatter\FormatterInterface;
use isys_cmdb_dao_list_objects;

/**
 * Provides basic functionality of operation: formats for column and value + conditions
 *
 * @package idoit\Component\Table\Filter
 */
abstract class Operation implements OperationInterface
{
    /** @var FormatterInterface */
    protected $formatter;

    /** @var FormatterInterface */
    protected $columnFormatter;

    /** @var ConditionInterface */
    protected $condition;

    /** @var bool */
    protected bool $isBroadsearch = false;

    /**
     * @param bool $isBroadsearch
     *
     * @return $this
     * @see ID-10110
     */
    public function setBroadsearch(bool $isBroadsearch): static
    {
        $this->isBroadsearch = $isBroadsearch;

        return $this;
    }

    /**
     * @param FormatterInterface $formatter
     *
     * @return $this
     */
    public function setFormatter(FormatterInterface $formatter): static
    {
        $this->formatter = $formatter;

        return $this;
    }

    /**
     * @param FormatterInterface $formatter
     *
     * @return $this
     */
    public function setColumnFormatter(FormatterInterface $formatter): static
    {
        $this->columnFormatter = $formatter;

        return $this;
    }

    /**
     * @param ConditionInterface $condition
     *
     * @return $this
     */
    public function setCondition(ConditionInterface $condition): static
    {
        $this->condition = $condition;

        return $this;
    }

    /**
     * Format the column name
     *
     * @param $name
     *
     * @return mixed
     */
    protected function formatColumn($name): mixed
    {
        if (isset($this->columnFormatter)) {
            return $this->columnFormatter->format($name);
        }

        return $name;
    }

    /**
     * Format the value
     *
     * @param $value
     *
     * @return mixed
     */
    protected function formatValue($value): mixed
    {
        if (isset($this->formatter)) {
            return $this->formatter->format($value);
        }

        return $value;
    }

    /**
     * Checks if the condition is met
     *
     * @param $filter
     *
     * @param $value
     *
     * @return bool
     */
    public function isApplicable($filter, $value): bool
    {
        if (isset($this->condition)) {
            return $this->condition->check($filter, $value);
        }

        return true;
    }

    /**
     * Formats the column and value and
     *
     * @param isys_cmdb_dao_list_objects $dao
     * @param                            $name
     * @param                            $value
     *
     * @return bool
     */
    public function apply(isys_cmdb_dao_list_objects $dao, $name, $value): bool
    {
        $applicable = $this->isApplicable($name, $value);
        if (!$applicable) {
            return false;
        }
        $column = $this->formatColumn($name);
        $value = $this->formatValue($value);

        return $this->applyFormatted($dao, $column, $value);
    }

    /**
     * Apply the formatted values to the DAO object
     *
     * @param isys_cmdb_dao_list_objects $dao
     * @param                            $name
     * @param                            $value
     *
     * @return bool
     */
    abstract protected function applyFormatted(isys_cmdb_dao_list_objects $dao, $name, $value): bool;
}
