<?php

/**
 * i-doit
 *
 * Tree visitor class.
 *
 * @package     modules
 * @subpackage  analytics
 * @author      Dennis Stücken <dstuecken@i-doit.org>
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */
class isys_tree_visitor_analytics implements isys_tree_visitor_interface
{
    /**
     * Filter array.
     *
     * @var  array
     */
    private $m_filter = [];

    /**
     * @var  array
     */
    private $m_path_objects = [];

    /**
     * @var  array
     */
    private $m_found_it_services = [];

    /**
     * @var  isys_tree_node[]
     */
    private $m_it_service_relations = [];

    /**
     * Returns an array of isys_tree_node objects.
     *
     * @return  array
     */
    public function get_path_objects()
    {
        return $this->m_path_objects;
    }

    /**
     * Returns an array of isys_tree_node objects.
     *
     * @return  array
     */
    public function get_it_services()
    {
        return $this->m_found_it_services;
    }

    /**
     * Returns all found it-service relations as an array of isys_tree_node objects.
     *
     * @return  isys_tree_node[]
     */
    public function get_it_service_relations()
    {
        return $this->m_it_service_relations;
    }

    /**
     * @param isys_tree_node $p_node
     * @param bool           $p_is_its
     *
     * @return isys_tree_node
     */
    private function remove(isys_tree_node $p_node, $p_is_its = false)
    {
        $l_descendants = $p_node->descendants(true);

        array_walk($l_descendants, function (isys_tree_node $p_desc) use ($p_is_its) {
            // Remove node.
            $p_desc->remove();

            if ($p_is_its) {
                // Clear it service.
                $l_data = $p_desc->get_data();
                unset($this->m_found_it_services[$l_data['data'][C__CMDB__GET__OBJECT]]);
            }
        });

        return $p_node;
    }

    /**
     *
     * @param   isys_tree|isys_tree_node $p_node
     *
     * @return  mixed
     */
    public function visit(isys_tree $p_node)
    {
        $l_return = [];

        if ($p_node instanceof isys_tree_node) {
            $l_data = $p_node->get_data();

            if (isset($l_data['data']['objTypeID']) && $l_data['data']['objTypeID'] == C__OBJTYPE__IT_SERVICE) {
                if (!isset($this->m_found_it_services[$l_data['data'][C__CMDB__GET__OBJECT]])) {
                    $this->m_found_it_services[$l_data['data'][C__CMDB__GET__OBJECT]] = $l_data;
                }

                $l_is_its = true;

                $this->m_it_service_relations[] = $p_node;
            } else {
                $l_is_its = false;
            }

            // Level cut off.
            if (isset($this->m_filter['level']) && $this->m_filter['level'] > 0 && $p_node->level() > $this->m_filter['level'] + 1) {
                $l_return[] = $this->remove($p_node, $l_is_its);
            }

            // Filter all but it services.
            if (isset($this->m_filter['all-but-it-services']) && $this->m_filter['all-but-it-services'] && !$p_node->has_children() && !$l_is_its) {
                // Remove all childs who are not it services.
                $l_return[] = $this->remove($p_node, $l_is_its);
            }

            // Filter all but the selected impact objects.
            if (isset($this->m_filter['all-but-impact-path-objects']) && is_array($this->m_filter['all-but-impact-path-objects'])) {
                if (!$p_node->has_children() && (isset($l_data['id']) && !in_array($l_data['id'], $this->m_filter['all-but-impact-path-objects']))) {
                    $l_return[] = $this->remove($p_node, $l_is_its);
                } else {
                    $this->m_path_objects[] = $p_node;
                }
            }

            // Filter by relation type.
            if (isset($this->m_filter['relation-type'], $l_data['data']['relation']['type']) && is_array($this->m_filter['relation-type'])) {
                foreach ($this->m_filter['relation-type'] as $l_hideRelation) {
                    if (defined($l_hideRelation) && constant($l_hideRelation) == $l_data['data']['relation']['type']) {
                        // @todo What about "$l_return[] = " ...?
                        $this->remove($p_node, $l_is_its);
                    }
                }
            }

            // Object type cut off.
            if (isset($this->m_filter['object-type']) && is_array($this->m_filter['object-type']) && count($this->m_filter['object-type']) > 0) {
                if (in_array($l_data['data']['objTypeID'], $this->m_filter['object-type'])) {
                    $l_return[] = $this->remove($p_node, $l_is_its);
                }
            }
        }

        // Iterate through childs and call accept as well.
        foreach ($p_node->get_childs() as $l_child) {
            $l_return = array_merge($l_return, $l_child->accept($this));
        }

        // @todo  Try to avoid this, because the "show only it-service" option will not work
        if ($p_node->count() === 0) {
            if ($p_node->get_name() === 'Root') {
                return [];
            }

            $l_parent = $p_node->get_parent();

            if (method_exists($l_parent, 'get_data')) {
                $l_data = $p_node->get_parent()
                    ->get_data();

                if ($l_data['id'] == -1) {
                    return [];
                }
            }
        }

        return $l_return;
    }

    /**
     * Construct visitor with special filter. Filter structure:
     * array(
     *   'all-but-it-services' => bool,
     *   'all-but-impact-path-objects' => bool,
     *   'level' => int
     *   'relation-type' array of relation-types as string constants
     * )
     *
     * @param  array $p_filter
     */
    public function __construct(array $p_filter)
    {
        $this->m_filter = $p_filter;
    }
}
