<?php

/**
 * i-doit
 *
 * Impact simulation report.
 *
 * @package     modules
 * @subpackage  analytics
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */
class isys_analytics_reports_impact_simulation extends isys_analytics_reports
{
    /**
     * This constant holds the name of the report.
     *
     * @var  string
     */
    const TITLE = 'LC__MODULE__ANALYTICS__FAIL_SIMULATION';

    /**
     * This constant holds the icon-path of the report.
     *
     * @var  string
     */
    const ICON = 'icons/silk/monitor.png';

    /**
     * @var  isys_analytics_dao
     */
    private $dao;

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();

        $this->dao = isys_analytics_dao::instance($this->database);
    } // function

    /**
     * Method for preparing the visual report view output.
     *
     * @return  isys_analytics_reports_impact_simulation
     */
    public function start()
    {
        $l_get = $_GET;
        $l_rules = $l_filters = $l_reports = [];

        // Remove some unnecessary parameters for the Ajax URL.
        unset($l_get[C__GET__MAIN_MENU__NAVIGATION_ID], $l_get[C__CMDB__GET__EDITMODE], $l_get[C__GET__TREE_NODE]);

        $l_itservice_filter_url = isys_helper_link::create_url([
            C__GET__MODULE_ID     => C__MODULE__ITSERVICE,
            C__GET__SETTINGS_PAGE => 'filter-config'
        ]);

        $l_itservice_filters = isys_itservice_dao_filter_config::instance($this->database)
            ->get_data();

        if (is_array($l_itservice_filters) && count($l_itservice_filters)) {
            foreach ($l_itservice_filters as $l_filter) {
                $l_filters[$l_filter['isys_itservice_filter_config__id']] = $l_filter['isys_itservice_filter_config__title'];
            } // foreach
        } // if

        // Both the service- and the result filter are based on the same data.
        $l_rules['impact_itservice_result_filter']['p_arData'] = $l_rules['impact_itservice_filter']['p_arData'] = $l_filters;

        $l_report_res = isys_report_dao::instance(isys_application::instance()->container->get('database_system'))
            ->get_reports();

        if (count($l_report_res)) {
            $l_user_id = isys_application::instance()->session->get_user_id();

            while ($l_row = $l_report_res->get_row()) {
                if (!$l_row['isys_report__user_specific'] || ($l_row['isys_report__user_specific'] && $l_row['isys_report__user'] == $l_user_id)) {
                    $l_reports[$l_row['isys_report__id']] = $l_row['isys_report__title'];
                } // if
            } // while
        } // if

        $l_rules['report_list']['p_arData'] = $l_reports;

        // Handle preselection by URL.
        $l_rules['impact_obj_selection']['p_strValue'] = $_GET[C__CMDB__GET__OBJECT];

        isys_component_template_navbar::getInstance()
            ->append_button('LC__MODULE__ANALYTICS__SIMULATION__START', 'start-simulation', [
                    'tooltip'    => 'LC__MODULE__ANALYTICS__FAIL_SIMULATION',
                    'icon'       => 'icons/silk/cog.png',
                    'js_onclick' => ';',
                    'navmode'    => 'simulation',
                ]);

        $l_cluster = $this->dao->get_object_type(C__OBJTYPE__CLUSTER);

        $this->template->activate_editmode()
            ->assign('filter_description', _L('LC__MODULE__ANALYTICS__FILTERS_DESCRIPTION', $l_itservice_filter_url))
            ->assign('ajax_url', isys_helper_link::create_url($l_get))
            ->assign('cluster_color', $l_cluster['isys_obj_type__color'])
            ->assign('use_monitoring', false)
            ->assign('title', _L(self::TITLE))
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', __DIR__ . DS . get_class() . '_view.tpl');

        return $this;
    } // function

    /**
     * This method will be called by the framework, to process ajax requests.
     *
     * @return  mixed
     * @throws  \idoit\Exception\JsonException
     * @throws  isys_exception_general
     */
    public function ajax_request()
    {
        switch ($_GET['func']) {
            default:
            case 'start-simulation':
                $l_impact_objects = isys_format_json::decode($_POST['impact_obj']);

                if (!is_array($l_impact_objects)) {
                    throw new isys_exception_general(_L('LC__MODULE__ANALYTICS__NOTIFY__NO_IMPACT_OBJECT'));
                } // if

                return $this->start_simulation($l_impact_objects, (int)$_POST['filter']);

            case 'load-report-impact-objects':
                return $this->load_report_impact_objects($_POST['report']);
        } // switch
    } // function

    /**
     * This method triggers the simulation algorithm!
     *
     * @param   array   $p_impact_obj
     * @param   integer $p_filter
     *
     * @throws  isys_exception_general
     * @return  array
     */
    public function start_simulation(array $p_impact_obj, $p_filter = null)
    {
        set_time_limit(0);

        if (count($p_impact_obj) === 0) {
            throw new isys_exception_general(_L('LC__MODULE__ANALYTICS__NOTIFY__NO_IMPACT_OBJECT'));
        } // if

        $l_filter = $l_it_service_mapping = [];

        // Prepare the filters.
        if ($p_filter > 0) {
            $l_filter = isys_itservice_dao_filter_config::instance($this->database)
                ->get_data($p_filter);

            $l_filter = $l_filter['formatted__data'];
        } // if

        if (empty($l_filter['level'])) {
            $l_filter['level'] = 15;
        } // if

        // Every simulation shall begin with an empty cache.
        $this->dao->get_cache()
            ->clear();

        /**
         * Initialize return structure
         */
        $l_return = [
            'tree_data'          => null,
            'simulation_objects' => $p_impact_obj,
            'relation_weighting' => [],
            'highest_weighting'  => [],
            'itservices'         => null
        ];

        // Populate simulation objects.
        $p_impact_obj = array_unique($p_impact_obj);
        $this->dao->set_simulation_objects($p_impact_obj);

        // Get keyvalue cache.
        $l_cache = new isys_cache_keyvalue_dummy; // Switching the cache off "isys_cache::keyvalue();"
        $l_cache->set_options(['flags' => MEMCACHE_COMPRESSED, 'expiration' => 60]);
        $l_cache_key = 'analytics.bottom-up.filter-' . $p_filter . '.' . md5(implode(',', $p_impact_obj)) . '.tree';

        if (!$l_tree = $l_cache->get($l_cache_key)) {
            // Initialize tree.
            $l_tree = new isys_tree();

            // Add root node.
            $l_root_node = new isys_tree_node_explorer([
                'name' => 'Root',
                'id'   => -1
            ]);

            $l_tree->add($l_root_node);

            $l_node_cache = [];

            $l_dao_algorithm = isys_itservice_dao_algorithm::instance($this->database);
            $l_dao_algorithm->set_filter($l_filter)
                ->set_formatter([new idoit\Module\Analytics\Component\Tree\Formatter, 'format']);

            foreach ($p_impact_obj as $l_object) {
                if (!isset($l_node_cache[$l_object])) {
                    // Get new root node for $l_object.
                    $l_node_cache[$l_object] = $l_dao_algorithm->clear_node_cache()
                        ->format_node($l_object);

                    try {
                        // Walk through relations and build the tree.
                        $l_dao_algorithm->relation_walk($l_node_cache[$l_object], $l_filter['level'] ?: null, true);
                    } catch (Exception $e) {
                        isys_notify::error($e->getMessage(), ['sticky' => true]);
                    } // try

                    // Modify subnode count.
                    $l_node_cache[$l_object]->set_subnodes($l_node_cache[$l_object]->count());
                } // if

                // Add the whole node structure as a root node to $l_tree.
                $l_root_node->add($l_node_cache[$l_object]);
            } // foreach

            // Cache the tree.
            $l_cache->set($l_cache_key, $l_tree, 60);
        } // if

        // Remove all but it services.
        $l_filter['all-but-it-services'] = (isset($_POST['it_services_only']) && $_POST['it_services_only']);

        // Prepare tree.
        if (isset($l_tree) && $l_tree instanceof isys_tree) {
            // Filter tree using the visitor pattern.
            $l_visitor = new isys_tree_visitor_analytics($l_filter);

            while ($l_tree->accept($l_visitor)) {
                // filtering..
            } // while

            $l_return['itservices'] = $l_visitor->getItServices();
            $l_it_services = $l_visitor->getItServiceRelations();

            /* @var  isys_tree_node_explorer $l_itservice_node */
            foreach ($l_it_services as $l_itservice_node) {
                /* @var  isys_tree_node_explorer $l_current */
                $l_current = $l_itservice_node;
                $l_current_level = $l_current->level();

                if ($l_current_level > 3) {
                    for ($i = $l_current_level;$i > 3;$i--) {
                        $l_current = $l_current->get_parent();
                    } // while
                } // if

                // We select the object on level 3 (1=root, 2=simulation object, 3=first affected object).
                if ($l_current && $l_current->level() === 3) {
                    $l_first_child_data = $l_current->get_data();

                    if (isset($l_first_child_data['data']['relation']) && isset($l_first_child_data['data']['relation']['id'])) {
                        $l_return['relation_weighting'][$l_itservice_node->get_id()][$l_first_child_data['data']['relation']['id']] = $this->calculate_relation_weighting($l_first_child_data['data']['relation'],
                            $p_impact_obj);
                    } // if
                } // if
            } // foreach

            // Now we calculate the "highest weighting".
            foreach ($l_return['relation_weighting'] as $l_itservice => $l_relations) {
                $l_return['highest_weighting'][$l_itservice] = [
                    'weighting' => 99,
                    'cluster'   => false
                ];

                foreach ($l_relations as $l_relation => $l_relation_data) {
                    if ($l_return['highest_weighting'][$l_itservice]['weighting'] > $l_return['relation_weighting'][$l_itservice][$l_relation]['weighting']) {
                        $l_return['highest_weighting'][$l_itservice]['weighting'] = $l_return['relation_weighting'][$l_itservice][$l_relation]['weighting'];
                        $l_return['highest_weighting'][$l_itservice]['cluster'] = $l_return['relation_weighting'][$l_itservice][$l_relation]['cluster_members'] > 0;
                    } // if
                } // foreach
            } // foreach

            // Pass tree data.
            $l_return['tree_data'] = $l_tree->toArray();
        } // if

        return $l_return;
    } // function

    /**
     * This method will calculate the relation weighting (necessary for cluster relations).
     *
     * @param   array $p_relation_data
     * @param   array $p_impact_obj
     *
     * @return  array
     */
    private function calculate_relation_weighting(array $p_relation_data, array $p_impact_obj = [])
    {
        if ($p_relation_data['cluster_members'] > 0) {
            $l_counter = 0;

            $l_cluster_res = isys_cmdb_dao_list_catg_cluster_members::instance($this->database)
                ->get_result(null, $p_relation_data['slave']);

            if (count($l_cluster_res)) {
                while ($l_row = $l_cluster_res->get_row()) {
                    if (in_array($l_row['isys_obj__id'], $p_impact_obj)) {
                        $l_counter++;
                    } // if
                } // while

                // New calculation logic.
                $p_relation_data['weighting'] = round(11 - ((11 - $p_relation_data['weighting']) * ($l_counter / $p_relation_data['cluster_members'])), 2);
//				$p_relation_data['weighting'] = round($p_relation_data['weighting'] - ($l_counter / $p_relation_data['cluster_members']) * $p_relation_data['weighting'], 2);

                if ($l_counter == $p_relation_data['cluster_members']) {
                    $p_relation_data['cluster_members'] = 0;
                } // if
            } // if
        } // if

        return $p_relation_data;
    } // function

    /**
     * Method for retrieving all objects of the given report.
     *
     * @param   integer $p_report_id
     *
     * @throws  isys_exception_general
     * @return  array
     */
    private function load_report_impact_objects($p_report_id)
    {
        $l_return = [];
        $l_dao = isys_cmdb_dao::instance($this->database);
        $l_report_dao = isys_report_dao::instance(isys_application::instance()->container->get('database_system'));

        $l_report = $l_report_dao->get_report($p_report_id);

        $l_report_data = $l_report_dao->query($l_report['isys_report__query']);

        if (isset($l_report_data['content'][0]['__id__'])) {
            foreach ($l_report_data['content'] as $l_row) {
                $l_obj = $l_dao->get_object_by_id($l_row['__id__'])
                    ->get_row();

                $l_return[] = [
                    'isys_obj__id'         => $l_obj['isys_obj__id'],
                    'isys_obj__title'      => $l_obj['isys_obj__title'],
                    'isys_obj_type__id'    => $l_obj['isys_obj_type__id'],
                    'isys_obj_type__title' => _L($l_obj['isys_obj_type__title'])
                ];
            }
        } else {
            throw new isys_exception_general(_L('LC__MODULE__ANALYTICS__SIMULATION__NO_OBJECTS_FOUND_IN_REPORT'));
        }

        return $l_return;
    }
}
