<?php

/**
 * i-doit
 *
 * Dataquality report.
 *
 * @package     modules
 * @subpackage  analytics
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */
class isys_analytics_reports_dataquality extends isys_analytics_reports
{
    /**
     * This constant holds the name of the report.
     *
     * @var  string
     */
    const TITLE = 'LC__MODULE__ANALYTICS__DATA_QUALITY';

    /**
     * This constant holds the icon-path of the report.
     *
     * @var  string
     */
    const ICON = 'icons/silk/monitor.png';

    /**
     * @var  isys_analytics_dao_dataquality
     */
    private $dao;

    /**
     * Configuration array, filled by isys_usersettings.
     *
     * @var  array
     */
    private $m_config = [];

    /**
     * This variable holds the currently active profile ID.
     *
     * @var  integer
     */
    private $m_current_profile;

    /**
     * This array holds blacklisted categories, which shall not be used for the data quality calculation.
     *
     * @var  array
     */
    private $m_category_blacklist = [
        'g' => [
            C__CATG__LOGBOOK,
            C__CATG__IMAGE,
            C__CATG__OVERVIEW,
            C__CATG__OBJECT,
            C__CATG__CLUSTER_ROOT,
            C__CATG__CLUSTER,
            C__CATG__CLUSTER_SHARED_STORAGE,
            C__CATG__CLUSTER_VITALITY,
            C__CATG__VIRTUAL_HOST_ROOT,
            C__CATG__CLUSTER_SHARED_VIRTUAL_SWITCH,
            C__CATG__ITS_LOGBOOK,
            C__CATG__OBJECT_VITALITY,
            C__CATG__RELATION,
            C__CATG__NETWORK_PORT_OVERVIEW,
            C__CATG__VIRTUAL_TICKETS,
            C__CATG__RACK_VIEW,
            C__CATG__LIVESTATUS,
            C__CATG__NET_CONNECTIONS_FOLDER,
            C__CATG__NDO,
            C__CATG__VIRTUAL_AUTH
        ],
        's' => [
            C__CATS__PRT,
            C__CATS__FILE,
            C__CATS__LICENCE
        ]
    ];

    /**
     * This array holds categories which need special treatment (because of untypical fieldnames or isys_connection).
     *
     * @var array
     */
    const SPECIAL_CATEGORIES = [
        'C__CATG__IT_SERVICE',
        'C__CATG__OPERATING_SYSTEM',
        'C__CATG__CLUSTER_MEMBERSHIPS',
        'C__CATG__GUEST_SYSTEMS',
    ];

    /**
     * This array holds blacklisted object types, which shall not be displayed in the frontend.
     *
     * @var  array
     */
    const OBJECT_TYPE_BLACKLIST = [
        'C__OBJTYPE__RELATION',
        'C__OBJTYPE__PARALLEL_RELATION'
    ];

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();

        $this->dao = isys_analytics_dao_dataquality::instance($this->database);

        $session = isys_application::instance()->container->get('session');
        $this->m_current_profile = $_GET[C__ANALYTICS__REPORT_PROFILE];

        $l_config = [];
        $l_config_res = $this->dao->get_profiles(null, $session->get_user_id());

        if (count($l_config_res)) {
            while ($l_config_row = $l_config_res->get_row()) {
                try {
                    $l_config_row['isys_analytics_dataquality_profiles__data'] = isys_format_json::decode($l_config_row['isys_analytics_dataquality_profiles__data']);
                } catch (Exception $e) {
                    $l_config_row = [];
                }

                $l_config[$l_config_row['isys_analytics_dataquality_profiles__id']] = $l_config_row;
            }
        }

        $this->m_config = $l_config;
    }

    /**
     * This method builds the tree for the menu.
     *
     * @param   isys_component_tree $p_tree
     * @param   integer             $p_parent
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function build_tree(isys_component_tree &$p_tree, $p_parent)
    {
        $l_cnt = 100;

        foreach ($this->m_config as $l_config) {
            $p_tree->add_node(
                C__MODULE__ANALYTICS . $l_cnt,
                $p_parent,
                $l_config['isys_analytics_dataquality_profiles__title'],
                isys_helper_link::create_url([
                    C__GET__MODULE_ID            => C__MODULE__ANALYTICS,
                    C__GET__TREE_NODE            => C__MODULE__ANALYTICS . $l_cnt,
                    C__ANALYTICS__REPORT         => __CLASS__,
                    C__ANALYTICS__REPORT_PROFILE => $l_config['isys_analytics_dataquality_profiles__id']
                ]),
                '',
                isys_application::instance()->www_path . 'images/icons/silk/table_multiple.png',
                $this->m_current_profile == $l_config['isys_analytics_dataquality_profiles__id']
            );

            $l_cnt++;
        }
    }

    /**
     * Build breadcrumb navifation
     *
     * @param  $p_gets
     *
     * @return array
     */
    public function breadcrumb_get($p_gets)
    {
        $l_return = [
            [
                _L($this->getTitle()) => [
                    C__GET__MODULE_ID    => C__MODULE__ANALYTICS,
                    C__GET__TREE_NODE    => $_GET[C__GET__TREE_NODE],
                    C__ANALYTICS__REPORT => $p_gets[C__ANALYTICS__REPORT]
                ]
            ]
        ];

        if ($this->m_current_profile > 0) {
            $l_return[] = [
                $this->m_config[$this->m_current_profile]['isys_analytics_dataquality_profiles__title'] => [
                    C__GET__MODULE_ID            => C__MODULE__ANALYTICS,
                    C__GET__TREE_NODE            => $_GET[C__GET__TREE_NODE],
                    C__ANALYTICS__REPORT         => $p_gets[C__ANALYTICS__REPORT],
                    C__ANALYTICS__REPORT_PROFILE => $this->m_current_profile
                ]
            ];
        }

        return $l_return;
    }

    /**
     * Method for preparing the visual report view output.
     *
     * @return  isys_analytics_reports_impact_simulation
     */
    public function start()
    {
        global $g_dirs;

        $l_get = $_GET;
        $l_object_types = [];

        // Remove some unnecessary parameters for the Ajax URL.
        unset($l_get[C__GET__MAIN_MENU__NAVIGATION_ID], $l_get[C__CMDB__GET__EDITMODE], $l_get[C__GET__TREE_NODE]);

        $l_object_type_res = $this->dao->get_object_types();

        if (count($l_object_type_res)) {
            $l_obj_types = array_keys($this->m_config[$this->m_current_profile]['isys_analytics_dataquality_profiles__data'] ?: []);

            while ($l_obj_type = $l_object_type_res->get_row()) {
                if (in_array($l_obj_type['isys_obj_type__const'], self::OBJECT_TYPE_BLACKLIST, true) ||
                    (count($l_obj_types) && !in_array($l_obj_type['isys_obj_type__const'], $l_obj_types, true))) {
                    continue;
                }

                $l_icon = isys_application::instance()->www_path . 'images/icons/silk/page_white.png';

                if (!empty($l_obj_type['isys_obj_type__icon'])) {
                    if (strpos($l_obj_type['isys_obj_type__icon'], '/') !== false) {
                        $l_icon = $l_obj_type['isys_obj_type__icon'];
                    } else {
                        $l_icon = isys_application::instance()->www_path . 'images/tree/' . $l_obj_type['isys_obj_type__icon'];
                    }
                }

                $l_obj_sql = 'SELECT COUNT(isys_obj__id) AS count
					FROM isys_obj
					WHERE isys_obj__isys_obj_type__id = ' . $this->dao->convert_sql_id($l_obj_type['isys_obj_type__id']) . '
					AND isys_obj__status = ' . $this->dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . ';';

                $l_obj_count = $this->dao->retrieve($l_obj_sql)
                    ->get_row_value('count');

                $l_object_types[_L($l_obj_type['isys_obj_type__title'])] = [
                    'id'               => $l_obj_type['isys_obj_type__id'],
                    'const'            => $l_obj_type['isys_obj_type__const'],
                    'title'            => _L($l_obj_type['isys_obj_type__title']),
                    'icon'             => $l_icon,
                    'obj_count'        => $l_obj_count,
                    'obj_count_string' => $l_obj_count . ' ' . (($l_obj_count == 1) ? _L('LC_UNIVERSAL__OBJECT') : _L('LC_UNIVERSAL__OBJECTS'))
                ];
            }
        }

        $l_rules['C__ANALYTICS__PROFILE_NAME']['p_strValue'] = $this->m_config[$this->m_current_profile]['isys_analytics_dataquality_profiles__title'];
        $l_rules['C__ANALYTICS__PROFILE_USERSPECIFIC']['p_arData'] = get_smarty_arr_YES_NO();

        $l_save_overlay = [
            [
                'title'   => _L('LC__MODULE__ANALYTICS__DATA_QUALITY__SAVE_AS_NEW_PROFILE'),
                'icon'    => 'icons/silk/page_save.png',
                'href'    => 'javascript:',
                'onclick' => '',
                'id'      => 'navbar_item_C__NAVMODE__SAVE_AS'
            ]
        ];

        $l_navbar = isys_component_template_navbar::getInstance();
        $l_auth = isys_auth_analytics::instance();

        if ($l_auth->is_allowed_to(isys_auth::EDIT, 'DATAQUALITY/' . isys_auth::WILDCHAR)) {
            $l_navbar->append_button('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__MANAGE_PROFILES', 'dataquality_profiles', [
                'tooltip'    => 'LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__MANAGE_PROFILES_DESCRIPTION',
                'active'     => true,
                'visible'    => true,
                'icon'       => 'icons/silk/pencil.png',
                'navmode'    => 'dataquality_profiles',
                'js_onclick' => "get_popup('analytics_dataquality_profiles', '', '900', '550', {}, '');"
            ]);
        }

        if ($this->m_current_profile > 0) {
            $l_navbar->append_button('LC__MODULE__ANALYTICS__DATA_QUALITY__RESET_VIEW', 'reset-object-types', [
                'tooltip'    => 'LC__MODULE__ANALYTICS__DATA_QUALITY__RESET_VIEW',
                'icon'       => 'icons/silk/arrow_refresh.png',
                'js_onclick' => ';',
                'navmode'    => 'reset'
            ]);

            if ($l_auth->is_allowed_to(isys_auth::EDIT, 'DATAQUALITY/' . $this->m_current_profile)) {
                $l_navbar->set_active(true, C__NAVBAR_BUTTON__SAVE)
                    ->set_overlay($l_save_overlay, C__NAVBAR_BUTTON__SAVE);

                if ($this->m_config[$this->m_current_profile]['isys_analytics_dataquality_profiles__isys_obj__id'] > 0) {
                    $l_navbar->append_button('LC__MODULE__ANALYTICS__DATA_QUALITY__MAKE_PUBLIC', 'publish-profile', [
                        'tooltip'    => 'LC__MODULE__ANALYTICS__DATA_QUALITY__MAKE_PUBLIC',
                        'icon'       => 'icons/silk/group.png',
                        'js_onclick' => ';',
                        'navmode'    => 'publish'
                    ]);
                }
            }

            if ($l_auth->is_allowed_to(isys_auth::DELETE, 'DATAQUALITY/' . $this->m_current_profile)) {
                $l_navbar->append_button('LC__MODULE__ANALYTICS__DATA_QUALITY__PURGE', 'purge-profile', [
                    'tooltip'    => 'LC__MODULE__ANALYTICS__DATA_QUALITY__PURGE',
                    'icon'       => 'icons/silk/page_delete.png',
                    'js_onclick' => ';',
                    'navmode'    => 'purge_profile'
                ]);
            }
        } else {
            if ($l_auth->is_allowed_to(isys_auth::EDIT, 'DATAQUALITY/' . isys_auth::WILDCHAR)) {
                $l_navbar->set_active(true, C__NAVBAR_BUTTON__SAVE);
            }
        }

        ksort($l_object_types);

        $l_categories = [];

        $l_category_data = $this->dao->get_all_categories();

        foreach ($l_category_data[C__CMDB__CATEGORY__TYPE_GLOBAL] as $l_cat) {
            $l_categories['g_' . $l_cat['id']] = [
                'title' => _L($l_cat['title']),
                'color' => '#' . substr(md5($l_cat['title']), 0, 6),
                'const' => $l_cat['const']
            ];
        }

        foreach ($l_category_data[C__CMDB__CATEGORY__TYPE_SPECIFIC] as $l_cat) {
            $l_categories['s_' . $l_cat['id']] = [
                'title' => _L($l_cat['title']),
                'color' => '#' . substr(md5($l_cat['title']), 0, 6),
                'const' => $l_cat['const']
            ];
        }

        foreach ($l_category_data[C__CMDB__CATEGORY__TYPE_CUSTOM] as $l_cat) {
            $l_categories['g_custom_' . $l_cat['id']] = [
                'title' => _L($l_cat['title']),
                'color' => '#' . substr(md5($l_cat['title']), 0, 6),
                'const' => $l_cat['const']
            ];
        }

        $this->template->activate_editmode()
            ->assign('profile_id', $this->m_current_profile)
            ->assign('ajax_url', isys_helper_link::create_url($l_get))
            ->assign('object_types', $l_object_types)
            ->assign('categories', $l_categories)
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', __DIR__ . DS . get_class() . '_view.tpl');

        return $this;
    }

    /**
     * This method will be called by the framework, to process ajax requests.
     *
     * @param   mixed $p_data
     *
     * @return  mixed
     * @throws  isys_exception_database
     */
    public function ajax_request($p_data = [])
    {
        $session = isys_application::instance()->container->get('session');

        $l_profile_id = $_GET[C__ANALYTICS__REPORT_PROFILE] ?: null;

        switch ($_GET['func']) {
            default:
            case 'load-category-data':
                $session->write_close();

                return $this->loadCategoryData($_POST['obj_type_id'], (bool)$_POST['reset_categories']);

            case 'load-objects-without-data':
                $session->write_close();

                return $this->loadObjectsWithoutData($_POST['obj_type_id'], $_POST['category'], $_POST['category_type']);

            case 'save-profile':
                $this->dao->save_profile($l_profile_id, ['data' => $_POST['data']]);

                return null;

            case 'save-profile-as':
                $l_data = [
                    'data'         => $_POST['data'],
                    'title'        => $_POST['title'],
                    'isys_obj__id' => $session->get_user_id(),
                ];

                $session->write_close();
                $this->dao->save_profile(null, $l_data);

                return isys_helper_link::create_url([
                    C__GET__MODULE_ID            => C__MODULE__ANALYTICS,
                    C__GET__TREE_NODE            => $_GET[C__GET__TREE_NODE],
                    C__ANALYTICS__REPORT         => __CLASS__,
                    C__ANALYTICS__REPORT_PROFILE => $this->dao->get_last_insert_id()
                ]);

            case 'publish-profile':
                $this->dao->save_profile($l_profile_id, ['isys_obj__id' => null]);

                return null;

            case 'delete-profile':
                $this->dao->delete_profile($l_profile_id);

                return isys_helper_link::create_url([
                    C__GET__MODULE_ID    => C__MODULE__ANALYTICS,
                    C__GET__TREE_NODE    => $_GET[C__GET__TREE_NODE],
                    C__ANALYTICS__REPORT => __CLASS__
                ]);
        }
    }

    /**
     * This function will retrieve the types and some "simple" statistics of the given objects.
     *
     * @param  integer $objectTypeId
     * @param  boolean $reset
     *
     * @return array
     * @throws isys_exception_database
     */
    private function loadCategoryData($objectTypeId, $reset)
    {
        $responseData = [];
        $sqlQueryParts = [];
        $incompatibleSqlTables = [
            'isys_catg_nagios_refs_services_list',
            'isys_catg_cards_list_2_isys_obj_list'
        ];

        $objectTypeData = $this->dao->get_object_type($objectTypeId);
        $categoryResult = $this->dao->get_catg_by_obj_type($objectTypeId);
        $statusNormal = $this->dao->convert_sql_int(C__RECORD_STATUS__NORMAL);

        $l_categories = $this->m_config[$this->m_current_profile]['isys_analytics_dataquality_profiles__data'][$objectTypeData['isys_obj_type__const']] ?: [];

        if ($reset) {
            $l_categories = [];
        }

        // Collect additional categories, which we don't want to display:
        $categoryBlacklist = $this->dao->get_all_categories([isys_cmdb_dao_category::TYPE_VIEW, isys_cmdb_dao_category::TYPE_REAR]);

        if (isset($categoryBlacklist[C__CMDB__CATEGORY__TYPE_GLOBAL]) && count($categoryBlacklist[C__CMDB__CATEGORY__TYPE_GLOBAL])) {
            foreach ($categoryBlacklist[C__CMDB__CATEGORY__TYPE_GLOBAL] as $l_g_category) {
                $this->m_category_blacklist['g'][] = $l_g_category['const'];
            }
        }

        if (isset($categoryBlacklist[C__CMDB__CATEGORY__TYPE_SPECIFIC]) && count($categoryBlacklist[C__CMDB__CATEGORY__TYPE_SPECIFIC])) {
            foreach ($categoryBlacklist[C__CMDB__CATEGORY__TYPE_SPECIFIC] as $l_s_category) {
                $this->m_category_blacklist['s'][] = $l_s_category['const'];
            }
        }

        if (count($categoryResult)) {
            while ($l_category = $categoryResult->get_row()) {
                if (!$reset && count($l_categories) > 0 && !in_array($l_category['isysgui_catg__const'], $l_categories, true)) {
                    continue;
                }

                if (!class_exists($l_category['isysgui_catg__class_name']) || in_array($l_category['isysgui_catg__id'], $this->m_category_blacklist['g'], false)) {
                    continue;
                }

                $sqlTable = $l_category['isysgui_catg__source_table'] . '_list';

                if (!$this->dao->table_exists($sqlTable)) {
                    continue;
                }

                if (in_array($l_category['isysgui_catg__const'], self::SPECIAL_CATEGORIES, true)) {
                    $sqlQueryParts[] = '(' . $this->getSpecialCategoryCountSql($l_category['isysgui_catg__const'], $sqlTable) . ') as count_g_' . $l_category['isysgui_catg__id'];
                    continue;
                }

                $sqlObjectReferenceField = $this->getObjectReferenceField($l_category['isysgui_catg__const'], $sqlTable);

                if (!in_array($sqlTable, $incompatibleSqlTables, true)) {
                    $sqlQueryParts[] = '(SELECT COUNT(' . $sqlTable . '__id) 
                        FROM ' . $sqlTable . ' 
                        WHERE ' . $sqlObjectReferenceField . ' = isys_obj__id
                        AND ' . $sqlTable . '__status = ' . $statusNormal . ') as count_g_' . $l_category['isysgui_catg__id'];
                }
            }
        }

        $categoryResult = $this->dao->gui_get_cats_by_objtype_id($objectTypeId);

        if (count($categoryResult)) {
            while ($l_category = $categoryResult->get_row()) {
                if (count($l_categories) > 0 && !in_array($l_category['isysgui_cats__const'], $l_categories, true)) {
                    continue;
                }

                if (!class_exists($l_category['isysgui_cats__class_name']) || in_array($l_category['isysgui_cats__id'], $this->m_category_blacklist['s'], false)) {
                    continue;
                }

                $sqlTable = $l_category['isysgui_cats__source_table'];

                if (!isset($sqlQueryParts[$sqlTable]) && !in_array($sqlTable, $incompatibleSqlTables, true)) {
                    $sqlQueryParts[$sqlTable] = '(SELECT COUNT(' . $sqlTable . '__id) 
                        FROM ' . $sqlTable . ' 
                        WHERE ' . $sqlTable . '__isys_obj__id = isys_obj__id
                        AND ' . $sqlTable . '__status = ' . $statusNormal . ') as count_s_' . $l_category['isysgui_cats__id'];
                }
            }
        }

        $categoryResult = $this->dao->gui_get_catg_custom_by_objtype_id($objectTypeId);

        if (count($categoryResult)) {
            while ($l_category = $categoryResult->get_row()) {
                if (count($l_categories) > 0 && !in_array($l_category['isysgui_catg_custom__const'], $l_categories, true)) {
                    continue;
                }

                $sqlTable = 'isys_catg_custom_fields_list_' . $l_category['isysgui_catg_custom__id'];

                if (!isset($sqlQueryParts[$sqlTable])) {
                    $sqlQueryParts[$sqlTable] = '(SELECT COUNT(isys_catg_custom_fields_list__id) 
                        FROM isys_catg_custom_fields_list 
                        WHERE isys_catg_custom_fields_list__isys_obj__id = isys_obj__id
                        AND isys_catg_custom_fields_list__isysgui_catg_custom__id = ' . $this->dao->convert_sql_id($l_category['isysgui_catg_custom__id']) . '
                        AND isys_catg_custom_fields_list__field_type = "commentary"
                        AND isys_catg_custom_fields_list__status = ' . $statusNormal . ') as count_g_custom_' .
                        $l_category['isysgui_catg_custom__id'];
                }
            }
        }

        // @see  ANALYSE-66  If there are no category selects, we'll just add the global category to at least count something (and prevent SQL errors).
        if (count($sqlQueryParts) === 0) {
            $message = $this->language->get('LC__MODULE__ANALYTICS__DATA_QUALITY__OBJECT_TYPE_WITHOUT_CATEGORIES', [
                $this->language->get($objectTypeData['isys_obj_type__title'])
            ]);

            isys_application::instance()->container->get('notify')->warning($message, ['sticky' => true]);

            $sqlQueryParts[] = '(SELECT COUNT(isys_catg_global_list__id)
                FROM isys_catg_global_list
                WHERE isys_catg_global_list__isys_obj__id = isys_obj__id
                AND isys_catg_global_list__status = ' . $statusNormal . ') as count_g_' . defined_or_default('C__CATG__GLOBAL', 1);
        }

        $sql = 'SELECT ' . implode(', ', $sqlQueryParts) . '
            FROM isys_obj
            WHERE isys_obj__isys_obj_type__id = ' . $this->dao->convert_sql_id($objectTypeId) . '
            AND isys_obj__status = ' . $statusNormal . ';';

        $l_result = $this->dao->retrieve($sql);

        while ($l_row = $l_result->get_row()) {
            foreach ($l_row as $l_cat => $l_count) {
                $l_cat = (string) substr($l_cat, 6);

                if (!isset($responseData[$l_cat])) {
                    $responseData[$l_cat] = 0;
                }

                if ($l_count > 0) {
                    $responseData[$l_cat]++;
                }
            }
        }

        return [
            'objects' => $l_result->count(),
            'data'    => $responseData
        ];
    }

    /**
     * Method for loading objects of a given type that are missing data in the given category.
     *
     * @param  integer $objectTypeId
     * @param  string  $categoryConstant
     * @param  string  $categoryType
     *
     * @return array
     * @throws isys_exception_cmdb
     * @throws isys_exception_database
     */
    private function loadObjectsWithoutData($objectTypeId, $categoryConstant, $categoryType)
    {
        $return = [];
        $countCondition = null;

        switch ($categoryType) {
            case 'g':
            default:
                $categoryRow = $this->dao->get_catg_by_const($categoryConstant)->get_row();
                $tableName = $categoryRow['isysgui_catg__source_table'] . '_list';

                $link = isys_helper_link::create_catg_url([
                    C__CMDB__GET__OBJECT => '%objID%',
                    C__CMDB__GET__CATG   => $categoryRow['isysgui_catg__id']
                ]);

                break;

            case 's':
                $categoryRow = $this->dao->get_cats_by_const($categoryConstant)->get_row();
                $tableName = $categoryRow['isysgui_cats__source_table'];

                $link = isys_helper_link::create_cats_url([
                    C__CMDB__GET__OBJECT => '%objID%',
                    C__CMDB__GET__CATS   => $categoryRow['isysgui_cats__id']
                ]);

                break;

            case 'g_custom':
                $customCategoryId = constant($categoryConstant);

                $tableName = 'isys_catg_custom_fields_list';
                $countCondition = 'AND isys_catg_custom_fields_list__isysgui_catg_custom__id = ' . $this->dao->convert_sql_id($customCategoryId);

                $sql = 'SELECT isys_obj__id, isys_obj__title, isys_obj__sysid, isys_obj__status 
                FROM isys_obj
                WHERE isys_obj__isys_obj_type__id = ' . $this->dao->convert_sql_id($objectTypeId) . ' 
                AND isys_obj__id NOT IN( SELECT DISTINCT(isys_catg_custom_fields_list__isys_obj__id) FROM isys_catg_custom_fields_list WHERE isys_catg_custom_fields_list__isysgui_catg_custom__id = ' . $this->dao->convert_sql_id($customCategoryId) . ' )
                HAVING isys_obj__status = ' . $this->dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
                LIMIT 100;';

                $link = isys_helper_link::create_catg_custom_url([
                    C__CMDB__GET__OBJECT      => '%objID%',
                    C__CMDB__GET__CATG_CUSTOM => $customCategoryId
                ]);

                break;
        }

        if (in_array($categoryConstant, self::SPECIAL_CATEGORIES, true)) {
            $sql = $this->getSpecialCategoryDataSql($objectTypeId, $categoryConstant, $tableName);
        } elseif (empty($sql)) {
            $objectReferenceField = $this->getObjectReferenceField($categoryConstant, $tableName);

            $sql = 'SELECT isys_obj__id, isys_obj__title, isys_obj__sysid, isys_obj__status 
                FROM isys_obj
                LEFT JOIN ' . $tableName . ' ON ' . $objectReferenceField . ' = isys_obj__id
                WHERE isys_obj__isys_obj_type__id = ' . $this->dao->convert_sql_id($objectTypeId) . ' 
                AND ' . $objectReferenceField . ' IS NULL
                HAVING isys_obj__status = ' . $this->dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
                LIMIT 100;';
        }


        $l_result = $this->dao->retrieve($sql);

        while ($row = $l_result->get_row()) {
            $return[] = [
                'id'    => $row['isys_obj__id'],
                'title' => _L($row['isys_obj__title']),
                'sysid' => $row['isys_obj__sysid'],
                'link'  => str_replace('%objID%', $row['isys_obj__id'], $link)
            ];
        }

        return $return;
    }

    /**
     * Some categories require special handling.
     *
     * @param  string $categoryConst
     * @param  string $tableName
     *
     * @return string
     */
    private function getObjectReferenceField($categoryConst, $tableName)
    {
        switch ($categoryConst) {
            case 'C__CATG__PERSON_ASSIGNED_WORKSTATION':
                return 'isys_catg_logical_unit_list__isys_obj__id__parent';

            default:
                return $tableName . '__isys_obj__id';
        }
    }

    /**
     * Some categories require special SQL queries to return a valid count.
     *
     * @param  string $categoryConst
     * @param  string $tableName
     *
     * @return string
     */
    private function getSpecialCategoryCountSql($categoryConst, $tableName)
    {
        $statusNormal = $this->dao->convert_sql_int(C__RECORD_STATUS__NORMAL);

        switch ($categoryConst) {
            case 'C__CATG__CLUSTER_MEMBERSHIPS':
            case 'C__CATG__GUEST_SYSTEMS':
            case 'C__CATG__IT_SERVICE':
                return 'SELECT COUNT(' . $tableName . '__id)
                    FROM ' . $tableName . '
                    INNER JOIN isys_connection ON isys_connection__id = ' . $tableName . '__isys_connection__id
                    WHERE isys_connection__isys_obj__id = isys_obj__id
                    AND ' . $tableName . '__status = ' . $statusNormal;

            case 'C__CATG__OPERATING_SYSTEM':
                return 'SELECT COUNT(' . $tableName . '__id)
                    FROM ' . $tableName . '
                    WHERE ' . $tableName . '__isys_obj__id = isys_obj__id
                    AND isys_catg_application_list__isys_catg_application_type__id = ' . $this->dao->convert_sql_id(C__CATG__APPLICATION_TYPE__OPERATING_SYSTEM) . '
                    AND ' . $tableName . '__status = ' . $statusNormal;
        }
    }

    /**
     * Some categories require special SQL queries to return a valid count.
     *
     * @param  integer $objectTypeId
     * @param  string  $categoryConst
     * @param  string  $tableName
     *
     * @return string
     */
    private function getSpecialCategoryDataSql($objectTypeId, $categoryConst, $tableName)
    {
        $statusNormal = $this->dao->convert_sql_int(C__RECORD_STATUS__NORMAL);

        switch ($categoryConst) {
            default:
                return 'SELECT isys_obj__id, isys_obj__title, isys_obj__sysid,
                    (' . $this->getSpecialCategoryCountSql($categoryConst, $tableName) . ') as count
                    FROM isys_obj
                    WHERE isys_obj__isys_obj_type__id = ' . $this->dao->convert_sql_id($objectTypeId) . '
                    AND isys_obj__status = ' . $statusNormal . '
                    HAVING count = 0
                    LIMIT 100;';
        }
    }
}
