<?php

/**
 * i-doit
 *
 * Tree visitor class.
 *
 * @package     modules
 * @subpackage  analytics
 * @author      Dennis Stücken <dstuecken@i-doit.org>
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */
class isys_tree_visitor_analytics implements isys_tree_visitor_interface
{
    /**
     * @var  array
     */
    private $filter = [];

    /**
     * @var  array
     */
    private $pathObjects = [];

    /**
     * @var  array
     */
    private $foundItServices = [];

    /**
     * @var  isys_tree_node[]
     */
    private $itServiceRelations = [];

    /**
     * @var  array
     */
    private $recursionPreventer = [];

    /**
     * Returns an array of isys_tree_node objects.
     *
     * @return  array
     */
    public function getPathObjects()
    {
        return $this->pathObjects;
    }

    /**
     * Returns an array of isys_tree_node objects.
     *
     * @return  array
     */
    public function getItServices()
    {
        return $this->foundItServices;
    }

    /**
     * Returns all found it-service relations as an array of isys_tree_node objects.
     *
     * @return  isys_tree_node[]
     */
    public function getItServiceRelations()
    {
        return $this->itServiceRelations;
    }

    /**
     * @param  isys_tree_node  $p_node
     * @param  bool            $isItService
     *
     * @return isys_tree_node
     */
    private function remove(isys_tree_node $p_node, $isItService = null)
    {
        $l_descendants = $p_node->descendants(true);

        array_walk($l_descendants, function (isys_tree_node $p_desc) use ($isItService) {
            // Remove node.
            $p_desc->remove();

            if ($isItService) {
                // Clear it service.
                $l_data = $p_desc->get_data();
                unset($this->foundItServices[$l_data['data'][C__CMDB__GET__OBJECT]]);
            }
        });

        return $p_node;
    }

    /**
     *
     * @param   isys_tree|isys_tree_node $p_node
     *
     * @return  mixed
     */
    public function visit(isys_tree $p_node)
    {
        $l_return = [];

        if ($p_node instanceof isys_tree_node) {
            $l_data = $p_node->get_data();
            $isItService = false;

            if (isset($l_data['data']['objTypeID']) && $l_data['data']['objTypeID'] == C__OBJTYPE__IT_SERVICE) {
                if (!isset($this->foundItServices[$l_data['data'][C__CMDB__GET__OBJECT]])) {
                    $this->foundItServices[$l_data['data'][C__CMDB__GET__OBJECT]] = $l_data;
                }

                $isItService = true;

                $this->itServiceRelations[] = $p_node;
            }

            // Level cut off.
            if (isset($this->filter['level']) && $this->filter['level'] > 0 && $p_node->level() > $this->filter['level'] + 1) {
                $l_return[] = $this->remove($p_node, $isItService);
            }

            // Filter all but it services.
            if (isset($this->filter['all-but-it-services']) && $this->filter['all-but-it-services'] && !$p_node->has_children() && !$isItService) {
                // Remove all childs who are not it services.
                $l_return[] = $this->remove($p_node, $isItService);
            }

            // Filter all but the selected impact objects.
            if (isset($this->filter['all-but-impact-path-objects']) && is_array($this->filter['all-but-impact-path-objects'])) {
                if (!$p_node->has_children() && (isset($l_data['id']) && !in_array($l_data['id'], $this->filter['all-but-impact-path-objects']))) {
                    $l_return[] = $this->remove($p_node, $isItService);
                } else {
                    $this->pathObjects[] = $p_node;
                }
            }

            // Filter by relation type.
            if (isset($this->filter['relation-type'], $l_data['data']['relation']['type']) && is_array($this->filter['relation-type'])) {
                foreach ($this->filter['relation-type'] as $l_hideRelation) {
                    if (defined($l_hideRelation) && constant($l_hideRelation) == $l_data['data']['relation']['type']) {
                        // @todo What about "$l_return[] = " ...?
                        $this->remove($p_node, $isItService);
                    }
                }
            }

            // Object type cut off.
            if (isset($this->filter['object-type']) && is_array($this->filter['object-type']) && count($this->filter['object-type']) > 0 &&
                in_array($l_data['data']['objTypeID'], $this->filter['object-type'], false)) {
                $l_return[] = $this->remove($p_node, $isItService);
            }
        }

        // Iterate through childs and call accept as well.
        foreach ($p_node->get_childs() as $l_child) {
            // @see  ANALYSE-57  Skip the child, if it has already been processed by the same parent.
            if ($p_node instanceof isys_tree_node) {
                $preventionIndex = ((int)$p_node->get_data('id')) . '#' . ((int)$l_child->get_data('id'));

                if (isset($this->recursionPreventer[$preventionIndex])) {
                    continue;
                }

                $this->recursionPreventer[$preventionIndex] = true;
            }

            $l_return = array_merge($l_return, $l_child->accept($this));
        }

        // @todo  Try to avoid this, because the "show only it-service" option will not work
        if ($p_node->count() === 0) {
            if ($p_node->get_name() === 'Root') {
                return [];
            }

            $l_parent = $p_node->get_parent();

            if (method_exists($l_parent, 'get_data')) {
                $l_data = $p_node->get_parent()
                    ->get_data();

                if ($l_data['id'] == -1) {
                    return [];
                }
            }
        }

        return $l_return;
    }

    /**
     * Construct visitor with special filter. Filter structure:
     * array(
     *   'all-but-it-services' => bool,
     *   'all-but-impact-path-objects' => bool,
     *   'level' => int
     *   'relation-type' array of relation-types as string constants
     * )
     *
     * @param  array $filter
     */
    public function __construct(array $filter)
    {
        $this->filter = $filter;
    }
}
