<?php
/**
 * i-doit
 *
 * Auth: Class for i-doit authorization rules.
 *
 * @package     Modules
 * @subpackage  Analytics
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */

class isys_auth_analytics extends isys_auth implements isys_auth_interface
{
    /**
     * Container for singleton instance
     *
     * @var isys_auth_analytics
     */
    private static $m_instance = null;

    /**
     * Method for returning the available auth-methods. This will be used for the GUI.
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function get_auth_methods()
    {
        return [
            'analytics_simulation' => [
                'title'  => _L('LC__MODULE__ANALYTICS__AUTH__SIMULATION'),
                'type'   => 'analytics_simulations',
                'rights' => [isys_auth::VIEW]
            ],
            'dataquality'          => [
                'title'  => _L('LC__MODULE__ANALYTICS__AUTH__DATAQUALITY_PROFILES'),
                'type'   => 'dataquality',
                'rights' => [isys_auth::VIEW, isys_auth::EDIT, isys_auth::DELETE]
            ]
        ];
    }

    /**
     * Get ID of related module
     *
     * @return int
     */
    public function get_module_id()
    {
        return C__MODULE__ANALYTICS;
    }

    /**
     * Get title of related module
     *
     * @return string
     */
    public function get_module_title()
    {
        return 'LC__MODULE__ANALYTICS';
    }

    /**
     * Retrieve singleton instance of authorization class
     *
     * @return isys_auth_analytics
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public static function instance()
    {
        // If the DAO has not been loaded yet, we initialize it now.
        if (self::$m_dao === null)
        {
            global $g_comp_database;

            self::$m_dao = new isys_auth_dao($g_comp_database);
        }

        if (self::$m_instance === null)
        {
            self::$m_instance = new self;
        }

        return self::$m_instance;
    }

    /**
     * Method for retrieving the "parameter" in the configuration GUI. Gets called generically by "ajax()" method.
     *
     * @see     isys_module_auth->ajax_retrieve_parameter();
     *
     * @param   string  $p_method
     * @param   string  $p_param
     * @param   integer $p_counter
     * @param   boolean $p_editmode
     * @param   boolean $p_combo_param This parameter is used, when more than one box is displayed at once (category in object, ...).
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@synetics.de>
     */
    public function retrieve_parameter($p_method, $p_param, $p_counter, $p_editmode = false, $p_combo_param = false)
    {
        global $g_comp_template, $g_comp_database;

        $l_return = [
            'html'    => '',
            'method'  => $p_method,
            'param'   => $p_param,
            'counter' => $p_counter
        ];

        switch ($p_method)
        {
            case 'analytics_simulations':
                $l_simulations = isys_analytics_helper::get_all_simulations();

                foreach ($l_simulations as &$l_simulation)
                {
                    $l_simulation = $l_simulation->get_title();
                }

                $l_popup  = new isys_smarty_plugin_f_dialog();
                $l_params = [
                    'name'              => 'auth_param_form_' . $p_counter,
                    'p_bInfoIconSpacer' => 0,
                    'p_strClass'        => 'input-small',
                    'p_arData'          => $l_simulations,
                    'p_editMode'        => $p_editmode,
                    'p_strSelectedID'   => $p_param
                ];

                $l_return['html'] = isys_glob_utf8_encode($l_popup->navigation_edit($g_comp_template, $l_params));
                break;

            case 'dataquality':
                $l_data = [];
                $l_res  = isys_factory::get_instance('isys_analytics_dao_dataquality', $g_comp_database)
                    ->get_profiles();

                if (count($l_res))
                {
                    while ($l_row = $l_res->get_row())
                    {
                        $l_data[$l_row['isys_analytics_dataquality_profiles__id']] = $l_row['isys_analytics_dataquality_profiles__title'];
                    }
                }

                $l_popup  = new isys_smarty_plugin_f_dialog();
                $l_params = [
                    'name'              => 'auth_param_form_' . $p_counter,
                    'p_bInfoIconSpacer' => 0,
                    'p_strClass'        => 'input-small',
                    'p_arData'          => $l_data,
                    'p_editMode'        => $p_editmode,
                    'p_strSelectedID'   => $p_param
                ];

                $l_return['html'] = $l_popup->navigation_edit($g_comp_template, $l_params);
                break;
        }

        return $l_return;
    }

    /**
     * Method for checking, if the user is allowed to view / execute a simulation.
     *
     * @param   integer $p_right
     * @param   string  $p_param
     *
     * @return  boolean
     * @throws  isys_exception_auth
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function analytics_simulation($p_right, $p_param)
    {
        if (!$this->is_auth_active())
        {
            return true;
        }

        return $this->generic_right($p_right, 'analytics_simulation', $p_param, new isys_exception_auth(_L('LC__MODULE__ANALYTICS__AUTH__SIMULATION_EXCEPTION')));
    }

    /**
     * Method for checking, if the user is allowed to administrate dataquality profiles (add, delete, publish).
     *
     * @param   integer $p_right
     * @param   string  $p_param
     *
     * @return  boolean
     * @throws  isys_exception_auth
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function dataquality($p_right, $p_param)
    {
        if (!$this->is_auth_active())
        {
            return true;
        }

        return $this->generic_right($p_right, 'dataquality', $p_param, new isys_exception_auth(_L('LC__MODULE__ANALYTICS__AUTH__DATAQUALITY_PROFILES_EXCEPTION')));
    }
}