<?php

/**
 * i-doit
 *
 * Analytics module class.
 *
 * @package     modules
 * @subpackage  analytics
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */
class isys_module_analytics extends isys_module implements isys_module_interface, isys_module_authable
{
    // Define, if this module shall be displayed in the named menus.
    const DISPLAY_IN_MAIN_MENU   = true;
    const DISPLAY_IN_SYSTEM_MENU = false;

    /**
     * Variable which the module request class.
     *
     * @var  isys_module_request
     */
    protected $m_modreq = null;

    /**
     * Variable which holds the template component.
     *
     * @var  isys_component_template
     */
    protected $m_tpl = null;

    /**
     * Variable which holds the analytics DAO class.
     *
     * @var  isys_analytics_dao
     */
    protected $m_dao = null;

    /**
     * Variable which holds the database component.
     *
     * @var  isys_component_database
     */
    protected $m_db = null;

    /**
     * @var bool
     */
    protected static $m_licenced = false;

    /**
     * Initializes the module.
     *
     * @param   isys_module_request & $p_req
     *
     * @return  isys_module_analytics
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function init(isys_module_request &$p_req)
    {
        $this->m_modreq = $p_req;
        $this->m_db     = $p_req->get_database();
        $this->m_tpl    = $p_req->get_template()
            ->assign('tpl_dir', __DIR__ . '/reports');
        $this->m_dao    = new isys_analytics_dao($this->m_db);

        return $this;
    } // function

    /**
     * Static method for retrieving the path, to the modules templates.
     *
     * @static
     * @global  array $g_dirs
     * @return  string
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function get_tpl_www_dir()
    {
        global $g_config;

        return $g_config['www_dir'] . 'src/classes/modules/analytics/templates/';
    } // function

    /**
     * This method builds the tree for the menu.
     *
     * @param   isys_component_tree $p_tree
     * @param   boolean             $p_system_module
     * @param   integer             $p_parent
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     * @see     isys_module_cmdb->build_tree();
     */
    public function build_tree(isys_component_tree $p_tree, $p_system_module = true, $p_parent = null)
    {
        global $g_dirs;

        if ($p_system_module === false)
        {
            $l_root = $p_tree->add_node(
                C__MODULE__ANALYTICS . 0,
                $p_parent,
                _L('LC__MODULE__ANALYTICS')
            );

            $l_cnt = 1;

            $l_auth            = isys_auth_analytics::instance();
            $l_simulations     = isys_analytics_helper::get_all_simulations();
            $l_selected_report = $_GET[C__ANALYTICS__REPORT] ?: 'isys_analytics_reports_impact_simulation';

            // Here we create a menu item for each found report.
            foreach ($l_simulations as $l_classname => $l_simulation)
            {
                if (method_exists($l_simulation, 'display_in_tree') && !$l_simulation->display_in_tree())
                {
                    continue;
                } // if

                $l_last_node = $p_tree->add_node(
                    C__MODULE__ANALYTICS . $l_cnt,
                    $l_root,
                    _L($l_simulation->get_title()),
                    isys_helper_link::create_url([
                        C__GET__MODULE_ID    => C__MODULE__ANALYTICS,
                        C__GET__TREE_NODE    => C__MODULE__ANALYTICS . $l_cnt,
                        C__ANALYTICS__REPORT => $l_classname,
                    ]),
                    '',
                    $g_dirs['images'] . $l_simulation->get_icon(),
                    (int) ($l_selected_report == $l_classname),
                    '',
                    '',
                    $l_auth->is_allowed_to(isys_auth::VIEW, 'analytics_simulation/' . $l_classname)
                );

                if (method_exists($l_simulation, 'build_tree'))
                {
                    $l_simulation->build_tree($p_tree, $l_last_node);
                } // if

                $l_cnt++;
            } // foreach
        } // if
    } // function

    /**
     * Build breadcrumb navifation
     *
     * @param &$p_gets
     *
     * @return array|null
     */
    public function breadcrumb_get(&$p_gets)
    {
        $l_return          = [];
        $l_gets            = $this->m_modreq->get_gets();
        $l_selected_report = $l_gets[C__ANALYTICS__REPORT] ?: 'isys_analytics_reports_impact_simulation';

        if (!empty($l_selected_report) && class_exists($l_selected_report))
        {
            $l_report = isys_factory::get_instance($l_selected_report);

            if (method_exists($l_selected_report, 'breadcrumb_get'))
            {
                $l_return = $l_report->breadcrumb_get($l_gets);
            }
            else
            {
                $l_return[] = [
                    _L($l_report->get_title()) => [
                        C__GET__MODULE_ID    => C__MODULE__ANALYTICS,
                        C__GET__TREE_NODE    => $_GET[C__GET__TREE_NODE],
                        C__ANALYTICS__REPORT => $l_gets[C__ANALYTICS__REPORT]
                    ]
                ];
            } // if
        } // if

        return $l_return;
    } // function

    /**
     * Start method.
     *
     * @throws  isys_exception_licence
     * @throws  isys_exception_general
     * @return  isys_module_analytics
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function start()
    {
        // Check for the licence.
        if (!isys_module_analytics::is_licenced())
        {
            throw new isys_exception_licence(_L('LC__LICENCE__NO_MODULE_LICENCE', [_L('LC__MODULE__ANALYTICS')]), 1);
        } // if

        $l_selected_report = $_GET[C__ANALYTICS__REPORT] ?: 'isys_analytics_reports_impact_simulation';

        $this->check_for_ajax($l_selected_report);

        // Build the module tree, but only if we are not in the system-module.
        if ($_GET[C__GET__MODULE_ID] != C__MODULE__SYSTEM)
        {
            $l_tree = isys_module_request::get_instance()
                ->get_menutree();

            $this->build_tree($l_tree, false, -1);

            $this->m_tpl->assign("menu_tree", $l_tree->process($_GET[C__GET__TREE_NODE]));
        } // if

        if (class_exists($l_selected_report))
        {
            isys_auth_analytics::instance()
                ->check(isys_auth::VIEW, 'analytics_simulation/' . $l_selected_report);

            isys_factory::get_instance($l_selected_report)
                ->start();
        }
        else
        {
            throw new isys_exception_general(_L('LC__MODULE__ANALYTICS__EXCEPTION__NO_REPORT_FOUND', $l_selected_report));
        } // if

        return $this;
    } // function

    /**
     * This method will check, if the current request is a AJAX request and (if so) process the necessary logic.
     *
     * @param   string $p_class
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function check_for_ajax($p_class)
    {
        if ($_SERVER['HTTP_X_REQUESTED_WITH'] == 'XMLHttpRequest')
        {
            // We set the header information because we don't accept anything than JSON.
            header('Content-Type: application/json');

            $l_return = [
                'success' => true,
                'data'    => null,
                'message' => null
            ];

            try
            {
                if (class_exists($p_class))
                {
                    $l_return['data'] = isys_factory::get_instance($p_class)->ajax_request();
                }
                else
                {
                    throw new isys_exception_general(_L('LC__MODULE__ANALYTICS__EXCEPTION__NO_CLASS_FOUND', $p_class));
                } // if
            }
            catch (isys_exception $e)
            {
                $l_return['success'] = false;
                $l_return['message'] = $e->getMessage();
            } // try

            echo isys_format_json::encode($l_return);
            die;
        } // if
    } // function

    /**
     * Method for adding links to the "sticky" category bar.
     *
     * @param  isys_component_template $p_tpl
     * @param  string                  $p_tpl_var
     * @param  integer                 $p_obj_id
     * @param  integer                 $p_obj_type_id
     */
    public static function process_menu_tree_links($p_tpl, $p_tpl_var, $p_obj_id, $p_obj_type_id)
    {
        global $g_dirs;

        $l_link_data = [
            'title' => _L('LC__MODULE__ANALYTICS'),
            'link'  => isys_helper_link::create_url([
                C__GET__MODULE_ID    => C__MODULE__ANALYTICS,
                C__ANALYTICS__REPORT => 'isys_analytics_reports_impact_simulation',
                C__CMDB__GET__OBJECT => $p_obj_id
            ]),
            'icon'  => $g_dirs['images'] . 'icons/silk/chart_pie.png'
        ];

        $p_tpl->append($p_tpl_var, ['analytics' => $l_link_data], true);
    } // function

    /**
     * Get related auth class for module
     *
     * @author Selcuk Kekec <skekec@i-doit.com>
     * @return isys_auth
     */
    public static function get_auth()
    {
        return isys_auth_analytics::instance();
    } // function
} // class