<?php

/**
 * i-doit
 *
 * Dataquality report.
 *
 * @package     modules
 * @subpackage  analytics
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */
class isys_analytics_reports_dataquality extends isys_analytics_reports
{
    /**
     * This constant holds the name of the report.
     *
     * @var  string
     */
    const TITLE = 'LC__MODULE__ANALYTICS__DATA_QUALITY';

    /**
     * This constant holds the icon-path of the report.
     *
     * @var  string
     */
    const ICON = 'icons/silk/monitor.png';

    /**
     * @var  isys_component_template
     */
    private $m_tpl = null;

    /**
     * @var  isys_component_database
     */
    protected $m_db = null;

    /**
     * @var  isys_analytics_dao_dataquality
     */
    private $m_dao = null;

    /**
     * Configuration array, filled by isys_usersettings.
     *
     * @var  array
     */
    private $m_config = [];

    /**
     * This variable holds the currently active profile ID.
     *
     * @var  integer
     */
    private $m_current_profile = null;

    /**
     * This array holds blacklisted categories, which shall not be used for the data quality calculation.
     *
     * @var  array
     */
    private $m_category_blacklist = [
        'g' => [
            C__CATG__LOGBOOK,
            C__CATG__IMAGE,
            C__CATG__OVERVIEW,
            C__CATG__OBJECT,
            C__CATG__CLUSTER_ROOT,
            C__CATG__CLUSTER,
            C__CATG__CLUSTER_SHARED_STORAGE,
            C__CATG__CLUSTER_VITALITY,
            C__CATG__VIRTUAL_HOST_ROOT,
            C__CATG__CLUSTER_SHARED_VIRTUAL_SWITCH,
            C__CATG__ITS_LOGBOOK,
            C__CATG__OBJECT_VITALITY,
            C__CATG__RELATION,
            C__CATG__NETWORK_PORT_OVERVIEW,
            C__CATG__VIRTUAL_TICKETS,
            C__CATG__RACK_VIEW,
            C__CATG__LIVESTATUS,
            C__CATG__NET_CONNECTIONS_FOLDER,
            C__CATG__NDO,
            C__CATG__VIRTUAL_AUTH
        ],
        's' => [
            C__CATS__PRT,
            C__CATS__FILE,
            C__CATS__LICENCE
        ]
    ];

    /**
     * This array holds blacklisted object types, which shall not be displayed in the frontend.
     *
     * @var  array
     */
    private $m_objtype_blacklist = [
        C__OBJTYPE__RELATION,
        C__OBJTYPE__PARALLEL_RELATION
    ];

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->m_db = isys_application::instance()->container->get('database');
        $this->m_tpl = isys_application::instance()->container->get('template');
        $this->m_dao = isys_analytics_dao_dataquality::instance($this->m_db);
        $this->m_current_profile = $_GET[C__ANALYTICS__REPORT_PROFILE];

        $l_config = [];
        $l_config_res = $this->m_dao->get_profiles(null, isys_application::instance()->container->get('session')->get_user_id());

        if (count($l_config_res)) {
            while ($l_config_row = $l_config_res->get_row()) {
                $l_config_row['isys_analytics_dataquality_profiles__data'] = isys_format_json::decode($l_config_row['isys_analytics_dataquality_profiles__data']);

                $l_config[$l_config_row['isys_analytics_dataquality_profiles__id']] = $l_config_row;
            }
        }

        $this->m_config = $l_config;
    }

    /**
     * This method builds the tree for the menu.
     *
     * @param   isys_component_tree $p_tree
     * @param   integer             $p_parent
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function build_tree(isys_component_tree &$p_tree, $p_parent)
    {
        global $g_dirs;

        $l_cnt = 100;

        foreach ($this->m_config as $l_config) {
            $p_tree->add_node(C__MODULE__ANALYTICS . $l_cnt, $p_parent, $l_config['isys_analytics_dataquality_profiles__title'], isys_helper_link::create_url([
                C__GET__MODULE_ID            => C__MODULE__ANALYTICS,
                C__GET__TREE_NODE            => C__MODULE__ANALYTICS . $l_cnt,
                C__ANALYTICS__REPORT         => __CLASS__,
                C__ANALYTICS__REPORT_PROFILE => $l_config['isys_analytics_dataquality_profiles__id']
            ]), '', $g_dirs['images'] . 'icons/silk/table_multiple.png', (int)($this->m_current_profile == $l_config['isys_analytics_dataquality_profiles__id']));

            $l_cnt++;
        }
    }

    /**
     * Build breadcrumb navifation
     *
     * @param  $p_gets
     *
     * @return array
     */
    public function breadcrumb_get($p_gets)
    {
        $l_return = [
            [
                _L($this->get_title()) => [
                    C__GET__MODULE_ID    => C__MODULE__ANALYTICS,
                    C__GET__TREE_NODE    => $_GET[C__GET__TREE_NODE],
                    C__ANALYTICS__REPORT => $p_gets[C__ANALYTICS__REPORT]
                ]
            ]
        ];

        if ($this->m_current_profile > 0) {
            $l_return[] = [
                $this->m_config[$this->m_current_profile]['isys_analytics_dataquality_profiles__title'] => [
                    C__GET__MODULE_ID            => C__MODULE__ANALYTICS,
                    C__GET__TREE_NODE            => $_GET[C__GET__TREE_NODE],
                    C__ANALYTICS__REPORT         => $p_gets[C__ANALYTICS__REPORT],
                    C__ANALYTICS__REPORT_PROFILE => $this->m_current_profile
                ]
            ];
        }

        return $l_return;
    }

    /**
     * Method for preparing the visual report view output.
     *
     * @return  isys_analytics_reports_impact_simulation
     */
    public function start()
    {
        global $g_dirs;

        $l_get = $_GET;
        $l_object_types = [];

        // Remove some unnecessary parameters for the Ajax URL.
        unset($l_get[C__GET__MAIN_MENU__NAVIGATION_ID], $l_get[C__CMDB__GET__EDITMODE], $l_get[C__GET__TREE_NODE]);

        $l_object_type_res = $this->m_dao->get_object_types();

        if (count($l_object_type_res)) {
            $l_obj_types = array_keys($this->m_config[$this->m_current_profile]['isys_analytics_dataquality_profiles__data'] ?: []);

            while ($l_obj_type = $l_object_type_res->get_row()) {
                if (in_array($l_obj_type['isys_obj_type__id'], $this->m_objtype_blacklist) ||
                    (count($l_obj_types) && !in_array($l_obj_type['isys_obj_type__const'], $l_obj_types))) {
                    continue;
                }

                $l_icon = $g_dirs['images'] . 'icons/silk/page_white.png';

                if (!empty($l_obj_type['isys_obj_type__icon'])) {
                    if (strpos($l_obj_type['isys_obj_type__icon'], '/') !== false) {
                        $l_icon = $l_obj_type['isys_obj_type__icon'];
                    } else {
                        $l_icon = $g_dirs["images"] . "tree/" . $l_obj_type['isys_obj_type__icon'];
                    }
                }

                $l_obj_sql = 'SELECT COUNT(isys_obj__id) AS count
					FROM isys_obj
					WHERE isys_obj__isys_obj_type__id = ' . $this->m_dao->convert_sql_id($l_obj_type['isys_obj_type__id']) . '
					AND isys_obj__status = ' . $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . ';';

                $l_obj_count = $this->m_dao->retrieve($l_obj_sql)
                    ->get_row_value('count');

                $l_object_types[_L($l_obj_type['isys_obj_type__title'])] = [
                    'id'               => $l_obj_type['isys_obj_type__id'],
                    'const'            => $l_obj_type['isys_obj_type__const'],
                    'title'            => _L($l_obj_type['isys_obj_type__title']),
                    'icon'             => $l_icon,
                    'obj_count'        => $l_obj_count,
                    'obj_count_string' => $l_obj_count . ' ' . (($l_obj_count == 1) ? _L('LC_UNIVERSAL__OBJECT') : _L('LC_UNIVERSAL__OBJECTS'))
                ];
            }
        }

        $l_rules['C__ANALYTICS__PROFILE_NAME']['p_strValue'] = $this->m_config[$this->m_current_profile]['isys_analytics_dataquality_profiles__title'];
        $l_rules['C__ANALYTICS__PROFILE_USERSPECIFIC']['p_arData'] = get_smarty_arr_YES_NO();

        $l_save_overlay = [
            [
                "title"   => _L('LC__MODULE__ANALYTICS__DATA_QUALITY__SAVE_AS_NEW_PROFILE'),
                "icon"    => "icons/silk/page_save.png",
                "href"    => "javascript:",
                "onclick" => "",
                "id"      => "navbar_item_C__NAVMODE__SAVE_AS",
            ]
        ];

        $l_navbar = isys_component_template_navbar::getInstance();
        $l_auth = isys_auth_analytics::instance();

        if ($this->m_current_profile > 0) {
            $l_navbar->append_button('LC__MODULE__ANALYTICS__DATA_QUALITY__RESET_VIEW', 'reset-object-types', [
                    'tooltip'    => 'LC__MODULE__ANALYTICS__DATA_QUALITY__RESET_VIEW',
                    'icon'       => 'icons/silk/arrow_refresh.png',
                    'js_onclick' => ';',
                    'navmode'    => 'reset'
                ]);

            if ($l_auth->is_allowed_to(isys_auth::EDIT, 'DATAQUALITY/' . $this->m_current_profile)) {
                $l_navbar->set_active(true, C__NAVBAR_BUTTON__SAVE)
                    ->set_overlay($l_save_overlay, C__NAVBAR_BUTTON__SAVE);

                if ($this->m_config[$this->m_current_profile]['isys_analytics_dataquality_profiles__isys_obj__id'] > 0) {
                    $l_navbar->append_button('LC__MODULE__ANALYTICS__DATA_QUALITY__MAKE_PUBLIC', 'publish-profile', [
                            'tooltip'    => 'LC__MODULE__ANALYTICS__DATA_QUALITY__MAKE_PUBLIC',
                            'icon'       => 'icons/silk/group.png',
                            'js_onclick' => ';',
                            'navmode'    => 'publish'
                        ]);
                }
            }

            if ($l_auth->is_allowed_to(isys_auth::DELETE, 'DATAQUALITY/' . $this->m_current_profile)) {
                $l_navbar->append_button('LC__MODULE__ANALYTICS__DATA_QUALITY__PURGE', 'purge-profile', [
                        'tooltip'    => 'LC__MODULE__ANALYTICS__DATA_QUALITY__PURGE',
                        'icon'       => 'icons/silk/page_delete.png',
                        'js_onclick' => ';',
                        'navmode'    => 'purge_profile'
                    ]);
            }
        } else {
            if ($l_auth->is_allowed_to(isys_auth::EDIT, 'DATAQUALITY/' . isys_auth::WILDCHAR)) {
                $l_navbar->set_active(true, C__NAVBAR_BUTTON__SAVE);
            } // if
        } // if

        ksort($l_object_types);

        $l_categories = [];

        $l_category_data = $this->m_dao->get_all_categories();

        foreach ($l_category_data[C__CMDB__CATEGORY__TYPE_GLOBAL] as $l_cat) {
            $l_categories['g_' . $l_cat['id']] = [
                'title' => _L($l_cat['title']),
                'color' => '#' . substr(md5($l_cat['title']), 0, 6),
                'const' => $l_cat['const']
            ];
        }

        foreach ($l_category_data[C__CMDB__CATEGORY__TYPE_SPECIFIC] as $l_cat) {
            $l_categories['s_' . $l_cat['id']] = [
                'title' => _L($l_cat['title']),
                'color' => '#' . substr(md5($l_cat['title']), 0, 6),
                'const' => $l_cat['const']
            ];
        }

        foreach ($l_category_data[C__CMDB__CATEGORY__TYPE_CUSTOM] as $l_cat) {
            $l_categories['g_custom_' . $l_cat['id']] = [
                'title' => _L($l_cat['title']),
                'color' => '#' . substr(md5($l_cat['title']), 0, 6),
                'const' => $l_cat['const']
            ];
        }

        $this->m_tpl->activate_editmode()
            ->assign('profile_id', $this->m_current_profile)
            ->assign('ajax_url', isys_helper_link::create_url($l_get))
            ->assign('object_types', $l_object_types)
            ->assign('categories', $l_categories)
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', __DIR__ . DS . get_class() . '_view.tpl');

        return $this;
    }

    /**
     * This method will be called by the framework, to process ajax requests.
     *
     * @param   mixed $p_data
     *
     * @return  mixed
     * @throws  isys_exception_database
     */
    public function ajax_request($p_data = [])
    {
        $session = isys_application::instance()->container->get('session');

        $l_profile_id = $_GET[C__ANALYTICS__REPORT_PROFILE] ?: null;

        switch ($_GET['func']) {
            default:
            case 'load-category-data':
                $session->write_close();

                return $this->load_category_data($_POST['obj_type_id'], !!$_POST['reset_categories']);

            case 'load-objects-without-data':
                $session->write_close();

                return $this->load_objects_without_data($_POST['obj_type_id'], $_POST['category'], $_POST['category_type']);

            case 'save-profile':
                $this->m_dao->save_profile($l_profile_id, ['data' => $_POST['data']]);

                return null;

            case 'save-profile-as':
                $l_data = [
                    'data'         => $_POST['data'],
                    'title'        => $_POST['title'],
                    'isys_obj__id' => $session->get_user_id(),
                ];

                $session->write_close();
                $this->m_dao->save_profile(null, $l_data);

                return isys_helper_link::create_url([
                    C__GET__MODULE_ID            => C__MODULE__ANALYTICS,
                    C__GET__TREE_NODE            => $_GET[C__GET__TREE_NODE],
                    C__ANALYTICS__REPORT         => __CLASS__,
                    C__ANALYTICS__REPORT_PROFILE => $this->m_dao->get_last_insert_id()
                ]);

            case 'publish-profile':
                $this->m_dao->save_profile($l_profile_id, ['isys_obj__id' => null]);

                return null;

            case 'delete-profile':
                $this->m_dao->delete_profile($l_profile_id);

                return isys_helper_link::create_url([
                    C__GET__MODULE_ID    => C__MODULE__ANALYTICS,
                    C__GET__TREE_NODE    => $_GET[C__GET__TREE_NODE],
                    C__ANALYTICS__REPORT => __CLASS__
                ]);
        }
    }

    /**
     * This function will retrieve the types and some "simple" statistics of the given objects.
     *
     * @param  integer $p_obj_type_id
     * @param  boolean $p_reset
     *
     * @return array
     * @throws isys_exception_database
     */
    private function load_category_data($p_obj_type_id, $p_reset = false)
    {
        $l_data = [];
        $l_sql_part = [];
        $l_sql_not_compatible = [
            'isys_catg_nagios_refs_services_list'
        ];

        $l_obj_type = $this->m_dao->get_object_type($p_obj_type_id);
        $l_category_res = $this->m_dao->get_catg_by_obj_type($p_obj_type_id);

        $l_categories = $this->m_config[$this->m_current_profile]['isys_analytics_dataquality_profiles__data'][$l_obj_type['isys_obj_type__const']] ?: [];

        if ($p_reset) {
            $l_categories = [];
        }

        // Collect additional categories, which we don't want to display:
        $l_category_blacklist = $this->m_dao->get_all_categories([isys_cmdb_dao_category::TYPE_VIEW, isys_cmdb_dao_category::TYPE_REAR]);

        if (isset($l_category_blacklist[C__CMDB__CATEGORY__TYPE_GLOBAL]) && count($l_category_blacklist[C__CMDB__CATEGORY__TYPE_GLOBAL])) {
            foreach ($l_category_blacklist[C__CMDB__CATEGORY__TYPE_GLOBAL] as $l_g_category) {
                $this->m_category_blacklist['g'][] = $l_g_category['const'];
            }
        }

        if (isset($l_category_blacklist[C__CMDB__CATEGORY__TYPE_SPECIFIC]) && count($l_category_blacklist[C__CMDB__CATEGORY__TYPE_SPECIFIC])) {
            foreach ($l_category_blacklist[C__CMDB__CATEGORY__TYPE_SPECIFIC] as $l_s_category) {
                $this->m_category_blacklist['s'][] = $l_s_category['const'];
            }
        }

        if (count($l_category_res)) {
            while ($l_category = $l_category_res->get_row()) {
                if (count($l_categories) > 0 && !in_array($l_category['isysgui_catg__const'], $l_categories) && !$p_reset) {
                    continue;
                }

                if (!class_exists($l_category['isysgui_catg__class_name']) || in_array($l_category['isysgui_catg__id'], $this->m_category_blacklist['g'])) {
                    continue;
                }

                $l_sql_table = $l_category['isysgui_catg__source_table'] . '_list';

                if (!in_array($l_sql_table, $l_sql_not_compatible)) {
                    $l_sql_part[] = '(SELECT COUNT(' . $l_sql_table . '__id) 
                        FROM ' . $l_sql_table . ' 
                        WHERE ' . $l_sql_table . '__isys_obj__id = isys_obj__id
                        AND ' . $l_sql_table . '__status = ' . $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . ') as count_g_' . $l_category['isysgui_catg__id'];
                }
            }
        }

        $l_category_res = $this->m_dao->gui_get_cats_by_objtype_id($p_obj_type_id);

        if (count($l_category_res)) {
            while ($l_category = $l_category_res->get_row()) {
                if (count($l_categories) > 0 && !in_array($l_category['isysgui_cats__const'], $l_categories)) {
                    continue;
                }

                if (!class_exists($l_category['isysgui_cats__class_name']) || in_array($l_category['isysgui_cats__id'], $this->m_category_blacklist['s'])) {
                    continue;
                }

                $l_sql_table = $l_category['isysgui_cats__source_table'];

                if (!in_array($l_sql_table, $l_sql_not_compatible) && !isset($l_sql_part[$l_sql_table])) {
                    $l_sql_part[$l_sql_table] = '(SELECT COUNT(' . $l_sql_table . '__id) 
                        FROM ' . $l_sql_table . ' 
                        WHERE ' . $l_sql_table . '__isys_obj__id = isys_obj__id
                        AND ' . $l_sql_table . '__status = ' . $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . ') as count_s_' . $l_category['isysgui_cats__id'];
                }
            }
        }

        $l_category_res = $this->m_dao->gui_get_catg_custom_by_objtype_id($p_obj_type_id);

        if (count($l_category_res)) {
            while ($l_category = $l_category_res->get_row()) {
                if (count($l_categories) > 0 && !in_array($l_category['isysgui_catg_custom__const'], $l_categories)) {
                    continue;
                }

                $l_sql_table = 'isys_catg_custom_fields_list_' . $l_category['isysgui_catg_custom__id'];

                if (!isset($l_sql_part[$l_sql_table])) {
                    $l_sql_part[$l_sql_table] = '(SELECT COUNT(isys_catg_custom_fields_list__id) 
                        FROM isys_catg_custom_fields_list 
                        WHERE isys_catg_custom_fields_list__isys_obj__id = isys_obj__id
                        AND isys_catg_custom_fields_list__isysgui_catg_custom__id = ' . $this->m_dao->convert_sql_id($l_category['isysgui_catg_custom__id']) . '
                        AND isys_catg_custom_fields_list__field_type = "commentary"
                        AND isys_catg_custom_fields_list__status = ' . $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . ') as count_g_custom_' .
                        $l_category['isysgui_catg_custom__id'];
                }
            }
        }

        $l_sql = 'SELECT ' . implode(', ', $l_sql_part) . '
            FROM isys_obj
            WHERE isys_obj__isys_obj_type__id = ' . $this->m_dao->convert_sql_id($p_obj_type_id) . '
            AND isys_obj__status = ' . $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL);

        $l_result = $this->m_dao->retrieve($l_sql);

        while ($l_row = $l_result->get_row()) {
            foreach ($l_row as $l_cat => $l_count) {
                $l_cat = substr($l_cat, 6);

                if (!isset($l_data[$l_cat])) {
                    $l_data[$l_cat] = 0;
                }

                if ($l_count > 0) {
                    $l_data[$l_cat]++;
                }
            }
        }

        return [
            'objects' => $l_result->count(),
            'data'    => $l_data
        ];
    }

    /**
     * Method for loading objects of a given type that are missing data in the given category. 
     * 
     * @param  integer $objectTypeId
     * @param  string  $categoryConstant
     * @param  string  $categoryType
     *
     * @return array
     * @throws isys_exception_cmdb
     * @throws isys_exception_database
     */
    private function load_objects_without_data($objectTypeId, $categoryConstant, $categoryType)
    {
        $return = [];
        $countCondition = null;

        switch ($categoryType) {
            case 'g':
            default:
                $categoryRow = $this->m_dao->get_catg_by_const($categoryConstant)->get_row();
                $tableName = $categoryRow['isysgui_catg__source_table'] . '_list';

                $link = isys_helper_link::create_catg_url([
                    C__CMDB__GET__OBJECT => '%objID%',
                    C__CMDB__GET__CATG   => $categoryRow['isysgui_catg__id']
                ]);

                break;

            case 's':
                $categoryRow = $this->m_dao->get_cats_by_const($categoryConstant)->get_row();
                $tableName = $categoryRow['isysgui_cats__source_table'];

                $link = isys_helper_link::create_cats_url([
                    C__CMDB__GET__OBJECT => '%objID%',
                    C__CMDB__GET__CATS   => $categoryRow['isysgui_cats__id']
                ]);

                break;

            case 'g_custom':
                $customCategoryId = constant($categoryConstant);
                
                $tableName = 'isys_catg_custom_fields_list';
                $countCondition = 'AND isys_catg_custom_fields_list__isysgui_catg_custom__id = ' . $this->m_dao->convert_sql_id($customCategoryId);

                $link = isys_helper_link::create_catg_custom_url([
                    C__CMDB__GET__OBJECT      => '%objID%',
                    C__CMDB__GET__CATG_CUSTOM => $customCategoryId
                ]);

                break;
        }

        $sql = 'SELECT isys_obj__id, isys_obj__title, isys_obj__sysid,
            (SELECT COUNT(' . $tableName . '__id) FROM ' . $tableName . ' WHERE ' . $tableName . '__isys_obj__id = isys_obj__id ' . $countCondition . ') as count
            FROM isys_obj
            WHERE isys_obj__isys_obj_type__id = ' . $this->m_dao->convert_sql_id($objectTypeId) . '
            AND isys_obj__status = ' . $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
            HAVING count = 0
            LIMIT 100;';

        $l_result = $this->m_dao->retrieve($sql);

        while ($row = $l_result->get_row()) {
            $return[] = [
                'id'    => $row['isys_obj__id'],
                'title' => _L($row['isys_obj__title']),
                'sysid' => $row['isys_obj__sysid'],
                'link'  => str_replace('%objID%', $row['isys_obj__id'], $link)
            ];
        }

        return $return;
    }
}