<?php

/**
 * i-doit
 *
 * Object catalog report.
 *
 * @package     modules
 * @subpackage  analytics
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */
class isys_analytics_reports_obj_catalog extends isys_analytics_reports
{
    /**
     * This constant holds the name of the report.
     *
     * @var  string
     */
    const TITLE = 'LC__MODULE__ANALYTICS__OBJECT_CATALOG';

    /**
     * This constant holds the icon-path of the report.
     *
     * @var  string
     */
    const ICON = 'icons/silk/monitor.png';

    /**
     * @var  isys_component_template
     */
    private $m_tpl = null;

    /**
     * @var  isys_component_database
     */
    private $m_db = null;

    /**
     * @var  isys_analytics_dao
     */
    private $m_dao = null;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->m_db  = isys_application::instance()->database;
        $this->m_tpl = isys_application::instance()->template;
        $this->m_dao = isys_factory::get_instance('isys_analytics_dao', $this->m_db);
    } // function

    /**
     * Method for preparing the visual report view output.
     *
     * @return  isys_analytics_reports_impact_simulation
     */
    public function start()
    {
        global $index_includes;

        $l_get     = $_GET;
        $l_filters = $l_itservices = [];

        $l_rules = [
            'obj_catalog_selection' => [
                isys_popup_browser_object_ng::C__CAT_FILTER => 'C__CATG__SERVICE;C__CATG__IT_SERVICE_COMPONENTS;C__CATG__ITS_LOGBOOK;C__CATG__IT_SERVICE_RELATIONS;C__CATG__ITS_TYPE'
            ]
        ];

        // Remove some unnecessary parameters for the Ajax URL.
        unset($l_get[C__GET__MAIN_MENU__NAVIGATION_ID], $l_get[C__CMDB__GET__EDITMODE], $l_get[C__GET__TREE_NODE]);

        $l_itservice_filter_url = isys_helper_link::create_url([
            C__GET__MODULE_ID     => C__MODULE__ITSERVICE,
            C__GET__SETTINGS_PAGE => 'filter-config'
        ]);

        $l_itservice_filters = isys_itservice_dao_filter_config::instance($this->m_db)->get_data();

        if (is_array($l_itservice_filters) && count($l_itservice_filters))
        {
            foreach ($l_itservice_filters as $l_filter)
            {
                $l_filters[$l_filter['isys_itservice_filter_config__id']] = $l_filter['isys_itservice_filter_config__title'];
            } // foreach

            $l_rules['impact_itservice_filter']['p_arData'] = $l_filters;
        } // if

        $l_rules['obj_catalog_attributes']['provide']      = C__PROPERTY__PROVIDES__REPORT;
        $l_rules['obj_catalog_attributes']['preselection'] = isys_format_json::encode([['g' => ['C__CATG__GLOBAL' => ['title']]]]);

        isys_component_template_navbar::getInstance()
            ->append_button(
                'LC__MODULE__ANALYTICS__OBJECT_CATALOG__LOAD_OBJECT_DATA',
                'start-simulation',
                [
                    'tooltip'    => 'LC__MODULE__ANALYTICS__OBJECT_CATALOG__LOAD_OBJECT_DATA',
                    'icon'       => 'icons/silk/cog.png',
                    'js_onclick' => ';',
                    'navmode'    => 'simulation',
                ]
            );

        $this->m_tpl->activate_editmode()
            ->assign('filter_description', _L('LC__MODULE__ANALYTICS__FILTERS_DESCRIPTION', $l_itservice_filter_url))
            ->assign('ajax_url', isys_helper_link::create_url($l_get))
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules);

        $index_includes['contentbottomcontent'] = __DIR__ . DS . get_class() . '_view.tpl';

        return $this;
    } // function

    /**
     * This method will be called by the framework, to process ajax requests.
     *
     * @param   mixed $p_data
     *
     * @return  mixed
     */
    public function ajax_request($p_data = [])
    {
        switch ($_GET['func'])
        {
            default:
            case 'start-simulation':
                return $this->start_simulation((int) $_POST['itservice_obj'], (int) $_POST['filter']);

            case 'retrieve-attributes':
                return $this->retrieve_attributes(isys_format_json::decode($_POST['obj_ids']), isys_format_json::decode($_POST['attributes']));
        } // switch
    } // function

    /**
     * This method will trigger the simulation algorithm!
     *
     * @param   integer $p_itservice_obj
     * @param   integer $p_filter
     *
     * @throws  isys_exception_general
     * @return  array
     */
    private function start_simulation($p_itservice_obj, $p_filter = null)
    {
        if (empty($p_itservice_obj))
        {
            throw new isys_exception_general(_L('LC__MODULE__ANALYTICS__NOTIFY__NO_IMPACT_OBJECT'));
        } // if

        $l_return = $l_filter = $l_it_service_mapping = [];

        // Every simulation shall begin with an empty cache.
        $this->m_dao->get_cache()->clear();

        // Prepare the filters.
        if ($p_filter > 0)
        {
            $l_filter = isys_factory::get_instance('isys_itservice_dao_filter_config', $this->m_db)->get_data($p_filter);

            $l_filter = $l_filter['formatted__data'];
        } // if

        if (empty($l_filter['level']))
        {
            $l_filter['level'] = 11;
        } // if

        // Add root node.
        $l_root_node = $this->m_dao->format_node($p_itservice_obj);
        $l_tree      = new isys_tree($l_root_node);

        $this->m_dao->relation_walk(
            $l_root_node,
            $l_filter['priority'],
            $l_filter['relation-type'] ?: [],
            $l_filter['object-type'] ?: []
        );

        $l_root_node->set_subnodes($l_root_node->count());

        // Prepare tree.
        if (isset($l_tree) && $l_tree instanceof isys_tree)
        {
            // Filter tree using the visitor pattern.
            $l_visitor = new isys_tree_visitor_analytics($l_filter);

            while ($l_tree->accept($l_visitor))
            {
                // filtering..
            } // while

            foreach ($l_tree->all_nodes() as $l_node)
            {
                $l_return[] = $l_node->get_id();
            } // foreach

            return array_values(array_unique($l_return, SORT_NUMERIC));
        } // if

        return [];
    } // function

    /**
     * This function will retrieve the selected attributes of the given objects.
     *
     * @param   array $p_obj_ids
     * @param   array $p_attributes
     *
     * @return  array
     */
    private function retrieve_attributes(array $p_obj_ids, array $p_attributes)
    {
        if (!count($p_obj_ids))
        {
            // We don't want an "error" message, so we can't use an exception.
            isys_notify::warning(_L('LC__MODULE__ANALYTICS__NOTIFY__EMPTY_ATTRIBUTE_RESULT'), ['life' => 10]);

            return [];
        } // if

        $l_return = [
            'content'      => [],
            'content_sums' => ['obj_count' => 0],
            'header'       => []
        ];

        if (count($p_attributes) > 0)
        {
            $l_row_data = $l_formattables = [];

            foreach ($p_attributes as $l_cat_types)
            {
                foreach ($l_cat_types as $l_cat_type => $l_categories)
                {
                    foreach ($l_categories as $l_category => $l_attributes)
                    {
                        if (!defined($l_category))
                        {
                            isys_notify::warning(_L('LC__MODULE__ANALYTICS__NOTIFY__CAT_CONST_NOT_FOUND', [$l_category]), ['life' => 8]);
                        } // if

                        switch ($l_cat_type)
                        {
                            default:
                            case 'g':
                                $l_cat_type_id = C__CMDB__CATEGORY__TYPE_GLOBAL;
                                break;
                            case 's':
                                $l_cat_type_id = C__CMDB__CATEGORY__TYPE_SPECIFIC;
                                break;
                            case 'g_custom':
                                $l_cat_type_id = C__CMDB__CATEGORY__TYPE_CUSTOM;
                                break;
                        } // switch

                        $l_dao = isys_factory_cmdb_category_dao::get_instance_by_id($l_cat_type_id, constant($l_category), $this->m_db);

                        $l_properties = $l_dao->get_properties();
                        $l_res        = $l_dao->get_data(null, $p_obj_ids);

                        if (count($l_res))
                        {
                            while ($l_row = $l_res->get_row())
                            {
                                $l_obj_id = $l_row['isys_obj__id'];

                                foreach ($l_attributes as $l_attribute)
                                {
                                    $l_data_key      = $l_category . '::' . $l_attribute;
                                    $l_property      = $l_properties[$l_attribute];
                                    $l_current_value = null;

                                    if (!isset($l_return['header'][$l_data_key]))
                                    {
                                        $l_return['header'][$l_data_key] = _L($l_property[C__PROPERTY__INFO][C__PROPERTY__INFO__TITLE]);
                                    } // if

                                    if (class_exists($l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][0]))
                                    {
                                        $l_helper = new $l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][0](
                                            $l_row, $this->m_db, $l_property[C__PROPERTY__DATA], $l_property[C__PROPERTY__FORMAT], $l_property[C__PROPERTY__UI]
                                        );

                                        if (!empty($l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__UNIT]))
                                        {
                                            $l_unit_key = $l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__UNIT];

                                            if (method_exists($l_helper, 'set_unit_const'))
                                            {
                                                if (isset($l_properties[$l_unit_key][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]))
                                                {
                                                    $l_const = $l_row[$l_properties[$l_unit_key][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]];
                                                }
                                                else
                                                {
                                                    $l_const = $l_row[$l_properties[$l_unit_key][C__PROPERTY__DATA][C__PROPERTY__DATA__REFERENCES][0] . '__const'];
                                                } // if

                                                $l_helper->set_unit_const($l_const);
                                            } // if
                                        } // if

                                        if (method_exists($l_helper, $l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1]))
                                        {
                                            if (isset($l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]) &&
                                                array_key_exists($l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS], $l_row)
                                            )
                                            {
                                                $l_current_value = $l_helper->$l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1](
                                                    $l_row[$l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]]
                                                );
                                            }
                                            else
                                            {
                                                $l_current_value = $l_helper->$l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1](
                                                    $l_row[$l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD]]
                                                );
                                            } // if

                                            if (is_array($l_current_value) && isset($l_current_value['title']))
                                            {
                                                $l_current_value = $l_row_data[$l_obj_id][$l_data_key][] = $l_current_value['title'];
                                            }
                                            else
                                            {
                                                $l_row_data[$l_obj_id][$l_data_key][] = $l_current_value;
                                            } // if
                                        } // if
                                    }
                                    else
                                    {
                                        if (isset($l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]) &&
                                            array_key_exists($l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS], $l_row)
                                        )
                                        {
                                            $l_current_value                      = $l_row[$l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]];
                                            $l_row_data[$l_obj_id][$l_data_key][] = $l_current_value;
                                        }
                                        else
                                        {
                                            $l_current_value                      = $l_row[$l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD]];
                                            $l_row_data[$l_obj_id][$l_data_key][] = $l_current_value;
                                        } // if
                                    } // if

                                    if (!empty($l_current_value))
                                    {
                                        $l_is_dialog      = in_array(
                                            $l_property[C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE],
                                            [C__PROPERTY__INFO__TYPE__DIALOG, C__PROPERTY__INFO__TYPE__DIALOG_PLUS]
                                        );
                                        $l_is_numeric     = in_array(
                                            $l_property[C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE],
                                            [C__PROPERTY__INFO__TYPE__FLOAT, C__PROPERTY__INFO__TYPE__DOUBLE, C__PROPERTY__INFO__TYPE__INT]
                                        );
                                        $l_is_formattable = in_array($l_property[C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE], [C__PROPERTY__INFO__TYPE__MONEY]) ||
                                            (isset($l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK]) &&
                                                count($l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK]) >= 2);

                                        if ($l_is_dialog)
                                        {
                                            if (!isset($l_return['content_sums'][$l_data_key]))
                                            {
                                                $l_return['content_sums'][$l_data_key] = [];

                                                if (!isset($l_return['content_sums'][$l_data_key][$l_current_value]))
                                                {
                                                    $l_return['content_sums'][$l_data_key][$l_current_value] = 0;
                                                } // if
                                            } // if

                                            $l_return['content_sums'][$l_data_key][$l_current_value]++;
                                        }
                                        else if ($l_is_formattable)
                                        {
                                            if (!isset($l_return['content_sums'][$l_data_key]))
                                            {
                                                $l_return['content_sums'][$l_data_key] = 0;
                                            } // if

                                            // For numbers we'd like to take the "raw" numbers (because of different units etc.).
                                            if (isset($l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]) &&
                                                array_key_exists($l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS], $l_row))
                                            {
                                                $l_return['content_sums'][$l_data_key] += $l_row[$l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]];
                                            }
                                            else
                                            {
                                                $l_return['content_sums'][$l_data_key] += $l_row[$l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD]];
                                            } // if

                                            $l_formattables[$l_data_key] = [
                                                'row'  => $l_row,
                                                'prop' => $l_property,
                                                'unit' => false
                                            ];

                                            if (!empty($l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__UNIT]))
                                            {
                                                $l_formattables[$l_data_key]['unit'] = $l_properties[$l_property[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__UNIT]];
                                            } // if
                                        }
                                        else if ($l_is_numeric)
                                        {
                                            if (!isset($l_return['content_sums'][$l_data_key]))
                                            {
                                                $l_return['content_sums'][$l_data_key] = 0;
                                            } // if

                                            // For numbers we'd like to take the "raw" numbers (because of different units etc.).
                                            if (isset($l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]) &&
                                                array_key_exists($l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS], $l_row))
                                            {
                                                $l_return['content_sums'][$l_data_key] += $l_row[$l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]];
                                            }
                                            else
                                            {
                                                $l_return['content_sums'][$l_data_key] += $l_row[$l_property[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD]];
                                            } // if
                                        } // if
                                    } // if
                                } // foreach
                            } // while
                        } // if
                    } // foreach
                } // foreach
            } // foreach

            // Another "foreach" for formattable data.
            foreach ($l_formattables as $l_key => $l_data)
            {
                $l_value = $l_return['content_sums'][$l_key];

                // We don't "format" object browser data.
                if ($l_data['prop'][C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE] == C__PROPERTY__INFO__TYPE__OBJECT_BROWSER)
                {
                    unset($l_return['content_sums'][$l_key], $l_formattables[$l_key]);
                    continue;
                } // if

                $l_unit = '';

                if (class_exists($l_data['prop'][C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][0]))
                {
                    if (isset($l_data['prop'][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]))
                    {
                        $l_data['row'][$l_data['prop'][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]] = $l_value;
                    }
                    else
                    {
                        $l_data['row'][$l_data['prop'][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD]] = $l_value;
                    } // if

                    $l_helper = new $l_data['prop'][C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][0](
                        $l_data['row'], $this->m_db, $l_data['prop'][C__PROPERTY__DATA], $l_data['prop'][C__PROPERTY__FORMAT], $l_data['prop'][C__PROPERTY__UI]
                    );

                    if ($l_data['unit'])
                    {
                        if (method_exists($l_helper, 'set_unit_const'))
                        {
                            if (isset($l_data['unit'][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]))
                            {
                                $l_const = $l_data['row'][$l_data['unit'][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]];
                            }
                            else
                            {
                                $l_const = $l_data['row'][$l_data['unit'][C__PROPERTY__DATA][C__PROPERTY__DATA__REFERENCES][0] . '__const'];
                                $l_unit  = ' ' . _L($l_data['row'][$l_data['unit'][C__PROPERTY__DATA][C__PROPERTY__DATA__REFERENCES][0] . '__title'] ?: '');
                            } // if

                            $l_helper->set_unit_const($l_const);
                        } // if
                    } // if

                    if (method_exists($l_helper, $l_data['prop'][C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1]))
                    {
                        if (isset($l_data['prop'][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]) &&
                            array_key_exists($l_data['prop'][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS], $l_data['row'])
                        )
                        {
                            $l_value = $l_helper->$l_data['prop'][C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1](
                                $l_data['row'][$l_data['prop'][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD_ALIAS]]
                            );
                        }
                        else
                        {
                            $l_value = $l_helper->$l_data['prop'][C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1](
                                $l_data['row'][$l_data['prop'][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD]],
                                ($l_key == 'C__CATG__CPU::frequency')
                            );
                        } // if

                        if (is_array($l_value) && isset($l_value['title']))
                        {
                            $l_value = $l_value['title'];
                        } // if
                    } // if
                } // if

                $l_return['content_sums'][$l_key] = $l_value . $l_unit;
            } // foreach

            $l_return['content'] = [];

            // Nearly last "foreach" for cleaning up the "content" array, so that the JSON response will be smaller.
            foreach ($l_row_data as $l_categories)
            {
                $l_row = [];

                foreach ($l_return['header'] as $l_category_identifier => $l_translation)
                {
                    if (!isset($l_categories[$l_category_identifier]) || empty($l_categories[$l_category_identifier]) || empty($l_categories[$l_category_identifier][0]))
                    {
                        $l_row[] = isys_tenantsettings::get('gui.empty_value', '-');
                    }
                    else
                    {
                        if (is_array($l_categories[$l_category_identifier]) && !is_object($l_categories[$l_category_identifier][0]))
                        {
                            $l_row[] = implode(', ', $l_categories[$l_category_identifier]);
                        }
                        else if ($l_categories[$l_category_identifier][0] instanceof isys_export_data)
                        {
                            $l_collection = [];

                            // This is all necessary, because sometimes we get a clean array... Other times we get an array of "isys_export_data" objects.
                            foreach ($l_categories[$l_category_identifier] as $l_data)
                            {
                                $l_data = $l_data->get_data();

                                if (is_assoc($l_data))
                                {
                                    if (isset($l_data['title']))
                                    {
                                        $l_collection[] = $l_data['title'];
                                    } // if
                                }
                                else
                                {
                                    foreach ($l_data as $l_item)
                                    {
                                        if (isset($l_item['title']))
                                        {
                                            $l_collection[] = $l_item['title'];
                                        } // if
                                    } // foreach
                                } // if
                            } // foreach

                            $l_row[] = implode(', ', $l_collection);
                        } // if
                    } // if
                } // foreach

                $l_return['content'][] = $l_row;
            } // foreach

            // One last "foreach" to sort the sum-array.
            foreach ($l_return['content_sums'] as &$l_data)
            {
                if (is_array($l_data))
                {
                    arsort($l_data);
                } // if
            } // foreach

            $l_return['header']['obj_count']       = _L('LC__MODULE__ANALYTICS__OBJECT_CATALOG__OBJ_COUNT');
            $l_return['content_sums']['obj_count'] = count($l_row_data);
        } // if

        return $l_return;
    } // function
} // class