<?php

/**
 * i-doit
 *
 * Service costs report.
 *
 * @package     modules
 * @subpackage  analytics
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */
class isys_analytics_reports_servicecosts extends isys_analytics_reports
{
    /**
     * This constant holds the name of the report.
     *
     * @var  string
     */
    const TITLE = 'LC__MODULE__ANALYTICS__SERVICECOSTS';

    /**
     * This constant holds the icon-path of the report.
     *
     * @var  string
     */
    const ICON = 'icons/silk/monitor.png';

    /**
     * @var  isys_component_template
     */
    private $m_tpl = null;

    /**
     * @var  isys_component_database
     */
    private $m_db = null;

    /**
     * @var  isys_analytics_dao
     */
    private $m_dao = null;

    /**
     * Variable which holds information about the selected categories (isysgui_catg, *_cats, ...).
     *
     * @var  array
     */
    private $m_categories = [];

    /**
     * Variable with all costs summed up.
     *
     * @var  integer
     */
    private $m_total_costs = 0;

    /**
     * Variable which holds the summed up amount of Watt.
     *
     * @var  integer
     */
    private $m_total_watt = 0;

    /**
     * Variable with the costs of all categories.
     *
     * @var  array
     */
    private $m_category_costs = [];

    /**
     * Variable with the values of miscellaneous categories.
     *
     * @var  array
     */
    private $m_category_misc = [];

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->m_db  = isys_application::instance()->database;
        $this->m_tpl = isys_application::instance()->template;
        $this->m_dao = isys_analytics_dao::instance($this->m_db);
    } // function

    /**
     * Method for preparing the visual report view output.
     *
     * @return  isys_analytics_reports_servicecosts
     */
    public function start()
    {
        $l_get     = $_GET;
        $l_filters = $l_itservices = [];

        $l_rules = [
            'obj_catalog_selection' => [
                isys_popup_browser_object_ng::C__CAT_FILTER => 'C__CATG__SERVICE;C__CATG__IT_SERVICE_COMPONENTS;C__CATG__ITS_LOGBOOK;C__CATG__IT_SERVICE_RELATIONS;C__CATG__ITS_TYPE'
            ]
        ];

        // Remove some unnecessary parameters for the Ajax URL.
        unset($l_get[C__GET__MAIN_MENU__NAVIGATION_ID], $l_get[C__CMDB__GET__EDITMODE], $l_get[C__GET__TREE_NODE]);

        $l_itservice_filter_url = isys_helper_link::create_url([
            C__GET__MODULE_ID     => C__MODULE__ITSERVICE,
            C__GET__SETTINGS_PAGE => 'filter-config'
        ]);

        $l_itservice_filters = isys_factory::get_instance('isys_itservice_dao_filter_config', $this->m_db)->get_data();

        if (is_array($l_itservice_filters) && count($l_itservice_filters))
        {
            foreach ($l_itservice_filters as $l_filter)
            {
                $l_filters[$l_filter['isys_itservice_filter_config__id']] = $l_filter['isys_itservice_filter_config__title'];
            } // foreach

            $l_rules['impact_itservice_filter']['p_arData'] = $l_filters;
        } // if

        $l_rules['obj_catalog_attributes']['provide']      = C__PROPERTY__PROVIDES__REPORT;
        $l_rules['obj_catalog_attributes']['preselection'] = isys_format_json::encode([['g' => ['C__CATG__GLOBAL' => ['title']]]]);

        isys_component_template_navbar::getInstance()
            ->append_button(
                'LC__MODULE__ANALYTICS__SERVICECOSTS__CALCULATE',
                'start-calculation',
                [
                    'tooltip'    => 'LC__MODULE__ANALYTICS__SERVICECOSTS__CALCULATE',
                    'icon'       => 'icons/silk/cog.png',
                    'js_onclick' => ';',
                    'navmode'    => 'calculation',
                ]
            );

        $this->m_tpl->activate_editmode()
            ->assign('filter_description', _L('LC__MODULE__ANALYTICS__FILTERS_DESCRIPTION', $l_itservice_filter_url))
            ->assign('ajax_url', isys_helper_link::create_url($l_get))
            ->assign('currency', isys_application::instance()->container->locales->get_currency())
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', __DIR__ . DS . get_class() . '_view.tpl');

        return $this;
    } // function

    /**
     * This method will be called by the framework, to process ajax requests.
     *
     * @param   mixed $p_data
     *
     * @return  mixed
     */
    public function ajax_request($p_data = [])
    {
        switch ($_GET['func'])
        {
            default:
            case 'start-calculation':
                return $this->start_calculation((int) $_POST['itservice_obj'], (int) $_POST['filter']);

            case 'calculate-costs':
                return $this->calculate_costs(isys_format_json::decode($_POST['obj_ids']));
        } // switch
    } // function

    /**
     * This method will trigger the calculation algorithm!
     *
     * @param   integer $p_itservice_obj
     * @param   integer $p_filter
     *
     * @throws  isys_exception_general
     * @return  array
     */
    private function start_calculation($p_itservice_obj, $p_filter = null)
    {
        if (empty($p_itservice_obj))
        {
            throw new isys_exception_general(_L('LC__MODULE__ANALYTICS__NOTIFY__NO_IMPACT_OBJECT'));
        } // if

        $l_filter = $l_it_service_mapping = $l_return = [];

        // Every calculation shall begin with an empty cache.
        $this->m_dao->get_cache()->clear();

        // Prepare the filters.
        if ($p_filter > 0)
        {
            $l_filter = isys_factory::get_instance('isys_itservice_dao_filter_config', $this->m_db)->get_data($p_filter);

            $l_filter = $l_filter['formatted__data'];
        } // if

        if (empty($l_filter['level']))
        {
            $l_filter['level'] = 11;
        } // if

        // Add root node.
        $l_root_node = $this->m_dao->format_node($p_itservice_obj);
        $l_tree      = new isys_tree($l_root_node);

        $this->m_dao->relation_walk(
            $l_root_node,
            $l_filter['priority'],
            $l_filter['relation-type'] ?: [],
            $l_filter['object-type'] ?: []
        );

        $l_root_node->set_subnodes($l_root_node->count());

        // Prepare tree.
        if (isset($l_tree) && $l_tree instanceof isys_tree)
        {
            // Filter tree using the visitor pattern.
            $l_visitor = new isys_tree_visitor_analytics($l_filter);

            while ($l_tree->accept($l_visitor))
            {
                // filtering..
            } // while

            foreach ($l_tree->all_nodes() as $l_node)
            {
                $l_return[] = $l_node->get_id();
            } // foreach

            return array_values(array_unique($l_return, SORT_NUMERIC));
        } // if

        return [];
    } // function

    /**
     * This function will retrieve the selected attributes of the given objects.
     *
     * @param   array $p_obj_ids
     *
     * @return  array
     */
    private function calculate_costs(array $p_obj_ids)
    {
        global $g_dirs;

        if (!count($p_obj_ids))
        {
            // We don't want an "error" message, so we can't use an exception.
            isys_notify::warning(_L('LC__MODULE__ANALYTICS__NOTIFY__EMPTY_ATTRIBUTE_RESULT'), ['life' => 10]);

            return [];
        } // if

        $l_objects = [];

        foreach ($p_obj_ids as $l_obj_id)
        {
            $l_object = $this->m_dao->get_object_by_id($l_obj_id)
                ->get_row();

            $l_objects[$l_obj_id] = [
                'type_icon'  => (strpos($l_object['isys_obj_type__icon'], '/') === false ? $g_dirs['images'] . 'tree/' : '') . $l_object['isys_obj_type__icon'],
                'title'      => _L($l_object['isys_obj__title']),
                'type_title' => _L($l_object['isys_obj_type__title'])
            ];
        } // foreach

        /*
         * Short syntax explanation:
         *
         * array("$key" => 123)
         * Will be rendered with 2 digits after the comma and get the currency appended: "123.00 EUR".
         *
         * array("+key" => array("id" => 123, "title" => "yoloswag"))
         * Marks a field as "dialog+" field: "yoloswag".
         *
         * array("()" => array("$" => 123, "" => "per month"))
         * Will render the array members inside brackets at the end of the field: "... (123.00 EUR per month)".
         */

        $l_categories = [
            'C__CATS__LICENCE'             => [
                'header'  => [_L('LC__UNIVERSAL__OBJECT_TITLE'), _L('LC__CMDB__CATG__LIC_ASSIGN__LICENSE'), _L('LC__CMDB__CATS__LICENCE_UNIT_PRICE')],
                'content' => $this->calculate_licence_data($p_obj_ids)
            ],
            'C__CATG__ACCOUNTING'          => [
                'header'  => [_L('LC__UNIVERSAL__OBJECT_TITLE'), _L('LC__CMDB__CATG__GLOBAL_PRICE'), _L('LC__CMDB__CATG__ACCOUNTING__YEARLY_OPERATION_EXPENSE')],
                'content' => $this->calculate_accounting_data($p_obj_ids)
            ],
            'C__CATG__CONTRACT_ASSIGNMENT' => [
                'header'  => [_L('LC__UNIVERSAL__OBJECT_TITLE'), ucfirst(_L('LC__CMDB__CATG__MAINTENANCE_OBJ_MAINTENANCE')), _L('LC__CMDB__OBJTYPE__CONTRACTS')],
                'content' => $this->calculate_contract_data($p_obj_ids)
            ],
            'C__CATG__POWER_CONSUMER'      => [
                'header'  => [_L('LC__UNIVERSAL__OBJECT_TITLE'), _L('LC__CMDB__CATG__POWER_CONSUMER__TITLE'), _L('LC__CMDB__CATS__POBJ_WATT')],
                'content' => $this->calculate_power_consumer_data($p_obj_ids)
            ]
        ];

        foreach ($this->m_categories as $l_const => &$l_data)
        {
            $l_data['count'] = count($l_categories[$l_const]['content']);
        } // foreach

        return [
            'categories' => $l_categories,
            'misc'       => [
                'currency'       => isys_application::instance()->container->locales->get_currency(),
                'objects'        => $l_objects,
                'categories'     => $this->m_categories,
                'category_costs' => $this->m_category_costs,
                'category_misc'  => $this->m_category_misc,
                'total_costs'    => $this->m_total_costs,
                'total_watt'     => $this->m_total_watt . ' ' . _L('LC__CMDB__CATG__POWER_SUPPLIER__WATT')
            ]
        ];
    } // function

    /**
     * Method for preparing the licence data.
     *
     * @param   array $p_obj_ids
     *
     * @return  array
     */
    private function calculate_licence_data(array $p_obj_ids)
    {
        $l_return = [];

        $this->m_categories['C__CATS__LICENCE']          = $this->m_dao->gui_get_info_by_category(C__CMDB__CATEGORY__TYPE_SPECIFIC, C__CATS__LICENCE)->get_row();
        $this->m_categories['C__CATS__LICENCE']['title'] = _L('LC__MODULE__ANALYTICS__SERVICECOSTS__LICENCECOST');

        $l_app_dao                                  = isys_factory_cmdb_category_dao::get_instance('isys_cmdb_dao_category_g_application', $this->m_db);
        $l_lic_dao                                  = isys_factory_cmdb_category_dao::get_instance('isys_cmdb_dao_category_s_lic', $this->m_db);
        $this->m_category_costs['C__CATS__LICENCE'] = 0;

        foreach ($p_obj_ids as $l_obj_id)
        {
            $l_res = $l_app_dao->get_data(null, $l_obj_id);

            if (count($l_res))
            {
                while ($l_row = $l_res->get_row())
                {
                    // The connected licence.
                    if ($l_row['isys_catg_application_list__isys_cats_lic_list__id'] > 0)
                    {
                        $l_licence = $l_lic_dao->get_data($l_row['isys_catg_application_list__isys_cats_lic_list__id'])
                            ->get_row();

                        $this->m_category_costs['C__CATS__LICENCE'] += (float) $l_licence['isys_cats_lic_list__cost'];
                        $this->m_total_costs                        += (float) $l_licence['isys_cats_lic_list__cost'];

                        $l_return[] = [
                            '_object_' => $l_obj_id,
                            'licence'  => $l_licence['isys_obj__title'],
                            '$cost'    => $l_licence['isys_cats_lic_list__cost']
                        ];
                    } // if
                } // while
            } // if
        } // foreach

        return $l_return;
    } // function

    /**
     * Method for preparing the accounting data.
     *
     * @param   array $p_obj_ids
     *
     * @return  array
     */
    private function calculate_accounting_data(array $p_obj_ids)
    {
        $l_return = [];

        $this->m_categories['C__CATG__ACCOUNTING']                 = $this->m_dao->gui_get_info_by_category(C__CMDB__CATEGORY__TYPE_GLOBAL, C__CATG__ACCOUNTING)->get_row();
        $this->m_categories['C__CATG__ACCOUNTING']['title']        = _L($this->m_categories['C__CATG__ACCOUNTING']['isysgui_catg__title']);
        $this->m_categories['C__CATG__ACCOUNTING_price']['title']  = _L('LC__CMDB__CATG__GLOBAL_PRICE');
        $this->m_categories['C__CATG__ACCOUNTING_yearly']['title'] = _L('LC__CMDB__CATG__ACCOUNTING__YEARLY_OPERATION_EXPENSE');

        $l_dao                                                = isys_factory_cmdb_category_dao::get_instance('isys_cmdb_dao_category_g_accounting', $this->m_db);
        $this->m_category_costs['C__CATG__ACCOUNTING_price']  = 0;
        $this->m_category_costs['C__CATG__ACCOUNTING_yearly'] = 0;
        $l_empty_value                                        = isys_tenantsettings::get('gui.empty_value', '-');

        foreach ($p_obj_ids as $l_obj_id)
        {
            $l_row = $l_dao->get_data(null, $l_obj_id)
                ->get_row();

            if (is_array($l_row))
            {
                $l_operation_expense = $l_empty_value;

                if ($l_row['isys_catg_accounting_list__isys_interval__id'] > 0)
                {
                    $l_yearly_costs = self::yearly_costs($l_row['isys_catg_accounting_list__operation_expense'], $l_row['isys_catg_accounting_list__isys_interval__id']);

                    $l_operation_expense = [
                        '$'  => $l_yearly_costs,
                        '()' => [
                            '$' => $l_row['isys_catg_accounting_list__operation_expense'],
                            ''  => _L($l_row['isys_interval__title'])
                        ]
                    ];

                    $this->m_category_costs['C__CATG__ACCOUNTING']        += $l_yearly_costs;
                    $this->m_category_costs['C__CATG__ACCOUNTING_yearly'] += $l_yearly_costs;
                    $this->m_total_costs                                  += $l_yearly_costs;
                } // if

                $this->m_category_costs['C__CATG__ACCOUNTING']       += (float) $l_row['isys_catg_accounting_list__price'];
                $this->m_category_costs['C__CATG__ACCOUNTING_price'] += (float) $l_row['isys_catg_accounting_list__price'];
                $this->m_total_costs                                 += (float) $l_row['isys_catg_accounting_list__price'];

                $l_return[] = [
                    '_object_'          => $l_obj_id,
                    '$price'            => $l_row['isys_catg_accounting_list__price'],
                    'operation_expense' => $l_operation_expense
                ];
            } // if
        } // foreach

        return $l_return;
    } // function

    /**
     * Method for preparing the contract data.
     *
     * @param   array $p_obj_ids
     *
     * @return  array
     */
    private function calculate_contract_data(array $p_obj_ids)
    {
        $l_return = [];

        // @See ID-4486 and ANALYSE-38: Certain contracts shall only be counted once.
        $l_unique_contracts = [];

        $this->m_categories['C__CATG__CONTRACT_ASSIGNMENT']          = $this->m_dao->gui_get_info_by_category(C__CMDB__CATEGORY__TYPE_GLOBAL, C__CATG__CONTRACT_ASSIGNMENT)->get_row();
        $this->m_categories['C__CATG__CONTRACT_ASSIGNMENT']['title'] = _L('LC__MODULE__ANALYTICS__SERVICECOSTS__CONTRACTCOST');

        $l_dao                                                  = isys_cmdb_dao_category_g_contract_assignment::instance($this->m_db);
        $l_contract_dao                                         = isys_cmdb_dao_category_s_contract::instance($this->m_db);
        $this->m_category_costs['C__CATG__CONTRACT_ASSIGNMENT'] = 0;

        $l_empty_value = isys_tenantsettings::get('gui.empty_value', '-');

        foreach ($p_obj_ids as $l_obj_id)
        {
            $l_res = $l_dao->get_data(null, $l_obj_id);

            if (count($l_res))
            {
                while ($l_row = $l_res->get_row())
                {
                    $l_contract = $l_contract_dao->get_data(null, $l_row['isys_cats_contract_list__isys_obj__id'])->get_row();

                    $l_yearly_costs = self::yearly_costs($l_row['isys_cats_contract_list__costs'], null, $l_row['isys_cats_contract_list__isys_contract_payment_period__id']);

                    // This check shall prevent the logic from older i-doit versions (< 1.9.2).
                    if (isset($l_contract['isys_cats_contract_list__cost_calculation']) && defined('isys_cmdb_dao_category_s_contract::COST_CALCULATION__OVERALL'))
                    {
                        if ($l_contract['isys_cats_contract_list__cost_calculation'] === isys_cmdb_dao_category_s_contract::COST_CALCULATION__OVERALL)
                        {
                            if (isset($l_unique_contracts[$l_contract['isys_obj__id']]))
                            {
                                $l_yearly_costs = 0;
                            }
                            else
                            {
                                $l_unique_contracts[$l_contract['isys_obj__id']] = true;
                                $l_contract['isys_obj__title'] .= ' <em class="text-grey">(' . _L('LC__MODULE__ANALYTICS__SERVICECOSTS__CONTRACTCOST__ALREADY_CALCULATED') . ')</em>';
                            } // if
                        } // if
                    } // if

                    if ($l_row['isys_cats_contract_list__isys_contract_payment_period__id'] > 0)
                    {
                        $l_period = isys_factory_cmdb_dialog_dao::get_instance('isys_contract_payment_period', $this->m_db)->get_data($l_row['isys_cats_contract_list__isys_contract_payment_period__id']);
                    }
                    else
                    {
                        $l_period = ['title' => $l_empty_value];
                    } // if

                    $this->m_category_costs['C__CATG__CONTRACT_ASSIGNMENT'] += $l_yearly_costs;
                    $this->m_total_costs                                    += $l_yearly_costs;

                    $l_return[] = [
                        '_object_' => $l_obj_id,
                        'contract' => $l_contract['isys_obj__title'],
                        'costs'    => [
                            '$'  => $l_yearly_costs,
                            '()' => [
                                '$' => $l_row['isys_cats_contract_list__costs'],
                                ''  => _L($l_period['title']),
                            ]
                        ]
                    ];
                } // while
            } // if
        } // foreach

        return $l_return;
    } // function

    /**
     * Method for preparing the power consumer data.
     * This method will not add any costs - Simply display the consumed power.
     *
     * @param   array $p_obj_ids
     *
     * @return  array
     */
    private function calculate_power_consumer_data(array $p_obj_ids)
    {
        $l_return = [];

        $this->m_categories['C__CATG__POWER_CONSUMER']          = $this->m_dao->gui_get_info_by_category(C__CMDB__CATEGORY__TYPE_GLOBAL, C__CATG__POWER_CONSUMER)
            ->get_row();
        $this->m_categories['C__CATG__POWER_CONSUMER']['title'] = _L($this->m_categories['C__CATG__POWER_CONSUMER']['isysgui_catg__title']);

        $l_dao         = isys_factory_cmdb_category_dao::get_instance('isys_cmdb_dao_category_g_power_consumer', $this->m_db);
        $l_empty_value = isys_tenantsettings::get('gui.empty_value', '-');
        foreach ($p_obj_ids as $l_obj_id)
        {
            $l_res = $l_dao->get_data(null, $l_obj_id);

            if (count($l_res))
            {
                while ($l_row = $l_res->get_row())
                {
                    $l_return[] = [
                        '_object_' => $l_obj_id,
                        'name'     => $l_row['isys_catg_pc_list__title'],
                        'watt'     => ($l_row['isys_catg_pc_list__watt'] ?: $l_empty_value) . ' W'
                    ];

                    $this->m_category_misc['C__CATG__POWER_CONSUMER'] += $l_row['isys_catg_pc_list__watt'];
                    $this->m_total_watt                               += $l_row['isys_catg_pc_list__watt'];
                } // while
            } // if
        } // foreach

        return $l_return;
    } // function

    /**
     * This method helps to calculate the "yearly" costs by a given price and interval.
     *
     * @static
     *
     * @param   float   $p_costs
     * @param   integer $p_accounting_interval_id
     * @param   integer $p_contract_interval_id
     *
     * @return  float
     */
    private static function yearly_costs($p_costs, $p_accounting_interval_id = null, $p_contract_interval_id = null)
    {
        if ($p_accounting_interval_id !== null)
        {
            switch ($p_accounting_interval_id)
            {
                case C__INTERVAL__PER_DAY:
                    return $p_costs * 365;

                case C__INTERVAL__PER_WEEK:
                    return $p_costs * 52;

                case C__INTERVAL__PER_MONTH:
                    return $p_costs * 12;

                default:
                case C__INTERVAL__PER_YEAR:
                    return $p_costs;
            } // switch
        } // if

        if ($p_contract_interval_id !== null)
        {
            switch ($p_contract_interval_id)
            {
                case C__CONTRACT__PAYMENT_PERIOD__MONTHLY:
                    return $p_costs * 12;

                case C__CONTRACT__PAYMENT_PERIOD__QUARTERLY:
                    return $p_costs * 4;

                case C__CONTRACT__PAYMENT_PERIOD__HALF_YEARLY:
                    return $p_costs * 2;

                default:
                case C__CONTRACT__PAYMENT_PERIOD__YEARLY:
                    return $p_costs;
            } // switch
        } // if

        return $p_costs;
    } // function
} // class