var ProfileManager = Class.create({
    initialize: function ($el, options) {
        // First we identify all the elements.
        this.$el = $el;
        this.$newProfileButton = new Element('button', {type: 'button', className: 'm5 btn'})
            .update(new Element('img', {src: window.dir_images + 'icons/silk/add.png', className: 'mr5'}))
            .insert(new Element('span').update(idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__ADD_NEW_PROFILE')));
        this.$profileList = new Element('ul', {className: 'list'});
        this.$objectTypeList = new Element('ul', {className: 'list'});
        this.$categoryList = new Element('ul', {className: 'list'});
        this.$reportList = new Element('select', {className: 'input input-size-mini m5'});
        this.$reportCheckButton = new Element('button', {className: 'btn m5', type: 'button'})
            .update(new Element('span').update(idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__CHECK_REPORT')));
        this.$overlay = new Element('div', {className: 'overlay'});
    
        this.counter = 1;
        this.profiles = [];
        this.reports = [];
        this.deletedProfiles = [];
        
        // Then we set the observers.
        this.$newProfileButton.on('click', this.createProfile.bind(this));
        this.$profileList.on('click', 'li', this.editProfile.bind(this));
        this.$profileList.on('click', '[data-action="rename"]', this.renameProfile.bind(this));
        this.$profileList.on('click', '[data-action="duplicate"]', this.duplicateProfile.bind(this));
        this.$profileList.on('click', '[data-action="delete"]', this.deleteProfile.bind(this));
        this.$objectTypeList.on('click', 'button', this.loadCategories.bind(this));
        this.$objectTypeList.on('change', 'input', this.toggleObjectType.bind(this));
        this.$categoryList.on('change', 'input', this.toggleCategory.bind(this));
        this.$reportList.on('change', this.selectReport.bind(this));
        this.$reportCheckButton.on('click', this.checkReport.bind(this))
        
        this.options = {
            objectTypes: [],
            categories: [],
            activeProfile: null,
            activeObjectType: null
        };
        
        Object.extend(this.options, options || {});
        
        this.$el
            .update(new Element('div')
                .update(this.$newProfileButton)
                .insert(this.$profileList))
            .insert(new Element('div')
                .update(new Element('label', {className: 'display-block ml5'})
                    .update(idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__REPORT_PRE_FILTER')))
                .insert(this.$reportList)
                .insert(this.$reportCheckButton)
                .insert(new Element('div', {className: 'configuration-content'})
                    .update(new Element('div')
                        .update(new Element('h3', {className: 'mt10 pl5 mb15'})
                            .update(idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__OBJECT_TYPES'))
                            .insert(new Element('span', {className: 'text-neutral-400 ml10'})))
                        .insert(this.$objectTypeList))
                    .insert(new Element('div')
                        .update(new Element('h3', {className: 'mt10 mb15'})
                            .update(idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__CATEGORIES'))
                            .insert(new Element('span', {className: 'text-neutral-400 ml10'})))
                        .insert(this.$categoryList))))
            .insert(this.$overlay);
        
        // Finally we load the detail data.
        this.loadProfiles();
        this.loadReports();
        this.renderObjectTypes();
    },
    
    getProfiles: function () {
        return this.profiles;
    },
    
    getDeletedProfiles: function () {
        return this.deletedProfiles;
    },
    
    toggleSelectedState: function ($input) {
        if ($input.checked) {
            $input.up('li').removeClassName('not-selected');
        } else {
            $input.up('li').addClassName('not-selected');
        }
    },
    
    createProfile: function () {
        this.profiles.push({id: 'new-' + this.counter, title: 'New profile', data: {}});
    
        this.counter++;
        
        this.renderProfiles();
    },
    
    getProfile: function (id) {
        return this.profiles.find(function (profile) {
            return profile.id == id;
        });
    },
    
    editProfile: function (ev) {
        this.$profileList.select('.active').invoke('removeClassName', 'active');
        
        const id = ev.findElement('li').addClassName('active').readAttribute('data-profile-id');
        const profile = this.getProfile(id);
        const objectTypes = Object.keys(profile.data);

        this.options.activeProfile = id;
        
        this.$objectTypeList.up('div').down('h3 span').update(profile.title);
        this.$reportList.setValue(-1);
        
        if (profile.data.reportId !== undefined && profile.data.reportId > 0) {
            this.$reportList.setValue(profile.data.reportId);
        }
        
        // Preselect object types.
        this.$objectTypeList.select('li').each(function ($li) {
            if (objectTypes.includes($li.readAttribute('data-profile-const'))) {
                $li.removeClassName('not-selected').down('input').setValue(1);
            } else {
                $li.addClassName('not-selected').down('input').setValue(0);
            }
        });
    
        this.$overlay.addClassName('hide');
    },
    
    renameProfile: function (ev) {
        ev.preventDefault();
        ev.stopPropagation();
        const id = ev.findElement('li').addClassName('active').readAttribute('data-profile-id');
        const profile = this.profiles.find(function (profile) {
            return profile.id == id;
        });
        
        const newName = prompt(idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__RENAME_PROFILE'), profile.title);
        
        if (newName !== null && !newName.blank()) {
            profile.title = newName;
            this.$objectTypeList.up('div').down('h3 span').update(newName);
            
            this.renderProfiles();
        }
    },
    
    duplicateProfile: function (ev) {
        ev.preventDefault();
        ev.stopPropagation();
        const id = ev.findElement('li').readAttribute('data-profile-id');
        const profile = this.profiles.find(function (profile) {
            return profile.id == id;
        });
    
        this.profiles.push({id: 'new-' + this.counter, title: profile.title + ' duplicate', data: profile.data});
    
        this.counter++;
    
        this.renderProfiles();
    },
    
    deleteProfile: function (ev) {
        const profileId = ev.findElement('li').readAttribute('data-profile-id');
        
        if (confirm(idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY__PURGE_CONFIRM'))) {
            this.deletedProfiles.push(profileId);
            
            this.profiles = this.profiles.filter(function (profile) {
                return profile.id != profileId;
            });
            
            this.options.activeProfile = null;
            this.options.activeObjectType = null;
            this.renderProfiles();
        }
    },
    
    loadProfiles: function () {
        var self = this;
    
        new Ajax.Request(window.www_dir + 'analytics/dataquality/loadProfiles', {
            parameters: {},
            onComplete: function (xhr) {
                var json = xhr.responseJSON;
                
                if (json.success) {
                    // Assign the loaded profiles to the internal state.
                    self.profiles = json.data

                    self.renderProfiles();
                } else {
                    idoit.Notify.error(json.message, {sticky: true});
                }
            }
        });
    },
    
    renderProfiles: function () {
        var profile;
        
        this.$profileList.update();

        for (var i in this.profiles) {
            if (! this.profiles.hasOwnProperty(i)) {
                continue;
            }
    
            profile = this.profiles[i];
            
            this.$profileList
                .insert(new Element('li', {className: 'mouse-pointer ' + (this.options.activeProfile == profile.id ? 'active' : ''), 'data-profile-id': profile.id})
                    .insert(new Element('span')
                        .update(profile.title))
                    .insert(new Element('button', {type: 'button', className: 'btn btn-small ml-auto', 'data-action': 'rename', title: idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__RENAME_PROFILE'), 'data-tooltip': 1})
                        .update(new Element('img', {src: window.dir_images + 'axialis/documents-folders/document-edit.svg'})))
                    .insert(new Element('button', {type: 'button', className: 'btn btn-small ml5', 'data-action': 'duplicate', title: idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__DUPLICATE_PROFILE'), 'data-tooltip': 1})
                        .update(new Element('img', {src: window.dir_images + 'axialis/documents-folders/pages.svg'})))
                    .insert(new Element('button', {type: 'button', className: 'btn btn-small ml5', 'data-action': 'delete', title: idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__DELETE_PROFILE'), 'data-tooltip': 1})
                        .update(new Element('img', {src: window.dir_images + 'axialis/basic/button-remove.svg'})))
            );
        }
    
        $('body').fire('update:tooltips');
    },
    
    loadReports: function () {
        var self = this;
        
        new Ajax.Request(window.www_dir + 'analytics/dataquality/loadReports', {
            parameters: {},
            onComplete: function (xhr) {
                var json = xhr.responseJSON;
                
                if (json.success) {
                    // Assign the loaded reports to the internal state.
                    self.reports = json.data
                    
                    self.renderReports();
                } else {
                    idoit.Notify.error(json.message, {sticky: true});
                }
            }
        });
    },
    
    renderReports: function () {
        this.$reportList.update(new Element('option', {value: -1}).update('-'));
        
        for (var i in this.reports) {
            if (! this.reports.hasOwnProperty(i)) {
                continue;
            }
    
            this.$reportList.insert(new Element('option', {value: this.reports[i].id}).update(this.reports[i].title));
        }
    },
    
    checkReport: function () {
        var self = this;
        const reportId = parseInt(self.$reportList.getValue());
    
        if (reportId === -1) {
            return;
        }
        
        new Ajax.Request(window.www_dir + 'analytics/dataquality/checkReport', {
            parameters: {
                reportId: reportId
            },
            onComplete: function (xhr) {
                var json = xhr.responseJSON;
            
                if (json.success) {
                    // Display the result of the report check.
                    if (json.data.compatible) {
                        idoit.Notify.success(idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__REPORT_COMPATIBLE').replace(':count', json.data.count), {sticky: true});
                    } else {
                        idoit.Notify.error(idoit.Translate.get('LC__MODULE__ANALYTICS__DATA_QUALITY_PROFILES__REPORT_INCOMPATIBLE'), {sticky: true});
                    }
                } else {
                    idoit.Notify.error(json.message, {sticky: true});
                }
            }
        });
    },
    
    selectReport: function () {
        const reportId = parseInt(this.$reportList.getValue());
        const profile = this.getProfile(this.options.activeProfile);
        
        profile.data.reportId = null;
        
        if (reportId > 0) {
            profile.data.reportId = reportId;
        }
    },
    
    renderObjectTypes: function () {
        var objectType;
        
        this.$objectTypeList.update();
    
        for (var i in this.options.objectTypes) {
            if (! this.options.objectTypes.hasOwnProperty(i)) {
                continue;
            }
    
            objectType = this.options.objectTypes[i];
        
            this.$objectTypeList
                .insert(new Element('li', { 'data-profile-const': objectType.const})
                    .insert(new Element('label', { className: 'mouse-pointer'})
                        .insert(new Element('input', { type:'checkbox', value: objectType.const}))
                        .insert(new Element('span')
                            .update(objectType.title)))
                    .insert(new Element('button', { type: 'button', className: 'btn btn-small ml-auto', title: idoit.Translate.get('LC__CMDB__LIST_LOAD_ALL'), 'data-tooltip': 1})
                        .update(new Element('img', { src: window.dir_images + 'axialis/basic/zoom.svg'}))));
        }
    
        $('body').fire('update:tooltips');
    },
    
    loadCategories: function (ev) {
        var self = this;
        
        this.$objectTypeList.select('.active').invoke('removeClassName', 'active');
        
        const $activeItem = ev.findElement('li');
        const $checkbox = $activeItem.down('input');
        
        if (!$checkbox.checked) {
            $checkbox.setValue(1);
            $checkbox.simulate('change');
        }
        
        const objectTypeConstant = $activeItem.addClassName('active').readAttribute('data-profile-const');
        this.options.activeObjectType = objectTypeConstant;
    
        this.$categoryList.up('div').down('h3 span').update($activeItem.down('span').innerText);
        
        new Ajax.Request(window.www_dir + 'analytics/dataquality/loadCategoriesByObjectType', {
            parameters: {
                objectTypeConst: objectTypeConstant
            },
            onComplete: function (xhr) {
                var json = xhr.responseJSON;
            
                if (json.success) {
                    // Assign the loaded profiles to the internal state.
                    self.options.categories = json.data
                
                    self.renderCategories();
                } else {
                    idoit.Notify.error(json.message, {sticky: true});
                }
            }
        });
    },
    
    renderCategories: function () {
        const profile = this.getProfile(this.options.activeProfile);
        const selectedCategories = profile.data[this.options.activeObjectType];
        let category;
        let selected;

        this.$categoryList.update();
    
        for (var i in this.options.categories) {
            if (! this.options.categories.hasOwnProperty(i)) {
                continue;
            }
        
            category = this.options.categories[i];
        
            this.$categoryList
                .insert(new Element('li', {'data-category-const': category.const})
                    .insert(new Element('label', {className: 'mouse-pointer'})
                        .insert(new Element('input', {type:'checkbox', value: category.const}))
                        .insert(new Element('span', {title: category.type})
                            .update(category.title))));
        }

        this.$categoryList.select('li').each(function($li) {
            if (selectedCategories.includes($li.readAttribute('data-category-const'))) {
                $li.removeClassName('not-selected').down('input').setValue(1);
            } else {
                $li.addClassName('not-selected').down('input').setValue(0);
            }
        });
    },
    
    toggleObjectType: function (ev) {
        const $input = ev.findElement('input');
        const objectTypeConstant = $input.up('li').readAttribute('data-profile-const');
        const profile = this.getProfile(this.options.activeProfile);
        
        this.toggleSelectedState($input);
        
        if (Object.isArray(profile.data)) {
            profile.data = {};
        }
        
        if ($input.checked) {
            // Add the object type to our data.
            profile.data[objectTypeConstant] = [];
        } else {
            // Remove the object type from our data.
            delete profile.data[objectTypeConstant];
        }
    },
    
    toggleCategory: function (ev) {
        const $input = ev.findElement('input');
        const categoryConstant = $input.up('li').readAttribute('data-category-const');
        const profile = this.getProfile(this.options.activeProfile);
    
        this.toggleSelectedState($input);
    
        if ($input.checked) {
            // Add the object type to our data.
            profile.data[this.options.activeObjectType].push(categoryConstant);
        } else {
            // Remove the object type from our data.
            profile.data[this.options.activeObjectType] = profile.data[this.options.activeObjectType].without(categoryConstant);
        }
    }
});
