<?php

use idoit\Component\Helper\Memory;

/**
 * i-doit
 *
 * Analytics DAO.
 *
 * @package     modules
 * @subpackage  analytics
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */
class isys_analytics_dao extends isys_cmdb_dao
{
	/**
	 * This array will contain the "simulation" objects.
	 * @var  array
	 */
	protected $m_simulation_objects = array();

	/**
	 * This array will contain the "recursing" objects to prevent endless loops.
	 * @var  array
	 */
	protected $m_recursion_objects = array();

	/**
	 * This array holds the standard options, which can be overwritten by "set_options".
	 * @var  array
	 */
	protected $m_options = [];

	/**
	 * The caching instance for our algorithm.
	 * @var  isys_caching
	 */
	protected $algorithmCache;

	/**
	 * This variable will cache all found object nodes.
	 * @var  array
	 */
	private $m_node_cache = array();

	/**
	 * This variable will be used to save the "relation weight" between simulation objects and it-services.
	 * @var  array
	 */
	protected $m_relation_weighting = array();

    /**
     * Memory helper.
     * @var  Memory
     */
    private $memoryHelper;

    /**
     * Memory counter - this will help us to only check the memory on every n'th iteration.
     * @var  integer
     */
    private $memoryCount = 0;


	/**
	 * Constructor.
	 *
	 * @param  isys_component_database $p_db
	 * @param  integer $p_cmdb_status
	 */
	public function __construct(isys_component_database $p_db, $p_cmdb_status = NULL)
	{
		$this->algorithmCache = isys_caching::factory('analytics-algorithm', isys_convert::MINUTE * 5);
        $this->memoryHelper = Memory::instance();

		parent::__construct($p_db, $p_cmdb_status);
	}

	/**
	 * Method for retrieving the cache instance.
	 *
	 * @return  isys_caching
	 */
	public function get_cache ()
	{
		return $this->algorithmCache;
	}


	/**
	 * Method for clearing the node cache.
	 *
	 * @return  isys_analytics_dao
	 */
	public function clear_node_cache ()
	{
		$this->m_node_cache = array();

		return $this;
	}


	/**
	 * Method for manually setting the simulated objects.
	 *
	 * @param   array  $p_objects
	 * @return  isys_analytics_dao
	 * @author  Leonard Fischer <lfischer@i-doit.com>
	 */
	public function set_simulation_objects (array $p_objects)
	{
		$this->m_simulation_objects = $p_objects;

		return $this;
	}

    /**
     * @return array
     */
    public function get_simulation_objects ()
    {
        return $this->m_simulation_objects;
    }

    /**
     * This method will recursively walk through every object and follow all relations.
     *
     * @param   isys_tree_node_explorer $p_parent_node
     * @param   integer                 $p_priority
     * @param   array                   $p_relation_filter @todo REMOVE
     * @param   array                   $p_obj_type_filter @todo REMOVE
     * @param   boolean                 $p_by_master
     * @param   integer                 $p_level
     *
     * @throws  \idoit\Exception\OutOfMemoryException
     * @throws  isys_exception_database
     * @throws  isys_exception_general
     * @author  Dennis Stücken <dstuecken@i-doit.de>
     */
	public function relation_walk (&$p_parent_node, $p_priority = null, array $p_relation_filter = array(), array $p_obj_type_filter = array(), $p_by_master = false, $p_level = null)
	{
        if ($p_level !== null && is_numeric($p_level))
        {
            if ($p_level > 0)
            {
                $p_level--;
            }
            else
            {
                return;
            }
        }

        // Retrieve all related relations and objects, which do not match the filters.
        $l_res = $this->get_object_relations_by_filters($p_parent_node->get_id(), $p_by_master, $p_priority, $p_relation_filter, $p_obj_type_filter);

		if ($l_res->count())
		{
			while ($l_row = $l_res->get_row())
			{
                $this->memoryCount ++;

                if ($this->memoryCount === 1000)
                {
                    // 1000 iterations will use about 4MB...
                    $this->memoryHelper->outOfMemoryBreak(4096000);

                    $this->memoryCount = 0;
                }

				// Check if this node is already attached anywhere.
				if (!isset($this->m_node_cache[$l_row['isys_obj__id']]) || $l_row['isys_obj__isys_obj_type__id'] == C__OBJTYPE__IT_SERVICE)
				{
					$this->m_node_cache[$l_row['isys_obj__id']] = $this->format_node($l_row['isys_obj__id'], $l_row);

					// Iterate relations
					$this->relation_walk(
						$this->m_node_cache[$l_row['isys_obj__id']],
						$p_priority,
						$p_relation_filter,
						$p_obj_type_filter,
						$p_by_master,
                        $p_level
					);

					$p_parent_node->add($this->m_node_cache[$l_row['isys_obj__id']]);
				}
				elseif (!$p_parent_node->has($l_row['isys_obj__id']))
                {
                    // If true, only attach the Name of that node here for not producing any loops.
                    $p_parent_node->add($this->format_node($l_row['isys_obj__id'])->set_name($l_row['isys_obj__title'] . ' [R]'), $l_row);
                }

				if (!isset($this->m_node_cache[$l_row['isys_catg_relation_list__isys_obj__id']]))
				{
					$this->m_node_cache[$l_row['isys_catg_relation_list__isys_obj__id']] = $this->format_node($l_row['isys_catg_relation_list__isys_obj__id'], $l_row);

					// Iterate relations
					$this->relation_walk(
						$this->m_node_cache[$l_row['isys_catg_relation_list__isys_obj__id']],
						$p_priority,
						$p_relation_filter,
						$p_obj_type_filter,
						$p_by_master,
                        $p_level
					);

					$p_parent_node->add($this->m_node_cache[$l_row['isys_catg_relation_list__isys_obj__id']]);
				}
			}

            // Free the result to save memory.
            $l_res->free_result();
		}
	}

    /**
     * Method for retrieving all relations which meet the given filter criteria.
     *
     * @param   integer $p_obj_id
     * @param   boolean $p_by_master
     * @param   integer $p_priority
     * @param   array   $p_relation_filter
     * @param   array   $p_obj_type_filter
     *
     * @return  isys_component_dao_result
     * @throws  isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
	public function get_object_relations_by_filters ($p_obj_id, $p_by_master = false, $p_priority = null, array $p_relation_filter = array(), array $p_obj_type_filter = array())
	{
		$p_obj_id = $this->convert_sql_id($p_obj_id);
		$l_status = $this->convert_sql_int(C__RECORD_STATUS__NORMAL);
		$l_cmdb_status = $this->prepare_in_condition(array(C__CMDB_STATUS__IDOIT_STATUS, C__CMDB_STATUS__IDOIT_STATUS_TEMPLATE), true);

		// We select all relationships, which inherit a certain object.
		// We only select relation-, master- and slave-objects which are no templates and have the "normal" status.
		$l_sql = 'SELECT isys_catg_relation_list.*, ' . ($p_by_master ? 'slave.*' : 'master.*') . ' FROM isys_catg_relation_list
			LEFT JOIN isys_obj AS master ON master.isys_obj__id = isys_catg_relation_list__isys_obj__id__master
			LEFT JOIN isys_obj AS slave ON slave.isys_obj__id = isys_catg_relation_list__isys_obj__id__slave
			LEFT JOIN isys_obj AS relation ON relation.isys_obj__id = isys_catg_relation_list__isys_obj__id
			WHERE isys_catg_relation_list__status = ' . $l_status . '
			AND relation.isys_obj__status = ' . $l_status . '
			AND relation.isys_obj__isys_cmdb_status__id ' . $l_cmdb_status . '
			AND master.isys_obj__status = ' . $l_status . '
			AND master.isys_obj__isys_cmdb_status__id ' . $l_cmdb_status . '
			AND slave.isys_obj__status = ' . $l_status . '
			AND slave.isys_obj__isys_cmdb_status__id ' . $l_cmdb_status . '
			AND (' . ($p_by_master ? 'master' : 'slave') . '.isys_obj__id = ' . $p_obj_id . '
			OR isys_catg_relation_list__isys_obj__id = ' . $p_obj_id . ')';

		if ($p_priority !== null)
		{
			$l_sql .= ' AND isys_catg_relation_list__isys_weighting__id < ' . $this->convert_sql_id($p_priority);
		} // if

		if (count($p_relation_filter))
		{
			$l_sql .= ' AND isys_catg_relation_list__isys_relation_type__id ' . $this->prepare_in_condition($p_relation_filter, true);
		} // if

		if (count($p_obj_type_filter))
		{
			$l_sql .= ' AND master.isys_obj__isys_obj_type__id ' . $this->prepare_in_condition($p_obj_type_filter, true) . '
				AND slave.isys_obj__isys_obj_type__id ' . $this->prepare_in_condition($p_obj_type_filter, true);
		} // if

		return $this->retrieve($l_sql . ';');
	}


	/**
	 * Method for setting options - will be used in the relation_walk method or to format the nodes.
	 *
	 * @param   array  $p_options
	 * @return  isys_analytics_dao
	 * @author  Leonard Fischer <lfischer@i-doit.com>
	 */
	public function set_options (array $p_options)
	{
		$this->m_options = $p_options + $this->m_options;

		return $this;
	}


	/**
	 * Method for retrieving the options (or just a given one, with a default value).
	 *
	 * @param   string  $p_key
	 * @param   mixed   $p_default
	 * @return  array
	 * @author  Leonard Fischer <lfischer@i-doit.com>
	 */
	public function get_options ($p_key = null, $p_default = null)
	{
		if ($p_key === null)
		{
			return $this->m_options;
		}

		return (isset($this->m_options[$p_key]) ? $this->m_options[$p_key] : $p_default);
	}

    /**
     * Method for formatting the single object nodes.
     *
     * @param   integer $p_obj_id
     * @param   array   $p_relationData
     *
     * @return  isys_tree_node_explorer
     * @throws  isys_exception_database
     * @throws  isys_exception_general
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function format_node($p_obj_id, $p_relationData = [])
    {
        $language = isys_application::instance()->container->get('language');

        $l_obj_data = $this->retrieve('SELECT * FROM isys_obj
			LEFT JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
			LEFT JOIN isys_cmdb_status ON isys_cmdb_status__id = isys_obj__isys_cmdb_status__id
			WHERE isys_obj__id = ' . $this->convert_sql_id($p_obj_id) . ';')
            ->get_row();

        $l_options = array_replace_recursive($this->m_options, [
            'id'                => $p_obj_id,
            'is_simulation_obj' => in_array($p_obj_id, $this->m_simulation_objects)
        ]);

        if (isset($p_relationData['isys_catg_relation_list__isys_relation_type__id'])) {
            $l_relationdata = [
                'title'           => $this->obj_get_title_by_id_as_string($p_relationData['isys_catg_relation_list__isys_obj__id']),
                'id'              => $p_relationData['isys_catg_relation_list__isys_obj__id'],
                'type'            => $p_relationData['isys_catg_relation_list__isys_relation_type__id'],
                'master'          => $p_relationData['isys_catg_relation_list__isys_obj__id__master'],
                'slave'           => $p_relationData['isys_catg_relation_list__isys_obj__id__slave'],
                'its'             => $p_relationData['isys_catg_relation_list__isys_obj__id__itservice'],
                'weighting'       => $p_relationData['isys_catg_relation_list__isys_weighting__id'],
                'cluster_members' => count(isys_cmdb_dao_list_catg_cluster_members::instance($this->m_db)->get_result(null, $p_relationData['isys_catg_relation_list__isys_obj__id__slave']))
            ];
        } else {
            $l_relationdata = [];
        }

        $l_obj_cat = $l_itservice_name = '';

        if ($l_obj_data['isys_obj__isys_obj_type__id'] == C__OBJTYPE__IT_SERVICE) {
            $l_row = isys_factory_cmdb_category_dao::get_instance('isys_cmdb_dao_category_g_its_type', $this->m_db)
                ->get_data(null, $p_obj_id)
                ->get_row();

            if ($l_row && !empty($l_row['isys_its_type__title'])) {
                $l_obj_cat = '(' . $language->get('LC__CMDB__CATG__ITS_TYPE') . ': ' . $language->get($l_row['isys_its_type__title']) . ')';
                $l_itservice_name = $language->get($l_row['isys_its_type__title']);
            }
        }

        // Find out if the current object is part of a parallel relation.
        $l_sql = 'SELECT isys_cats_relpool_list__isys_obj__id
			FROM isys_cats_relpool_list_2_isys_obj AS rel
			LEFT JOIN isys_cats_relpool_list AS relpool ON relpool.isys_cats_relpool_list__id = rel.isys_cats_relpool_list__id
			WHERE isys_obj__id = ' . $this->convert_sql_id($p_obj_id) . ';';
        $l_parallel_res = $this->retrieve($l_sql);
        $l_parallel_relation = null;

        if (count($l_parallel_res)) {
            $l_parallel_relation = [];

            while ($l_parallel_row = $l_parallel_res->get_row()) {
                $l_parallel_relation[] = $l_parallel_row['isys_cats_relpool_list__isys_obj__id'];
            }
        }

        $l_icon = isys_application::instance()->www_path . 'images/icons/silk/page_white.png';

        if (!empty($l_obj_data['isys_obj_type__icon'])) {
            if (strpos($l_obj_data['isys_obj_type__icon'], '/') !== false) {
                $l_icon = $l_obj_data['isys_obj_type__icon'];
            } else {
                $l_icon = isys_application::instance()->www_path . 'images/tree/' . $l_obj_data['isys_obj_type__icon'];
            }
        }

        return new isys_tree_node_explorer([
                'id'       => $l_options['id'],
                'name'     => trim($l_obj_data['isys_obj__title']),
                'children' => [],
                'data'     => [
                    C__CMDB__GET__OBJECT    => $p_obj_id,
                    'objTypeID'             => $l_obj_data['isys_obj__isys_obj_type__id'],
                    'icon'                  => $l_icon,
                    'image'                 => isys_application::instance()->www_path . 'images/objecttypes/' . ($l_obj_data['isys_obj_type__obj_img_name'] ?: 'empty.png'),
                    'objectType'            => $language->get($l_obj_data['isys_obj_type__title']),
                    'relation'              => $l_relationdata,
                    'subNodes'              => $l_options['children_count'],
                    'color'                 => isset($l_obj_data['isys_obj_type__color']) ? '#' . $l_obj_data['isys_obj_type__color'] : false,
                    'statusColor'           => $l_obj_data['isys_cmdb_status__color'],
                    'cmdb_status'           => $language->get($l_obj_data['isys_cmdb_status__title']),
                    'objectCategory'        => $l_obj_cat,
                    'itserviceType'         => $l_itservice_name,
                    'is_simulation_obj'     => $l_options['is_simulation_obj'],
                    'parallel_relation_obj' => $l_parallel_relation
                ]
            ]);
    }
}
