<?php

namespace idoit\Module\Analytics\Controller;

use Exception;
use idoit\Module\Analytics\Model\Profile;
use idoit\Module\Analytics\Model\Util;
use isys_analytics_reports_dataquality;
use isys_application as Application;
use isys_auth;
use isys_auth_report;
use isys_exception_general;
use isys_format_json as JSON;
use isys_module_analytics;
use isys_notify;
use isys_register as Register;
use isys_report_dao;

/**
 * Class Dataquality
 *
 * @package idoit\Module\Analytics\Controller
 */
class Dataquality extends Main
{
    protected $response;

    public function pre()
    {
        header('Content-Type: application/json');

        $this->response = [
            'success' => true,
            'data'    => null,
            'message' => null
        ];
    }

    /**
     * @param Register $request
     */
    public function loadProfiles(Register $request)
    {
        try {
            $this->response['data'] = [];

            $currentUser = $this->getDi()->get('session')->get_user_id();
            $result = Profile::instance($this->getDi()->get('database'))->getProfiles();

            while ($row = $result->get_row()) {
                // Only display global and private profiles (of the current user).
                if ($row['ownerObjectId'] > 0 && $row['ownerObjectId'] != $currentUser) {
                    continue;
                }

                try {
                    $row['data'] = JSON::decode($row['data']);
                } catch (Exception $e) {
                    isys_notify::warning($e->getMessage(), ['sticky' => true]);
                    $row['data'] = [];
                }

                $this->response['data'][] = $row;
            }

            usort($this->response['data'], function ($a, $b) {
                return strnatcasecmp($a['title'], $b['title']);
            });
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Register $request
     */
    public function loadReports(Register $request)
    {
        try {
            $this->response['data'] = [];

            if (!class_exists(isys_auth_report::class) || !class_exists(isys_report_dao::class)) {
                throw new isys_exception_general('The necessary report classes could not be found.');
            }

            $result = isys_report_dao::instance($this->getDi()->get('database'))->get_reports(
                null,
                isys_auth_report::instance()->get_allowed_reports()
            );

            while ($row = $result->get_row()) {
                 $this->response['data'][] = [
                    'id' => (int)$row['isys_report__id'],
                    'title' => $row['isys_report__title']
                ];
            }

            usort($this->response['data'], function ($a, $b) {
                return strnatcasecmp($a['title'], $b['title']);
            });
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Register $request
     */
    public function checkReport(Register $request)
    {
        try {
            if (!class_exists(isys_report_dao::class)) {
                throw new isys_exception_general('The necessary report classes could not be found.');
            }

            $dao = isys_report_dao::instance($this->getDi()->get('database'));

            $report = $dao->get_report((int) $request->get('POST')['reportId']);
            $result = $dao->retrieve($report['isys_report__query']);

            $this->response['data'] = [
                'count' => count($result),
                'compatible' => $result->get_row_value('__id__') !== null
            ];
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Register $request
     */
    public function loadCategoriesByObjectType(Register $request)
    {
        try {
            $this->response['data'] = [];

            $post = $request->get('POST');
            $language = $this->getDi()->get('language');

            $utilDao = Util::instance($this->getDi()->get('database'));

            $result = $utilDao->getCategoriesByObjectType($post['objectTypeConst']);

            $blockedCategories = $utilDao->getBlockedCategories();

            while ($row = $result->get_row()) {
                // Skip blocked categories.
                if (in_array($row['const'], $blockedCategories['g'], true) || in_array($row['const'], $blockedCategories['s'], true)) {
                    continue;
                }

                $row['title'] = $language->get($row['title']);
                $row['type'] = $language->get($row['type']);

                $this->response['data'][] = $row;
            }

            usort($this->response['data'], function($a, $b) {
                return strnatcasecmp($a['title'], $b['title']);
            });
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Register $request
     */
    public function saveProfiles(Register $request)
    {
        try {
            $this->response['data'] = [];

            $post = $request->get('POST');

            $auth = isys_module_analytics::getAuth();
            $data = JSON::decode($post['data']);
            $profileModel = Profile::instance($this->getDi()->get('database'));

            foreach ($data as $profile) {
                $id = $profile['id'];
                $profile['isys_obj__id'] = $profile['ownerObjectId'];

                if (strpos($id, 'new-') === 0) {
                    $id = null;
                }

                unset ($profile['id']);

                $profileModel->saveProfile($id, $profile);
            }

            // Prepare the profiles to be deleted (only delete profiles that actually exist).
            $deletedProfiles = array_filter((array)JSON::decode($post['deletedData']), function($profileId) use ($auth) {
                return strpos($profileId, 'new-') === false && $auth->is_allowed_to(isys_auth::DELETE, 'DATAQUALITY/' . $profileId);
            });

            if (count($deletedProfiles)) {
                $profileModel->deleteProfiles($deletedProfiles);
            }
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    public function post()
    {
        echo JSON::encode($this->response);
        die;
    }
}
