<?php

use \idoit\Module\Analytics\Helper\Simulation;

/**
 * i-doit
 *
 * Auth: Class for i-doit authorization rules.
 *
 * @package     Modules
 * @subpackage  Analytics
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */

class isys_auth_analytics extends isys_auth implements isys_auth_interface
{
    /**
     * Container for singleton instance
     *
     * @var isys_auth_analytics
     */
    private static $instance;

    /**
     * Method for returning the available auth-methods. This will be used for the GUI.
     *
     * @return  array
     */
    public function get_auth_methods()
    {
        return [
            'analytics_simulation' => [
                'title'  => 'LC__MODULE__ANALYTICS__AUTH__SIMULATION',
                'type'   => 'analytics_simulations',
                'rights' => [isys_auth::VIEW]
            ],
            'dataquality'          => [
                'title'  => 'LC__MODULE__ANALYTICS__AUTH__DATAQUALITY_PROFILES',
                'type'   => 'dataquality',
                'rights' => [isys_auth::VIEW, isys_auth::EDIT, isys_auth::DELETE]
            ]
        ];
    }

    /**
     * Get ID of related module
     *
     * @return int
     */
    public function get_module_id()
    {
        return C__MODULE__ANALYTICS;
    }

    /**
     * Get title of related module
     *
     * @return string
     */
    public function get_module_title()
    {
        return 'LC__MODULE__ANALYTICS';
    }

    /**
     * Retrieve singleton instance of authorization class
     *
     * @return isys_auth_analytics
     */
    public static function instance()
    {
        // If the DAO has not been loaded yet, we initialize it now.
        if (self::$m_dao === null) {
            self::$m_dao = new isys_auth_dao(isys_application::instance()->container->get('database'));
        }

        if (self::$instance === null) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    /**
     * Method for retrieving the "parameter" in the configuration GUI. Gets called generically by "ajax()" method.
     *
     * @see     isys_module_auth->ajax_retrieve_parameter();
     *
     * @param   string  $method
     * @param   string  $parameter
     * @param   integer $counter
     * @param   boolean $editMode
     * @param   boolean $comboParameter This parameter is used, when more than one box is displayed at once (category in object, ...).
     *
     * @return  array
     * @throws  Exception
     */
    public function retrieve_parameter($method, $parameter, $counter, $editMode = false, $comboParameter = false)
    {
        $return = [
            'html'    => '',
            'method'  => $method,
            'param'   => $parameter,
            'counter' => $counter
        ];

        $database = isys_application::instance()->container->get('database');
        $template = isys_application::instance()->container->get('template');

        switch ($method) {
            case 'analytics_simulations':
                $simulations = Simulation::getAll();

                foreach ($simulations as &$simulation) {
                    $simulation = $simulation->getTitle();
                }

                $dialog = new isys_smarty_plugin_f_dialog();
                $smartyParameters = [
                    'name'              => 'auth_param_form_' . $counter,
                    'p_bInfoIconSpacer' => 0,
                    'p_strClass'        => 'input-small',
                    'p_arData'          => $simulations,
                    'p_editMode'        => $editMode,
                    'p_strSelectedID'   => $parameter
                ];

                $return['html'] = isys_glob_utf8_encode($dialog->navigation_edit($template, $smartyParameters));
                break;

            case 'dataquality':
                $data = [];
                $result = isys_analytics_dao_dataquality::instance($database)
                    ->get_profiles();

                if (count($result)) {
                    while ($row = $result->get_row()) {
                        $data[$row['isys_analytics_dataquality_profiles__id']] = $row['isys_analytics_dataquality_profiles__title'];
                    }
                }

                $dialog = new isys_smarty_plugin_f_dialog();
                $smartyParameters = [
                    'name'              => 'auth_param_form_' . $counter,
                    'p_bInfoIconSpacer' => 0,
                    'p_strClass'        => 'input-small',
                    'p_arData'          => $data,
                    'p_editMode'        => $editMode,
                    'p_strSelectedID'   => $parameter
                ];

                $return['html'] = $dialog->navigation_edit($template, $smartyParameters);
                break;
        }

        return $return;
    }

    /**
     * Method for checking, if the user is allowed to view / execute a simulation.
     *
     * @param $right
     * @param $parameter
     *
     * @return bool
     * @throws isys_exception_auth
     */
    public function analytics_simulation($right, $parameter)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        $language = isys_application::instance()->container->get('language');

        return $this->generic_right($right, 'analytics_simulation', $parameter, new isys_exception_auth($language->get('LC__MODULE__ANALYTICS__AUTH__SIMULATION_EXCEPTION')));
    }

    /**
     * Method for checking, if the user is allowed to administrate dataquality profiles (add, delete, publish).
     *
     * @param   integer $right
     * @param   string  $parameter
     *
     * @return  boolean
     * @throws  isys_exception_auth
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function dataquality($right, $parameter)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        $language = isys_application::instance()->container->get('language');

        return $this->generic_right($right, 'dataquality', $parameter, new isys_exception_auth($language->get('LC__MODULE__ANALYTICS__AUTH__DATAQUALITY_PROFILES_EXCEPTION')));
    }
}
