<?php

use idoit\AddOn\AuthableInterface;
use idoit\Module\Analytics\Helper\Simulation;

/**
 * i-doit
 *
 * Analytics module class.
 *
 * @package     modules
 * @subpackage  analytics
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_module_analytics extends isys_module implements AuthableInterface
{
    // Define, if this module shall be displayed in the named menus.
    const DISPLAY_IN_MAIN_MENU   = true;
    const DISPLAY_IN_SYSTEM_MENU = false;

    /**
     * @var bool
     */
    protected static $m_licenced = false;

    /**
     * Initializes the module.
     *
     * @param   isys_module_request & $p_req
     *
     * @return  isys_module_analytics
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function init(isys_module_request &$p_req)
    {
        return $this;
    }

    /**
     * This method builds the tree for the menu.
     *
     * @param   isys_component_tree $p_tree
     * @param   boolean             $p_system_module
     * @param   integer             $p_parent
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     * @see     isys_module_cmdb->build_tree();
     */
    public function build_tree(isys_component_tree $p_tree, $p_system_module = true, $p_parent = null)
    {
        if ($p_system_module === false) {
            $l_root = $p_tree->add_node(
                C__MODULE__ANALYTICS . 0,
                $p_parent,
                $this->language->get('LC__MODULE__ANALYTICS'),
                '',
                '',
                self::getWwwPath() . 'assets/add-on-icon.svg'
            );

            $l_cnt = 1;

            $l_auth = isys_auth_analytics::instance();
            $l_simulations = Simulation::getAll();
            $reportClassName = $_GET[C__ANALYTICS__REPORT] ?: 'isys_analytics_reports_impact_simulation';

            // Here we create a menu item for each found report.
            foreach ($l_simulations as $l_classname => $l_simulation) {
                if (method_exists($l_simulation, 'display_in_tree') && !$l_simulation->display_in_tree()) {
                    continue;
                }

                $l_last_node = $p_tree->add_node(
                    C__MODULE__ANALYTICS . $l_cnt,
                    $l_root,
                    $this->language->get($l_simulation->getTitle()),
                    isys_helper_link::create_url([
                        C__GET__MODULE_ID    => C__MODULE__ANALYTICS,
                        C__GET__TREE_NODE    => C__MODULE__ANALYTICS . $l_cnt,
                        C__ANALYTICS__REPORT => $l_classname,
                    ]),
                    '',
                    isys_application::instance()->www_path . 'images/' . $l_simulation->getIcon(),
                    (int)($reportClassName === $l_classname),
                    '',
                    '',
                    $l_auth->is_allowed_to(isys_auth::VIEW, 'analytics_simulation/' . $l_classname)
                );

                if (method_exists($l_simulation, 'build_tree')) {
                    $l_simulation->build_tree($p_tree, $l_last_node);
                }

                $l_cnt++;
            }
        }
    }

    /**
     * Build breadcrumb navifation
     *
     * @param &$p_gets
     *
     * @return array|null
     */
    public function breadcrumb_get(&$p_gets)
    {
        $l_return = [];
        $getParameter = $_GET;
        $reportClassName = $getParameter[C__ANALYTICS__REPORT] ?: 'isys_analytics_reports_impact_simulation';

        if (!is_a($reportClassName, isys_analytics_reports::class, true)) {
            return $l_return;
        }

        if (!empty($reportClassName) && class_exists($reportClassName)) {
            $l_report = isys_factory::get_instance($reportClassName);

            if (method_exists($reportClassName, 'breadcrumb_get')) {
                $l_return = $l_report->breadcrumb_get($getParameter);
            } else {
                $l_return[] = [
                    $this->language->get($l_report->getTitle()) => [
                        C__GET__MODULE_ID    => C__MODULE__ANALYTICS,
                        C__GET__TREE_NODE    => $_GET[C__GET__TREE_NODE],
                        C__ANALYTICS__REPORT => $getParameter[C__ANALYTICS__REPORT]
                    ]
                ];
            }
        }

        return $l_return;
    }

    /**
     * Start method.
     *
     * @return $this|isys_module|isys_module_interface
     * @throws isys_exception_auth
     * @throws isys_exception_general
     * @throws isys_exception_licence
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function start()
    {
        // Check for the licence.
        if (!self::is_licenced()) {
            throw new isys_exception_licence($this->language->get('LC__LICENCE__NO_MODULE_LICENCE', [$this->language->get('LC__MODULE__ANALYTICS')]), 1);
        }

        // @see  ANALYSE-68  This will happen with rewritten URLs.
        if (strpos($_SERVER['REQUEST_URI'], '/analytics/') !== false) {
            return $this;
        }

        $template = isys_application::instance()->container->get('template')
            ->assign('analyticsPath', self::getPath());

        $reportClassName = $_GET[C__ANALYTICS__REPORT] ?: 'isys_analytics_reports_impact_simulation';

        if (!is_a($reportClassName, isys_analytics_reports::class, true)) {
            throw new isys_exception_general($this->language->get('LC__MODULE__ANALYTICS__EXCEPTION__NO_REPORT_FOUND', $reportClassName));
        }

        if (isys_core::is_ajax_request()) {
            $this->ajaxRequest($reportClassName);
        }

        // Build the module tree, but only if we are not in the system-module.
        if ($_GET[C__GET__MODULE_ID] != C__MODULE__SYSTEM) {
            $l_tree = isys_module_request::get_instance()
                ->get_menutree();

            $this->build_tree($l_tree, false, -1);

            $template->assign('menu_tree', $l_tree->process($_GET[C__GET__TREE_NODE]));
        }

        if (!class_exists($reportClassName)) {
            throw new isys_exception_general($this->language->get('LC__MODULE__ANALYTICS__EXCEPTION__NO_REPORT_FOUND', $reportClassName));
        }

        self::getAuth()
            ->check(isys_auth::VIEW, 'analytics_simulation/' . $reportClassName);

        (new $reportClassName)->start();

        return $this;
    }

    /**
     * This method will check, if the current request is a AJAX request and (if so) process the necessary logic.
     *
     * @param   string $className
     */
    private function ajaxRequest($className)
    {
        // We set the header information because we don't accept anything than JSON.
        header('Content-Type: application/json');

        $response = [
            'success' => true,
            'data'    => null,
            'message' => null
        ];

        try {
            if (!class_exists($className)) {
                throw new isys_exception_general($this->language->get('LC__MODULE__ANALYTICS__EXCEPTION__NO_CLASS_FOUND', $className));
            }

            $response['data'] = (new $className)->ajax_request();
        } catch (isys_exception $e) {
            $response['success'] = false;
            $response['message'] = $e->getMessage();
        }

        echo isys_format_json::encode($response);
        die;
    }

    /**
     * Method for adding links to the "sticky" category bar.
     *
     * @param  isys_component_template $template
     * @param  string                  $variableName
     * @param  integer                 $objectId
     * @param  integer                 $objectTypeId
     */
    public static function process_menu_tree_links($template, $variableName, $objectId, $objectTypeId)
    {
        $language = isys_application::instance()->container->get('language');

        $linkData = [
            'title' => $language->get('LC__MODULE__ANALYTICS'),
            'link'  => isys_helper_link::create_url([
                C__GET__MODULE_ID    => C__MODULE__ANALYTICS,
                C__ANALYTICS__REPORT => 'isys_analytics_reports_impact_simulation',
                C__CMDB__GET__OBJECT => $objectId
            ]),
            'icon'  => isys_application::instance()->www_path . 'images/axialis/basic/graph-bars.svg'
        ];

        $template->append($variableName, ['analytics' => $linkData], true);
    }

    /**
     * @return isys_auth_analytics
     */
    public static function getAuth()
    {
        return isys_auth_analytics::instance();
    }
}
