<?php

/**
 * i-doit
 *
 * Livestatus (impact simulation) report.
 *
 * @package     modules
 * @subpackage  analytics
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.4.3
 */
class isys_analytics_reports_livestatus extends isys_analytics_reports
{
    /**
     * This constant holds the name of the report.
     *
     * @var  string
     */
    const TITLE = 'LC__MODULE__ANALYTICS__LIVE_IMPACT';

    /**
     * This constant holds the icon-path of the report.
     *
     * @var  string
     */
    const ICON = 'axialis/hardware-network/computer-activity.svg';

    /**
     * @var  isys_analytics_dao
     */
    private $dao;

    /**
     * Constructor
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function __construct()
    {
        parent::__construct();

        $this->dao = isys_analytics_dao::instance($this->database);
    }

    /**
     * This method will return a boolean value, if this report is allowed to be displayed inside the menu tree.
     *
     * @return  boolean
     */
    public function displayInNavigation(): bool
    {
        return (defined('C__MODULE__CHECK_MK') && class_exists('isys_module_check_mk'));
    }

    /**
     * Method for preparing the visual report view output.
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function start()
    {
        global $index_includes;

        $l_get = $_GET;
        $l_rules = $l_filters = $l_monitoring = [];

        // Remove some unnecessary parameters for the Ajax URL.
        unset($l_get[C__GET__MAIN_MENU__NAVIGATION_ID], $l_get[C__CMDB__GET__EDITMODE], $l_get[C__GET__TREE_NODE]);

        $l_itservice_filter_url = isys_helper_link::create_url([
            C__GET__MODULE_ID     => C__MODULE__ITSERVICE,
            C__GET__SETTINGS_PAGE => 'filter-config'
        ]);

        $l_itservice_filters = isys_itservice_dao_filter_config::instance($this->database)
            ->get_data();

        if (is_array($l_itservice_filters) && count($l_itservice_filters)) {
            foreach ($l_itservice_filters as $l_filter) {
                $l_filters[$l_filter['isys_itservice_filter_config__id']] = $l_filter['isys_itservice_filter_config__title'];
            } // foreach
        } // if

        // Both the service- and the result filter are based on the same data.
        $l_rules['impact_itservice_result_filter']['p_arData'] = $l_rules['impact_itservice_filter']['p_arData'] = $l_filters;

        $l_monitoring_res = isys_monitoring_dao_hosts::instance($this->database)
            ->get_data(null, C__MONITORING__TYPE_LIVESTATUS, true);

        if (count($l_monitoring_res)) {
            while ($l_row = $l_monitoring_res->get_row()) {
                if ($l_row['isys_monitoring_hosts__active']) {
                    $l_monitoring[$l_row['isys_monitoring_hosts__id']] = $l_row['isys_monitoring_hosts__title'] . ' (' . $l_row['isys_monitoring_hosts__type'] . ')';
                }
            }
        }

        $l_rules['monitoring_instances']['p_arData'] = $l_monitoring;

        isys_component_template_navbar::getInstance()
            ->append_button('LC__MODULE__ANALYTICS__LIVE_IMPACT__START', 'start-simulation', [
                'tooltip'    => 'LC__MODULE__ANALYTICS__LIVE_IMPACT__START',
                'icon'       => 'axialis/basic/gear.svg',
                'js_onclick' => ';',
                'navmode'    => 'simulation',
            ]);

        $l_cluster = $this->dao->get_object_type(C__OBJTYPE__CLUSTER);

        $this->template->activate_editmode()
            ->assign('content_title', $this->language->get($this->getTitle()))
            ->assign('filter_description', $this->language->get('LC__MODULE__ANALYTICS__FILTERS_DESCRIPTION', $l_itservice_filter_url))
            ->assign('ajax_url', isys_helper_link::create_url($l_get))
            ->assign('cluster_color', isys_helper_color::unifyHexColor((string)$l_cluster['isys_obj_type__color']))
            ->assign('use_monitoring', true)
            ->assign('monitoring_active', (count($l_monitoring) > 0))
            ->assign('title', $this->language->get($this->getTitle()))
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules);

        $index_includes['contentbottomcontent'] = __DIR__ . '/isys_analytics_reports_impact_simulation_view.tpl';

        return $this;
    } // function

    /**
     * This method will be called by the framework, to process ajax requests.
     *
     * @todo    Check why this method isn't returning anything but rendering directly!
     * @return  array
     */
    public function ajax_request()
    {
        // We set the header information because we don't accept anything than JSON.
        header('Content-Type: application/json');

        $l_return = [
            'success' => true,
            'data'    => null,
            'message' => null
        ];

        try {
            switch ($_GET['func']) {
                default:
                case 'start-simulation':
                    $l_impact_objects = isys_format_json::decode($_POST['impact_obj']);

                    if (!is_array($l_impact_objects)) {
                        throw new isys_exception_general($this->language->get('LC__MODULE__ANALYTICS__NOTIFY__NO_IMPACT_OBJECT'));
                    } // if

                    return $this->start_simulation($l_impact_objects, (int)$_POST['filter']);

                case 'load-not-ok-objects':
                    return $this->load_not_ok_objects($_POST['monitoring']);
            } // switch
        } catch (Exception $e) {
            $l_return['success'] = false;
            $l_return['message'] = $e->getMessage();
        } // try

        echo isys_format_json::encode($l_return);
        die;
    } // function

    /**
     * Basically, this method is a wrapper for "isys_analytics_reports_impact_simulation->start_simulation()".
     *
     * @param   array   $p_impact_obj
     * @param   integer $p_filter
     *
     * @throws  isys_exception_general
     * @return  array
     */
    public function start_simulation(array $p_impact_obj, $p_filter = null)
    {
        return (new isys_analytics_reports_impact_simulation)->start_simulation($p_impact_obj, $p_filter);
    } // function

    /**
     * This method will return services, which states which are not "OK".
     *
     * @param   mixed $p_host_id
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    protected function load_not_ok_objects($p_host_id)
    {
        $l_host_objects = [];
        $l_host_states = isys_monitoring_helper::get_host_state_info();
        $l_livestatus = isys_monitoring_livestatus::factory($p_host_id);

        // At first we load the "not ok" hosts.
        $l_not_ok_hosts = $l_livestatus->query(["GET hosts", "Filter: state > 0", "Columns: name plugin_output state"], true);

        foreach ($l_not_ok_hosts as $l_host) {
            $l_object = isys_monitoring_helper::get_objects_by_hostname($p_host_id, $l_host[0]);

            if ($l_object['isys_obj__status'] != C__RECORD_STATUS__NORMAL) {
                continue;
            } // if

            $l_host_objects[$l_object['isys_obj__id']] = [
                'obj_id'         => $l_object['isys_obj__id'],
                'obj_title'      => $l_object['isys_obj__title'],
                'obj_type_id'    => $l_object['isys_obj_type__id'],
                'obj_type_title' => $this->language->get($l_object['isys_obj_type__title']),
                'state'          => $l_host_states[$l_host[2]],
                'state_info'     => $l_host[1]
            ];
        } // foreach

        // LF: I'm not exactly sure why, but the socket needs a refresh, after data has been received.
        $l_not_ok_services = $l_livestatus->disconnect()
            ->connect()
            ->query(["GET services", "Filter: state > 0", "Columns: host_name description plugin_output state"], true);

        $l_service_objects = $this->load_related_objects_by_service_data($p_host_id, $l_not_ok_services);

        return [
            'hosts'    => $l_host_objects,
            'services' => $l_service_objects
        ];
    } // function

    /**
     * This method will load all objects, which are affected by the "not OK" services.
     *
     * @param   integer $p_host_id
     * @param   array   $p_service_data
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    protected function load_related_objects_by_service_data($p_host_id, $p_service_data)
    {
        $l_return = [];

        if (!class_exists('isys_cmdb_dao_category_g_cmk_host_service')) {
            return $l_return;
        }

        $l_livestatus = isys_monitoring_livestatus::factory($p_host_id);
        $l_service_states = isys_monitoring_helper::get_state_info();
        $l_host_states = isys_monitoring_helper::get_host_state_info();
        $l_host_service_dao = isys_cmdb_dao_category_g_cmk_host_service::instance($this->database);
        $l_app_dao = isys_cmdb_dao_category_g_application::instance($this->database);

        foreach ($p_service_data as $l_service) {
            $l_host = $l_host_service_dao->get_objects_by_inherited_service($l_service[1], $l_service[0]);

            // Get the objects, which "inherit" this service.
            if (count($l_host)) {
                $l_host = current($l_host);

                if (!isset($l_return[$l_host['isys_obj__id']])) {
                    $l_state = $l_livestatus->disconnect()
                        ->connect()
                        ->query(['GET hosts', 'Filter: host_name = ' . $l_service[0], 'Columns: state']);

                    $l_return[$l_host['isys_obj__id']] = [
                        'obj_id'            => $l_host['isys_obj__id'],
                        'obj_title'         => $l_host['isys_obj__title'],
                        'obj_type_id'       => $l_host['isys_obj_type__id'],
                        'obj_type_title'    => $this->language->get($l_host['isys_obj_type__title']),
                        'hostname'          => $l_service[0],
                        'state'             => $l_host_states[$l_state[0][0]],
                        'host_service'      => [],
                        'inherited_service' => [],
                    ];
                } // if

                $l_app = $l_app_dao->get_data($l_host['isys_catg_cmk_host_service_list__application__id'], null, '', null, C__RECORD_STATUS__NORMAL)
                    ->get_row();

                $l_return[$l_host['isys_obj__id']]['inherited_service'][] = [
                    'app_id'         => $l_app['isys_obj__id'],
                    'app_title'      => $l_app['isys_obj__title'],
                    'app_type_id'    => $l_app['isys_obj_type__id'],
                    'app_type_title' => $this->language->get($l_app['isys_obj_type__title']),
                    'service'        => $l_service[1],
                    'state'          => $l_service_states[$l_service[3]],
                    'state_info'     => $l_service[2]
                ];
            } // if

            $l_host = $l_host_service_dao->get_object_by_service($l_service[1], $l_service[0]);

            if (count($l_host)) {
                $l_host = current($l_host);

                if (!isset($l_return[$l_host['isys_obj__id']])) {
                    $l_state = $l_livestatus->disconnect()
                        ->connect()
                        ->query(['GET hosts', 'Filter: host_name = ' . $l_service[0], 'Columns: state']);

                    $l_return[$l_host['isys_obj__id']] = [
                        'obj_id'            => $l_host['isys_obj__id'],
                        'obj_title'         => $l_host['isys_obj__title'],
                        'obj_type_id'       => $l_host['isys_obj_type__id'],
                        'obj_type_title'    => $this->language->get($l_host['isys_obj_type__title']),
                        'state'             => $l_host_states[$l_state[0][0]],
                        'state_info'        => '',
                        'host_service'      => [],
                        'inherited_service' => [],
                    ];
                } // if

                $l_app = $l_app_dao->get_data($l_host['isys_catg_cmk_host_service_list__application__id'], null, '', null, C__RECORD_STATUS__NORMAL)
                    ->get_row();

                $l_return[$l_host['isys_obj__id']]['host_service'][] = [
                    'app_id'         => $l_app['isys_obj__id'],
                    'app_title'      => $l_app['isys_obj__title'],
                    'app_type_id'    => $l_app['isys_obj_type__id'],
                    'app_type_title' => $this->language->get($l_app['isys_obj_type__title']),
                    'service'        => $l_service[1],
                    'state'          => $l_service_states[$l_service[3]],
                    'state_info'     => $l_service[2]
                ];
            } // if
        } // foreach

        return $l_return;
    } // function
} // class
