var DataQuality = Class.create({
    initialize: function ($el, options) {
        // First we identify all the elements.
        this.$el = $el;
        this.$headline = $el.previous('.obj-type-header');
        this.$infotext = $el.down('p');
        this.$progressbar = $el.down('div.progress-bar');
        this.$left_detail = $el.down('td:first-child div.detail');
        this.$right_detail = $el.down('td:last-child div.detail');
        this.$toggle_detail_button = $el.down('.toggle-detail');
        this.$reset_categories_button = $el.down('.reset-categories');

        // Then some other variables.
        this.object_type_id = this.$headline.readAttribute('data-objtype-id');
        this.data = {};
        this.chart = null;
        this.chart_width = $el.down('td:first-child').getWidth() - 20;

        // Then we set the observers.
        this.$toggle_detail_button.on('click', this.toggle_detail.bind(this));
        this.$reset_categories_button.on('click', this.reset_categories.bind(this));
        this.$right_detail.on('click', 'button.remove-cat', this.filter_category.bind(this));
        this.$right_detail.on('click', 'button.objects-without-data', this.display_objects_without_data.bind(this));
        this.$right_detail.on('click', 'button.close-object-table', this.hide_objects_without_data.bind(this));
  
        this.options = {
            categories: {}
        };
        
        Object.extend(this.options, options || {});
        
        // Finally we load the detail data.
        this.reload();
    },

    check_response: function (response) {
        if (!is_json_response(response)) {
            throw 'The request did not answer with proper JSON: ' + response.responseText;
        }

        if (!response.responseJSON.success) {
            throw response.responseJSON.message;
        }
    },

    reload: function (reset) {
        var reset_category = (reset ? 1 : 0);
        
        if (reset) {
            this.data = {};
        }

        new Ajax.Request('[{$ajax_url}]&func=load-category-data', {
            parameters: {
                obj_type_id: this.object_type_id,
                reset_categories: reset_category
            },
            onComplete: function (response) {
                try {
                    this.check_response(response);
                } catch (e) {
                    // This should only happen, if we abort the request.
                    if (response.responseJSON !== null) {
                        idoit.Notify.error(e);
                    }

                    this.$reset_categories_button.removeClassName('disabled').enable();

                    this.$infotext
                        .down('img').writeAttribute('src', window.dir_images + 'axialis/basic/button-info.svg')
                        .next('span').update('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__ABORT_REQUEST_INFO"}]');

                    return;
                }

                var json = response.responseJSON,
                    percent,
                    total = parseInt(json.data.objects) * parseInt(Object.keys(json.data.data).length),
                    rows = 0,
                    cnt = 0,
                    category;

                for (category in json.data.data) {
                    if (json.data.data.hasOwnProperty(category)) {
                        var category_type = category.split('_');
                        
                        if (category_type.length === 3) {
                            category_type = category_type[0] + '_' + category_type[1];
                        } else {
                            category_type = category_type[0];
                        }
                        
                        this.data[this.options.categories[category].const] = {
                            title: this.options.categories[category].title,
                            const: this.options.categories[category].const,
                            total: json.data.objects,
                            rows: json.data.data[category],
                            color: this.options.categories[category].color,
                            category_type: category_type
                        };

                        rows += json.data.data[category];

                        cnt++;
                    }
                }

                percent = ((rows / total) * 100).toFixed(2);
                
                this.$toggle_detail_button.removeClassName('disabled').enable();
                this.$reset_categories_button.removeClassName('disabled').enable();

                this.$progressbar
                    .setStyle({width:0, backgroundColor: Color.retrieve_color_by_percent(percent)})
                    .morph('width:' + percent + '%;', {duration:0.75});

                this.$infotext
                    .update('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__CALCULATED_QUALITY"}] ' + percent + '%')
                    .insert(new Element('small', {className:'ml10 text-neutral-400'}).update(('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__OF_X_CATEGORIES"}]'.replace('{x}', cnt))));
            }.bind(this)
        });
    },

    filter_category: function (ev) {
        var $button = ev.findElement('button'),
            $row = $button.up('tr').remove(),
            $rows = this.$right_detail.select('tr[data-category-const]'),
            category = $row.readAttribute('data-category-const'),
            percent,
            rows = 0,
            total = 0,
            item,
            cnt = 0,
            i;

        // Delete the data.
        delete this.data[category];

        // After deleting the data, we render the chart.
        this.draw_chart();

        // Re-calculate the object-type quality.
        for (i in $rows) {
            if ($rows.hasOwnProperty(i)) {
                item = this.data[$rows[i].readAttribute('data-category-const')];

                rows += item.rows;
                total += item.total;
                cnt ++;
            }
        }

        percent = ((rows / total) * 100).toFixed(2);

        this.$infotext
            .update('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__CALCULATED_QUALITY"}] ' + percent + '%')
            .insert(new Element('small', {className:'ml10 text-neutral-400'}).update(('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__OF_X_CATEGORIES"}]'.replace('{x}', cnt))));

        this.$progressbar
            .morph('width:' + percent + '%;background-color:' + Color.retrieve_color_by_percent(percent) + ';', {duration:0.75});
    },

    display_objects_without_data: function (ev) {
        var $button = ev.findElement('button'),
            $img = $button.down('img'),
            category = $button.up('tr').readAttribute('data-category-const'),
            category_type = $button.up('tr').readAttribute('data-category-type');

        $img.addClassName('animation-rotate').writeAttribute('src', window.dir_images + 'axialis/user-interface/loading.svg');

        // Retrieve all objects with no entries in the given category.
        new Ajax.Request('[{$ajax_url}]&func=load-objects-without-data', {
            parameters: {
                obj_type_id: this.object_type_id,
                category: category,
                category_type: category_type
            },
            onComplete: function (response) {
                var json,
                    $container = this.$right_detail.down('div.object-table'),
                    $table = $container.down('tbody').update(),
                    i;
                
                $container.down('h4 span').update(this.data[category].title);

                $img.removeClassName('animation-rotate').writeAttribute('src', window.dir_images + 'axialis/basic/zoom.svg');

                try {
                    this.check_response(response);
                } catch (e) {
                    idoit.Notify.error(e);
                    return;
                }

                json = response.responseJSON;

                for (i in json.data) {
                    if (json.data.hasOwnProperty(i)) {
                        $table.insert(new Element('tr')
                            .update(new Element('td').update(json.data[i].id))
                            .insert(new Element('td', { className: 'filter' }).update(json.data[i].title))
                            .insert(new Element('td').update(json.data[i].sysid))
                            .insert(new Element('td', { className: 'p5' }).update(
                                new Element('a', { className: 'btn btn-small', href: json.data[i].link, target: '_blank' })
                                    .update(new Element('img', {src:window.dir_images + 'axialis/basic/link.svg'}))
                                    .insert(new Element('span').update('[{isys type="lang" ident="LC__UNIVERSAL__LINK"}]')))));
                    }
                }

                if ($container.visible()) {
                    $container.highlight()
                } else {
                    new Effect.BlindDown($container, {duration:0.5});
                }
            }.bind(this)
        });
    },

    hide_objects_without_data: function () {
        new Effect.BlindUp(this.$right_detail.down('div.object-table'), {duration:0.5});
    },

    draw_chart: function () {
        var that = this,
            $chart = new Element('div', {id: 'chart_' + this.object_type_id});

        this.$left_detail.update($chart);

        idoit.Require.require(['d3', 'd3ChartPie'], function () {
            var dataCache = [], pie, i;

            for (i in that.data) {
                if (!that.data.hasOwnProperty(i)) {
                    continue;
                }

                dataCache.push({
                    id: i,
                    color: that.data[i].color,
                    count: that.data[i].rows
                });
            }

            pie = new D3ChartPie($chart, {width: that.$left_detail.up('td').getWidth(), height: 400, innerRadius: 60})
                .setData(dataCache)
                .update();

            $chart.on('toggle:category', function (ev) {
                that.$right_detail.select('td.filter').invoke('removeClassName', 'text-bold');

                if (ev.memo.filter === null) {
                    pie.unfilter();
                } else {
                    pie.filter(ev.memo.filter);
                    ev.memo.$td.addClassName('text-bold');
                }
            });

            that.$right_detail.on('click', 'td.filter', function (ev) {
                var $td = ev.findElement('td'),
                    categoryConst = $td.hasClassName('text-bold') ? null : $td.up('tr').readAttribute('data-category-const');

                $chart.fire('toggle:category', {$td: $td, filter: categoryConst});
            });
        })
    },

    toggle_detail: function () {
        const $table = new Element('table', {className:'mainTable border mt5'});
        const $thead = new Element('thead');
        const $tbody = new Element('tbody');
        var percent,
            cnt,
            i,
            cache = Object.values(this.data);

        if (this.$toggle_detail_button.readAttribute('data-open') == 1) {
            this.$toggle_detail_button.writeAttribute('data-open', 0)
                .down('img').writeAttribute('src', window.dir_images + 'axialis/user-interface/angle-right-small.svg')
                .next('span').update('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__SHOW_DETAIL_DATA"}]');

            new Effect.BlindUp(this.$right_detail, {duration:0.5});
            new Effect.BlindUp(this.$left_detail, {duration:0.5});
        } else {
            cnt = 0;

            this.$right_detail.hide();
            this.$left_detail.hide();

            if (!this.$right_detail.down('table')) {
                this.draw_chart();
    
                $thead
                    .update(new Element('tr')
                        .update(new Element('th', {style:'width:20px;'}))
                        .insert(new Element('th').update('[{isys type="lang" ident="LC_UNIVERSAL__CATEGORY"}]'))
                        .insert(new Element('th').update('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY"}]'))
                        .insert(new Element('th').update('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__CATEGORY_CONTENTS"}]'))
                        .insert(new Element('th')));
                
                // Preparing the right side: The category table.
                $table
                    .update($thead)
                    .insert($tbody);
      
                // Implementing sorting :)
                cache.sort(function(a, b) {
                    return a.title.localeCompare(b.title);
                });
                
                for (i in cache) {
                    if (cache.hasOwnProperty(i)) {
                        percent = ((cache[i].rows / cache[i].total) * 100).toFixed(2);
    
                        $tbody
                            .insert(new Element('tr', {'data-category-const':cache[i].const, 'data-category-type':cache[i].category_type})
                                .update(new Element('td', { className: 'p5' })
                                    .update(new Element('button', {type:'button', className:'btn btn-small btn-secondary remove-cat', title:'[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__REMOVE_CATEGORY"}]', 'data-table': 1})
                                        .update(new Element('img', {src: window.dir_images + 'axialis/basic/symbol-cancel.svg'}))))
                                .insert(new Element('td', {className: 'filter mouse-pointer'})
                                    .update(new Element('div', {className:'cmdb-marker', style:'background-color:' + cache[i].color}))
                                    .insert(cache[i].title))
                                .insert(new Element('td')
                                    .update(new Element('i', {className:'hidden', 'data-sort':('00' + percent).substr(-6)}))
                                    .insert(percent + '%')
                                    .insert(new Element('div', {className:'progress'})
                                        .update(new Element('div', {className:'progress-bar', style:'width:' + percent + '%; background-color:' + Color.retrieve_color_by_percent(percent) + ';'}))))
                                .insert(new Element('td')
                                    .update(new Element('span').update(cache[i].rows + ' / ' + cache[i].total)))
                                .insert(new Element('td', { className: 'p5' })
                                    .update(new Element('button', {type:'button', className:'btn btn-small objects-without-data fr', title:'[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__SHOW_RESPONSIBLE_OBJECTS"}]', 'data-table': 1})
                                        .update(new Element('img', {src:window.dir_images + 'axialis/basic/zoom.svg'})))));

                        cnt++;
                    }
                }

                this.$right_detail
                    .update($table)
                    .insert(new Element('div', { className: 'object-table' }).hide()
                        .update(new Element('div', { className: 'bg-neutral-200 p5' })
                            .update(new Element('h4').update('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__RESPONSIBLE_OBJECTS_HEADLINE" p_bHtmlEncode=false}]'))
                            .insert(new Element('button', { type: 'button', className: 'btn btn-mini btn-secondary close-object-table ml-auto'})
                                .update(new Element('img', { src: window.dir_images + 'axialis/basic/symbol-cancel.svg' }))))
                        .insert(new Element('table', { className: 'mainTable border'})
                            .update(new Element('thead')
                                .update('tr')
                                    .update(new Element('th').update('ID'))
                                    .insert(new Element('th').update('[{isys type="lang" ident="LC__UNIVERSAL__TITLE"}]'))
                                    .insert(new Element('th').update('[{isys type="lang" ident="LC__CMDB__CATG__GLOBAL_SYSID"}]'))
                                    .insert(new Element('th').update('[{isys type="lang" ident="LC__UNIVERSAL__TITLE_LINK"}]')))
                            .insert(new Element('tbody'))));
            }

            this.$toggle_detail_button.writeAttribute('data-open', 1)
                .down('img').writeAttribute('src', window.dir_images + 'axialis/user-interface/angle-down-small.svg')
                .next('span').update('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__HIDE_DETAIL_DATA"}]');

            new Effect.BlindDown(this.$right_detail, {duration:0.5});
            new Effect.BlindDown(this.$left_detail, {duration:0.5});
        }
    },

    reset_categories: function () {
        // Reset the buttons and the "detail" content.
        this.$left_detail.hide().update(new Element('canvas', {id:'chart_' + this.object_type_id}));
        this.$right_detail.hide().update();
        this.$reset_categories_button.addClassName('disabled').disable();
        this.$toggle_detail_button.addClassName('disabled').disable()
            .writeAttribute('data-open', 0)
            .down('img').writeAttribute('src', window.dir_images + 'axialis/user-interface/angle-right-small.svg')
            .next('span').update('[{isys type="lang" ident="LC__MODULE__ANALYTICS__DATA_QUALITY__SHOW_DETAIL_DATA"}]');

        this.reload(true);
    }
});
