<?php

/**
 * i-doit
 *
 * API model
 *
 * @package    i-doit
 * @subpackage API
 * @author     Dennis Stücken <dstuecken@synetics.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_cmdb_reports extends isys_api_model_cmdb implements isys_api_model_interface
{
    /**
     * Data formatting used in format methods
     *
     * @var array
     */
    protected $m_mapping = [
        'isys_report__id'           => 'id',
        'category_title'            => 'category',
        'isys_report__title'        => 'title',
        'isys_report__description'  => 'description',
        'isys_report__datetime'     => 'created',
        'isys_report__last_editied' => 'modified',
    ];

    /**
     * Possible options and their parameters
     *
     * @var array
     */
    protected $m_options = [
        'read'   => [
            'id' => [
                'type'        => 'int',
                'description' => 'Report ID',
                'reference'   => 'isys_report__id',
                'optional'    => true
            ]
        ],
        'create' => [],
        'update' => [],
        'delete' => [],
    ];

    /**
     * Validation
     *
     * @var array
     */
    protected $m_validation = [];

    /**
     * Fetches information about an object.
     * [
     *    integer  $p_params['id']  The ID of the report
     * ]
     *
     * @param array $p_params
     *
     * @return array Returns an empty array when an error occures.
     * @throws isys_exception_api
     */
    public function read($p_params)
    {
        $l_return = [];

        $l_dao = isys_report_dao::instance(isys_application::instance()->container->database_system);

        if (isset($p_params['id'])) {
            $this->m_log->info('Retrieving report with id ' . $p_params['id']);

            if ($this->useAuth) {
                isys_auth_report::instance()
                    ->custom_report(isys_auth::VIEW, $p_params['id']);
            }

            $report = $l_dao->get_reports(null, [$p_params['id']], null, true, false);

            if (isset($report[0]) && !empty($report[0]['isys_report__query'])) {
                if (!$l_dao->hasQueryOnlyInternalPlaceholder($report[0]['isys_report__query'])) {
                    throw new isys_exception_api(sprintf('Report with id %s contains non internal placeholders, therefore cannot be executed by the api', $p_params['id']));
                }

                $query = $l_dao->replacePlaceHolders($report[0]['isys_report__query']);

                // Execute report.
                $l_report_res = $l_dao->query($query);
                $showHtml = (bool)$report[0]['isys_report__show_html'];

                // Remove __id__ keys as of __obj_id__ is available as well.
                if (is_array($l_report_res['content'])) { // @see API-73
                    array_walk($l_report_res['content'], [$this, 'remove_id']);
                    if (!$showHtml) {
                        array_walk($l_report_res['content'], [$this, 'stripTags']);
                    }
                }

                return $l_report_res['content'];
            }

            throw new isys_exception_api(sprintf('Report with id %s does not exist', $p_params['id']));
        }

        $reports = $l_dao->get_reports(null, null, null, true, false);

        foreach ($reports as $report) {
            if ($this->useAuth) {
                try {
                    isys_auth_report::instance()->custom_report(isys_auth::VIEW, $report['isys_report__id']);
                } catch (isys_exception_auth $e) {
                    $this->m_log->error($e->getMessage());

                    continue;
                }
            }

            // Skip reports with placeholders
            if (!$l_dao->hasQueryOnlyInternalPlaceholder($report['isys_report__query'])) {
                continue;
            }

            $l_return[] = $this->format_by_mapping($this->m_mapping, $report);
        }

        return $l_return;
    }

    /**
     * @param $p_array
     */
    public function stripTags(&$p_array)
    {
        foreach ($p_array as $fieldName => $fieldValue) {
            $p_array[$fieldName] = $this->processHtml($this->removeForbiddenTags($fieldValue, ['script']));
        }
    }

    /**
     * Method for removing tags and appending whitespaces between them
     * @url https://www.php.net/manual/en/function.strip-tags
     *
     * @param $string
     * @return string
     */
    private function processHtml($string) : string
    {
        // ----- replace BR TAGs -----
        $string = preg_replace ('/<br\s*\/?>/i', "\n", $string);
        // ----- remove HTML TAGs -----
        $string = preg_replace ('/<[^>]*>/', ' ', $string);

        // ----- remove control characters -----
        $string = str_replace("\t", ' ', $string);   // --- replace with space

        // ----- remove multiple spaces -----
        $string = trim(preg_replace('/\s+/', ' ', $string));
        $string = trim(preg_replace('/(\r\n|\r|\n)+/', " ", $string));
        $string = isys_glob_html_entity_decode($string);

        return $string;
    }

    /**
     * Method for cutting undesired html tags out of html content like script, style and so on.
     * @param string $html
     * @param array $forbiddenTags
     * @return string
     */
    public function removeForbiddenTags($html, $forbiddenTags = [])
    {
        if (empty($html) || !is_string($html)) {
            return $html;
        }

        $dom = new DOMDocument();
        $dom->loadHTML(mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8'));

        foreach ($forbiddenTags as $tag) {
            $element = $dom->getElementsByTagName($tag);

            foreach ($element as $item) {
                $item->parentNode->removeChild($item);
            }
        }

        return $dom->saveHTML();
    }

    /**
     * @param $p_array
     */
    public function remove_id(&$p_array)
    {
        unset($p_array['__id__']);
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function create($p_params)
    {
        throw new isys_exception_api('Creating is not possible here.');
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function delete($p_params)
    {
        throw new isys_exception_api('Deleting is not possible here.');
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function update($p_params)
    {
        throw new isys_exception_api('Updating is not possible here.');
    }

    /**
     * isys_api_model_cmdb_reports constructor.
     *
     * @param isys_cmdb_dao $p_dao
     */
    public function __construct(isys_cmdb_dao $p_dao)
    {
        parent::__construct();
    }
}
