<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api;

use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use idoit\Module\Api\Exception\JsonRpc\ParameterException;

/**
 * Class CmdbObject
 *
 * @package idoit\Module\Api
 */
class CmdbObject
{
    /**
     * Valid status transitions
     */
    const TRANSITIONS = [
        C__RECORD_STATUS__BIRTH                 => [

        ],
        C__RECORD_STATUS__NORMAL                => [
            C__RECORD_STATUS__ARCHIVED,
            C__RECORD_STATUS__DELETED,
            C__RECORD_STATUS__TEMPLATE,
            C__RECORD_STATUS__MASS_CHANGES_TEMPLATE
        ],
        C__RECORD_STATUS__ARCHIVED              => [
            C__RECORD_STATUS__NORMAL,
            C__RECORD_STATUS__DELETED,
        ],
        C__RECORD_STATUS__DELETED               => [
            C__RECORD_STATUS__NORMAL,
            C__RECORD_STATUS__ARCHIVED,
        ],
        C__RECORD_STATUS__MASS_CHANGES_TEMPLATE => [
            C__RECORD_STATUS__NORMAL,
            C__RECORD_STATUS__TEMPLATE,
        ],
        C__RECORD_STATUS__TEMPLATE              => [
            C__RECORD_STATUS__NORMAL,
            C__RECORD_STATUS__MASS_CHANGES_TEMPLATE,
        ],
        C__RECORD_STATUS__PURGE                 => [
            C__RECORD_STATUS__BIRTH,
            C__RECORD_STATUS__NORMAL,
            C__RECORD_STATUS__ARCHIVED,
            C__RECORD_STATUS__DELETED,
            C__RECORD_STATUS__TEMPLATE,
            C__RECORD_STATUS__MASS_CHANGES_TEMPLATE
        ]
    ];

    /**
     * ObjectId
     *
     * @var int
     */
    protected $objectId;

    /**
     * @var
     */
    protected $targetStatus;

    /**
     * @var \isys_cmdb_dao
     */
    protected $dao;

    /**
     * @var array
     */
    protected $objectData;

    /**
     * @return int
     */
    public function getObjectId()
    {
        return $this->objectId;
    }

    /**
     * @param int $objectId
     *
     * @return CmdbObject
     */
    public function setObjectId($objectId)
    {
        $this->objectId = $objectId;

        return $this;
    }

    /**
     * Get object status
     *
     * @return int
     */
    public function getObjectStatus()
    {
        return $this->objectData['isys_obj__status'];
    }

    /**
     * @return int
     */
    public function getTargetStatus()
    {
        return $this->targetStatus;
    }

    /**
     * @param int $targetStatus
     *
     * @return CmdbObject
     */
    public function setTargetStatus($targetStatus)
    {
        $this->targetStatus = $targetStatus;

        return $this;
    }

    /**
     * Recycle entry stepwise
     *
     * @return bool
     * @throws \idoit\Module\Api\Exception\JsonRpc\InternalErrorException
     * @throws \idoit\Module\Api\Exception\JsonRpc\ParameterException
     * @throws \isys_exception_cmdb
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     * @throws \isys_exception_general
     * @throws \Exception
     */
    public function recycle()
    {
        // Store entry status id
        $sourceStatus = $this->getObjectStatus();

        // Check whether entry is normal
        if ($sourceStatus == C__RECORD_STATUS__NORMAL) {
            throw new \idoit\Module\Api\Exception\JsonRpc\ParameterException('Unable to delete object that is already normal.');
        }

        if (!in_array($sourceStatus, [C__RECORD_STATUS__DELETED, C__RECORD_STATUS__ARCHIVED])) {
            throw new ParameterException('Status transition is only allowed for archived and deleted objects');
        }

        // Try to rank entry in category
        if (!$this->dao->rank_record($this->objectId, C__CMDB__RANK__DIRECTION_RECYCLE, 'isys_obj')) {
            throw new \idoit\Module\Api\Exception\JsonRpc\InternalErrorException('Unable to rank object.');
        }

        // Store target status id
        $this->setTargetStatus($sourceStatus == C__RECORD_STATUS__ARCHIVED ? C__RECORD_STATUS__NORMAL : C__RECORD_STATUS__ARCHIVED);

        return true;
    }

    /**
     * Delete entry stepwise
     *
     * @return bool
     * @throws \idoit\Module\Api\Exception\JsonRpc\InternalErrorException
     * @throws \idoit\Module\Api\Exception\JsonRpc\ParameterException
     * @throws \isys_exception_cmdb
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     * @throws \isys_exception_general
     * @throws \Exception
     */
    public function delete()
    {
        // Store entry status id
        $sourceStatus = $this->getObjectStatus();

        // Check whether entry is normal
        if ($sourceStatus == C__RECORD_STATUS__DELETED) {
            throw new \idoit\Module\Api\Exception\JsonRpc\ParameterException('Unable to delete object that is already deleted.');
        }

        if (!in_array($sourceStatus, [C__RECORD_STATUS__ARCHIVED, C__RECORD_STATUS__NORMAL])) {
            throw new ParameterException('Unable to delete object that is not in status normal or archived.');
        }

        // Try to rank entry in category
        if (!$this->dao->rank_record($this->objectId, C__CMDB__RANK__DIRECTION_DELETE, 'isys_obj')) {
            throw new \idoit\Module\Api\Exception\JsonRpc\InternalErrorException('Unable to rank object.');
        }

        // Store target status id
        $this->setTargetStatus($sourceStatus == C__RECORD_STATUS__ARCHIVED ? C__RECORD_STATUS__DELETED : C__RECORD_STATUS__ARCHIVED);

        return true;
    }

    /**
     * Purge entry
     *
     * @return bool
     * @throws \idoit\Module\Api\Exception\JsonRpc\InternalErrorException
     * @throws \idoit\Module\Api\Exception\JsonRpc\ParameterException
     * @throws \isys_exception_cmdb
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     * @throws \isys_exception_general
     * @throws \Exception
     */
    public function purge()
    {
        // Check whether quickpurge is enabled
        if (!\isys_settings::get('cmdb.quickpurge', false)) {
            throw new \Exception('Quickpurge is not enabled');
        }

        if (!$this->isTransitionAllowed($this->getObjectStatus(), C__RECORD_STATUS__PURGE)) {
            throw new ParameterException('Unable to purge object that has status ' . $this->getObjectStatus());
        }

        // Try to rank entry in category
        if (!$this->dao->rank_record($this->objectId, C__CMDB__RANK__DIRECTION_DELETE, 'isys_obj', null, true)) {
            throw new \idoit\Module\Api\Exception\JsonRpc\InternalErrorException('Unable to purge object.');
        }

        // Store target status id
        $this->setTargetStatus(C__RECORD_STATUS__PURGE);

        // Destruct entry instance
        /**
         * @todo Prevent further using because object is not present anymore
         */
        //unset($this);

        return true;
    }

    /**
     * Mark object as template
     *
     * @return bool
     * @throws \Exception
     */
    public function markAsTemplate() {
        // Check whether transition is allowed
        if (!$this->isTransitionAllowed($this->getObjectStatus(), C__RECORD_STATUS__TEMPLATE)) {
            throw new \Exception('Unable to mark objects as template that are not in status normal or mass change.');
        }

        // Set object status
        if ($this->dao->set_object_status($this->objectId, C__RECORD_STATUS__TEMPLATE)) {
            return true;
        }

        return false;
    }

    /**
     * Mark object as mass change template
     *
     * @return bool
     * @throws \Exception
     */
    public function markAsMassChangeTemplate() {
        // Check whether transition is allowed
        if (!$this->isTransitionAllowed($this->getObjectStatus(), C__RECORD_STATUS__MASS_CHANGES_TEMPLATE)) {
            throw new \Exception('Unable to mark objects as template that are not in status normal or template.');
        }

        // Set object status
        if ($this->dao->set_object_status($this->objectId, C__RECORD_STATUS__MASS_CHANGES_TEMPLATE)) {
            return true;
        }

        return false;
    }

    /**
     * Object ranker
     *
     * @param int $targetStatus
     *
     * @return bool
     * @throws \Exception
     */
    public function rank($targetStatus)
    {
        // Get source status
        $sourceStatus = $this->getObjectStatus();

        // Check whether source and target are equal
        if ($targetStatus == $sourceStatus) {
            throw new \Exception('Source and target status are the same.');
        }

        // Redirect to template ranker
        if ($targetStatus == C__RECORD_STATUS__TEMPLATE) {
            return $this->markAsTemplate();
        }

        // Redirect to mass change template handler
        if ($targetStatus == C__RECORD_STATUS__MASS_CHANGES_TEMPLATE) {
            return $this->markAsMassChangeTemplate();
        }

        // Check whether purging is allowed
        if ($targetStatus === C__RECORD_STATUS__PURGE) {
            return $this->purge();
        }

        // Check whether transition is allowed
        if (!$this->isTransitionAllowed($sourceStatus, $targetStatus)) {
            throw new \Exception('Transition of ranking from status ' . $sourceStatus . ' to ' . $targetStatus . ' is not allowed.');
        }

        // Calculate status difference
        $statusDifference = $targetStatus - $sourceStatus;

        // Calculate direction
        $direction = $statusDifference > 0 ? C__CMDB__RANK__DIRECTION_DELETE : C__CMDB__RANK__DIRECTION_RECYCLE;

        // Iterate needed transitions
        for ($i = 0;$i < abs($statusDifference);$i++) {
            \isys_component_signalcollection::get_instance()
                ->emit("mod.cmdb.beforeObjectRank", $this->dao, $direction, [$this->getObjectId()]);

            // Rank record
            if (!$this->dao->rank_records([$this->getObjectId()], $direction, 'isys_obj')) {
                throw new \Exception('Unable to transition object status.');
            }

            \isys_component_signalcollection::get_instance()
                ->emit("mod.cmdb.afterObjectRank", $this->dao, $direction, [$this->getObjectId()]);
        }

        // Set target status
        $this->setTargetStatus($targetStatus);

        return true;
    }

    /**
     * Check whether transition is allowed
     *
     *
     * @param $sourceStatus
     * @param $targetStatus
     *
     * @return bool
     */
    protected function isTransitionAllowed($sourceStatus, $targetStatus)
    {
        return in_array($sourceStatus, self::TRANSITIONS[$targetStatus]);
    }

    /**
     * Initialize Ranker instance
     *
     * @return CmdbObject
     * @throws \Exception
     */
    protected function initialize()
    {
        // Validate objectId
        if (!is_int($this->objectId) || $this->objectId <= 0) {
            throw new \Exception('ObjectId has to be a positive numeric value.');
        }

        // Check whether object does exist
        if (!$this->dao->obj_exists($this->objectId)) {
            throw new \Exception('ObjectId does not exist.');
        }

        // Get object data
        $this->objectData = $this->dao->get_object($this->objectId)
            ->get_row();

        // Validate object status
        if (empty($this->getObjectStatus())) {
            throw new InternalErrorException('Unable to determine object status.');
        }

        return $this;
    }

    /**
     * Ranker constructor.
     *
     * @param int $objectId
     *
     * @throws \Exception
     */
    public function __construct($objectId)
    {
        // Create dao instance
        $this->dao = new \isys_cmdb_dao(\isys_application::instance()->container->get('database'));

        // Store information
        $this->setObjectId($objectId)
            ->initialize();
    }
}
