<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use Exception;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\RequestModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\SyncModifier;
use isys_application;
use isys_cmdb_dao_category_g_custom_fields;

/**
 * Class CustomFieldsProcessor
 *
 * @package idoit\Module\Api\Model\Cmdb\Category\Processor
 */
class CustomFieldsProcessor extends AbstractCategoryProcessor implements SyncModifier, RequestModifier
{
    /**
     * Modify sync data
     *
     * @param array $syncData
     *
     * @return array
     * @throws \isys_exception_api_validation
     * @throws \Exception
     */
    public function modifySyncData(array $syncData)
    {
        // Just to go sure we always have the correct DAO.
        if (!$this->getDao() instanceof isys_cmdb_dao_category_g_custom_fields) {
            return $syncData;
        }

        if (\is_array($syncData['properties'])) {
            $dao = $this->getDao();
            $virtualProperties = [];
            $providesVirtual = $dao->convert_sql_int(defined_or_default('C__PROPERTY__PROVIDES__VIRTUAL', 128));
            $customCategoryId = $dao->convert_sql_id($this->getDao()->get_catg_custom_id());

            // First get the property keys.
            $propertyKeys = array_keys($syncData['properties']);

            // Then make the keys 'secure'.
            $propertyKeys = array_map(function ($key) use ($dao) {
                return $dao->convert_sql_text($key);
            }, $propertyKeys);

            // Check, just to go sure we don't risk a faulty SQL query.
            if (\count($propertyKeys) === 0) {
                return $syncData;
            }

            // And finally use them in the query as comma separated list.
            $properties = implode(',', $propertyKeys);

            $sql = "SELECT isys_property_2_cat__prop_key AS propertyKey 
                FROM isys_property_2_cat 
                WHERE isys_property_2_cat__prop_key IN ({$properties}) 
                AND isys_property_2_cat__isysgui_catg_custom__id = {$customCategoryId}
                AND isys_property_2_cat__prop_provides & {$providesVirtual};";

            $result = $dao->retrieve($sql);

            while ($row = $result->get_row()) {
                $virtualProperties[$row['propertyKey']] = 'This property is a "virtual" property and can not contain data.';
            }

            // If any virtual properties were found: trigger an error.
            if (\count($virtualProperties)) {
                throw new \isys_exception_api_validation('You can not populate virtual properties with data.', $virtualProperties);
            }
        }

        return $syncData;
    }

    /**
     * Modify request in case of used constants for dialogs.
     *
     * @param array $request
     *
     * @return array
     */
    public function modifyRequest(array $request)
    {
        // Just to go sure we always have the correct DAO.
        if (!$this->getDao() instanceof isys_cmdb_dao_category_g_custom_fields) {
            return $request;
        }

        // Also skip non-writing actions.
        if (!in_array($request['option'], ['save', 'update', 'create'], true)) {
            return $request;
        }

        // @see  API-227  Check for provided dialog+ data: We'll need to check for constants.
        $dao = $this->getDao();
        $properties = $dao->get_properties();
        $dialogTypes = [C__PROPERTY__INFO__TYPE__DIALOG_PLUS, C__PROPERTY__INFO__TYPE__MULTISELECT];

        // see API-221 checking values for Yes-No field
        foreach ($request['data'] as $l_key => $l_value) {
            if (!isset($properties[$l_key]['ui']['params']['extra'])) {
                continue;
            }
            if ($properties[$l_key]['ui']['params']['extra'] !== 'yes-no') {
                continue;
            }

            if (is_array($l_value)) {
                $dialogValues = $l_value;
                $isArray = true;
            } else {
                $dialogValues = [$l_value];
                $isArray = false;
            }

            foreach ($dialogValues as $val_key => $dialogValue) {
                $dialogValue = strtolower((string)$dialogValue);
                if (!$dialogValue) {
                    $dialogValues[$val_key] = 'LC__UNIVERSAL__NO';
                } else {
                    switch ($dialogValue) {
                        case "1":
                        case "true":
                        case "yes":
                            $dialogValues[$val_key] = 'LC__UNIVERSAL__YES';
                            break;
                        case "0":
                        case "false":
                        case "no":
                            $dialogValues[$val_key] = 'LC__UNIVERSAL__NO';
                            break;
                    }
                }
            }

            if ($isArray) {
                $request['data'][$l_key] = $dialogValues;
            } else {
                $request['data'][$l_key] = $dialogValues[0];
            }
        }

        foreach ($request['data'] as $propertyKey => $value) {
            // Skip all integers.
            if (is_int($value)) {
                continue;
            }

            // Skip non-existing properties.
            if (!isset($properties[$propertyKey])) {
                continue;
            }

            // Also skip, if the property is no dialog.
            if (!in_array($properties[$propertyKey][C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE], $dialogTypes, true)) {
                continue;
            }

            $dialogIdentifier = (string)$properties[$propertyKey][C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_identifier'];

            if (is_array($value)) {
                foreach ($value as $index => $item) {
                    if (is_int($item)) {
                        continue;
                    }

                    $request['data'][$propertyKey][$index] = $this->processPossibleDialogConstants((string)$item, $dialogIdentifier);
                }
            } else {
                $request['data'][$propertyKey] = $this->processPossibleDialogConstants((string)$value, $dialogIdentifier);
            }
        }

        return $request;
    }

    /**
     * @param string $value
     * @param string $identifier
     *
     * @return string|int
     */
    private function processPossibleDialogConstants(string $value, string $identifier)
    {
        $result = $value;
        $dao = $this->getDao();

        try {
            $sql = 'SELECT isys_dialog_plus_custom__id AS id
                FROM isys_dialog_plus_custom 
                WHERE isys_dialog_plus_custom__const = ' . $dao->convert_sql_text($value) . '
                AND isys_dialog_plus_custom__identifier = ' . $dao->convert_sql_text($identifier) . '
                LIMIT 1;';

            $id = (int)$dao->retrieve($sql)->get_row_value('id');

            if ($id > 0) {
                $result = $id;
            }
        } catch (Exception $e) {
            // What could have gone wrong? In this case we don't change the value.
        }

        return $result;
    }
}
