<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use idoit\Module\Api\Dialog\DialogPlusAdmin;
use idoit\Module\Api\Exception\JsonRpc\ParameterException;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\RequestModifier;

/**
 * Class ModelProcessor
 *
 * @package idoit\Module\Api\Model\Cmdb\Category\Processor
 */
class ModelProcessor extends AbstractCategoryProcessor implements RequestModifier
{

    /**
     * Modify api request
     *
     * @param array $request
     *
     * @return array
     * @throws ParameterException
     * @throws \Exception
     */
    public function modifyRequest(array $request)
    {
        // Check whether requested option is equal to read
        if ($request['option'] === 'read') {
            return $request;
        }

        // Initialize needed data
        $title = $request['data']['title'];
        $manufacturer = $request['data']['manufacturer'];
        $objectId = $request['objID'];
        $modelDao = \isys_cmdb_dao_category_g_model::instance(\isys_application::instance()->container->get('database'));
        $usingExistingManufacturer = false;

        // Check whether processing is necessary or not
        if (empty($title) && empty($manufacturer)) {
            return $request;
        }

        // Check whether creation is requested without setted manufacturer but title
        if (empty($manufacturer) && !empty($title) && $request['option'] === 'create') {
            throw new ParameterException('Cannot set title without specified manufacturer.');
        }

        if (!empty($title) && \array_key_exists('manufacturer', $request['data']) && $request['data']['manufacturer'] === null) {
            throw new ParameterException('Unable to set title during unsetting manufacturer parameter.');
        }

        // Create DialogAdmin
        $dialogAdmin = DialogPlusAdmin::byTableName('isys_model_manufacturer', [
            C__PROPERTY__DATA => [C__PROPERTY__DATA__REFERENCES => ['isys_model_manufacturer']]
        ]);

        // Handle manufacturer value
        if (\is_string($manufacturer)) {
            // Create new manufacturer value
            $manufacturer = $dialogAdmin->findOrCreate($manufacturer);
        } else if (\is_int($manufacturer)) {
            // Check whether given id exists
            if (!$dialogAdmin->idExists($manufacturer)) {
                $manufacturer = null;
            }
        } else if ($manufacturer === null) {
            // Get actual manufacturer
            $manufacturer = $modelDao->get_data(null, $objectId)
                ->get_row_value('isys_catg_model_list__isys_model_manufacturer__id');

            $usingExistingManufacturer = true;
        }

        // Check whether manufacturer could be determined
        if (!empty($manufacturer)) {
            // Handle title value
            if (\is_string($title)) {
                // Check whether title exists
                $sql = 'SELECT isys_model_title__id FROM isys_model_title WHERE 
                        isys_model_title__title = ' . $modelDao->convert_sql_text($title) . ' AND 
                        isys_model_title__isys_model_manufacturer__id = ' . $modelDao->convert_sql_id($manufacturer) . ';';

                $titleId = (int)$modelDao->retrieve($sql)
                    ->get_row_value('isys_model_title__id');

                if ($titleId > 0) {
                    // Use existing value
                    $title = $titleId;
                } else {
                    // Create new entry
                    $sql = 'INSERT INTO isys_model_title SET 
                            isys_model_title__title = ' . $modelDao->convert_sql_text($title) . ',
                            isys_model_title__isys_model_manufacturer__id = ' . $modelDao->convert_sql_id($manufacturer) . ',
                            isys_model_title__status = ' . C__RECORD_STATUS__NORMAL . ';';

                    if ($modelDao->update($sql) && $modelDao->apply_update()) {
                        $title = $modelDao->get_last_insert_id();
                    }
                }
            } elseif (\is_int($title)) {
                $sql = 'SELECT isys_model_title__id FROM isys_model_title WHERE 
                        isys_model_title__id = ' . $modelDao->convert_sql_id($title) . ' AND 
                        isys_model_title__isys_model_manufacturer__id = ' . $modelDao->convert_sql_id($manufacturer) . ';';

                if (\count($modelDao->retrieve($sql)) === 0) {
                    $title = null;
                }
            } else {
                $title = null;
            }
        } else {
            throw new ParameterException('Unable to determine manufacturer');
        }

        // Set calculated data
        if (!$usingExistingManufacturer) {
            $request['data']['manufacturer'] = $manufacturer;
        }

        // Set title
        $request['data']['title'] = $title;

        return $request;
    }
}
