<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use idoit\Component\Helper\Ip;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\ResponseModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\SyncModifier;
use isys_application;
use isys_cmdb_dao_category_g_ip;
use isys_request;

/**
 * IpProcessor
 *
 * @package    idoit\Module\Api\Model\Category
 */
class IpProcessor extends AbstractCategoryProcessor implements ResponseModifier, SyncModifier
{

    /**
     * Modify api response
     *
     * @param array $response
     *
     * @return array
     */
    public function modifyResponse(array $response)
    {
        // Get request
        $request = $this->getRequest();

        // Check whether api request is 'read' and response object has results
        if (is_array($request) && $request['option'] === 'read' && is_array($response) && count($response) > 0) {
            $language = isys_application::instance()->container->get('language');
            $ipDao = isys_cmdb_dao_category_g_ip::instance(isys_application::instance()->container->get('database'));

            // Modify response.
            array_walk($response, function (&$row) use ($ipDao, $language) {
                // Switch net type and unset unnecessary type
                switch ($row['net_type']['const']) {
                    case 'C__CATS_NET_TYPE__IPV4':
                        unset($row['ipv6_address']);
                        break;
                    case 'C__CATS_NET_TYPE__IPV6':
                        unset($row['ipv4_address']);
                        break;
                }

                // @see  API-220  The property should contain a 'yes' or 'no' value, depending on if it's the default gateway.
                $request = isys_request::factory()->set_category_data_id($row['id']);

                $row['use_standard_gateway'] = [
                    'value' => 0,
                    'title' => $language->get('LC__UNIVERSAL__NO')
                ];

                if ($ipDao->callback_property_use_standard_gateway($request)) {
                    $row['use_standard_gateway'] = [
                        'value' => 1,
                        'title' => $language->get('LC__UNIVERSAL__YES'),
                    ];
                }
            });
        }

        return array_values($response);
    }

    /**
     * Modify sync data
     *
     * @param array $syncData
     *
     * @return array
     */
    public function modifySyncData(array $syncData)
    {
        // Get request
        $request = $this->getRequest();

        /**
         * @see API-23 Unset DNS-Server which will represent category data id by default
         */
        if (empty($request['data']['dns_server']) && !empty($syncData['properties']['dns_server'][C__DATA__VALUE])) {
            unset($syncData['properties']['dns_server']);
        }

        /**
         * @see API-293 Here we check if a IP address has been passed, if it's invalid we re-set it by getting the actual category value.
         */
        if ($syncData['properties']['net_type'][C__DATA__VALUE] != C__CATS_NET_TYPE__IPV6) {
            if (!isset($request['data']['ipv4_address']) && !Ip::validate_ip($syncData['properties']['ipv4_address'][C__DATA__VALUE])) {
                $syncData['properties']['ipv4_address'][C__DATA__VALUE] = $this->getDao()->get_ip_by_id($syncData['data_id'])['isys_cats_net_ip_addresses_list__title'];
            }
        } else {
            if (!isset($request['data']['ipv6_address']) && !Ip::validate_ipv6($syncData['properties']['ipv6_address'][C__DATA__VALUE])) {
                $syncData['properties']['ipv6_address'][C__DATA__VALUE] = $this->getDao()->get_ip_by_id($syncData['data_id'])['isys_cats_net_ip_addresses_list__title'];
            }
        }

        /**
         * @see API-282 The property should contain a 'yes' or 'no' value, depending on if it's the default gateway.
         */
        if (isset($request['data']['id'], $syncData['properties']['use_standard_gateway']) && !isset($request['data']['use_standard_gateway'])) {
            $callbackRequest = isys_request::factory()->set_category_data_id($request['data']['id']);

            $syncData['properties']['use_standard_gateway'][C__DATA__VALUE] = $this->dao->callback_property_use_standard_gateway($callbackRequest);
        }

        return $syncData;
    }
}
