<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Session;

use isys_application;
use isys_cmdb_dao_category_s_person_login;
use isys_component_session;
use isys_helper_crypt;

/**
 * Class UserSession
 *
 * @package idoit\Module\Api
 */
class UserSession implements SessionInterface
{
    /**
     * Username
     *
     * @var string
     */
    protected $username;

    /**
     * Password
     *
     * @var string
     */
    protected $password;

    /**
     * ApiKey
     *
     * @var string
     */
    protected $apiKey;

    /**
     * Persistent session
     *
     * @var bool
     */
    protected $persistent;

    /**
     * SessionComponent
     *
     * @var isys_component_session
     */
    protected $sessionComponent;

    /**
     * UserId
     *
     * @var int
     */
    protected $userId;

    /**
     * SessionId
     *
     * @var string
     */
    protected $sessionId;

    /**
     * MandatorId
     *
     * @var int
     */
    protected $mandatorId;

    /**
     * Check whether session exists
     *
     * @return bool
     */
    public function isLoggedIn()
    {
        return $this->getSessionComponent()->is_logged_in();
    }

    /**
     * Refresh session
     *
     * @return bool
     * @throws \Exception
     */
    public function refreshSession()
    {
        // Get database component
        $databseComponent = \isys_application::instance()->container->get('database');

        // Check whether requirements are met
        if ($this->isLoggedIn() && is_object($databseComponent) && !empty($this->getSessionId())) {
            $sql = 'UPDATE isys_user_session SET
                    isys_user_session__time_last_action = CURRENT_TIMESTAMP,
                    isys_user_session__description      = \'' . $databseComponent->escape_string($_SERVER['REQUEST_URI']) . '\'
                    WHERE isys_user_session__php_sid    = \'' . $databseComponent->escape_string($this->getSessionId()) . '\'';

            // Refresh session by updating `last_action` column
            return $databseComponent->query($databseComponent->limit_update($sql, 1));
        }

        throw new \Exception('Unable to refresh session that does not exist.');
    }

    /**
     * @return string
     */
    public function getUsername()
    {
        return $this->username;
    }

    /**
     * @param string $username
     *
     * @return UserSession
     */
    public function setUsername($username)
    {
        $this->username = $username;

        return $this;
    }

    /**
     * @return string
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * @param string $password
     *
     * @return UserSession
     */
    public function setPassword($password)
    {
        $this->password = $password;

        return $this;
    }

    /**
     * @return string
     */
    public function getApiKey()
    {
        return $this->apiKey;
    }

    /**
     * @param string $apiKey
     *
     * @return UserSession
     */
    public function setApiKey($apiKey)
    {
        $this->apiKey = $apiKey;

        return $this;
    }

    /**
     * @return isys_component_session
     */
    public function getSessionComponent()
    {
        return $this->sessionComponent;
    }

    /**
     * @param isys_component_session $sessionComponent
     *
     * @return UserSession
     */
    public function setSessionComponent($sessionComponent)
    {
        $this->sessionComponent = $sessionComponent;

        return $this;
    }

    /**
     * @return bool
     */
    public function isPersistent()
    {
        return $this->persistent;
    }

    /**
     * @param bool $persistent
     *
     * @return UserSession
     */
    public function setPersistent($persistent)
    {
        $this->persistent = $persistent;

        return $this;
    }

    /**
     * @return int
     */
    public function getMandatorId()
    {
        return $this->mandatorId;
    }

    /**
     * @param int $mandatorId
     *
     * @return UserSession
     */
    public function setMandatorId($mandatorId)
    {
        $this->mandatorId = $mandatorId;

        return $this;
    }

    /**
     * @return string
     */
    public function getSessionId()
    {
        return $this->sessionId;
    }

    /**
     * @param int $sessionId
     *
     * @return UserSession
     */
    public function setSessionId($sessionId)
    {
        $this->sessionId = $sessionId;

        return $this;
    }

    /**
     * @return int
     */
    public function getUserId()
    {
        return $this->userId;
    }

    /**
     * @param int $userId
     *
     * @return UserSession
     */
    public function setUserId($userId)
    {
        $this->userId = $userId;

        return $this;
    }

    /**
     * Get system database component
     *
     * @return \isys_component_database|mixed|object
     * @throws \Exception
     */
    protected function getSystemDatabase()
    {
        return \isys_application::instance()->container->get('database_system');
    }

    /**
     * @throws \isys_exception_api
     * @throws \Exception
     */
    public function login()
    {
        // Check whether requirements are met
        if (empty($this->getUsername()) || empty($this->getPassword()) || empty($this->getApiKey())) {
            throw new \Exception('Please provide valid username, password and apikey information');
        }

        // Initialize session related data
        $this->initialize();

        // Check whether presistence is wanted and session exists already
        if ($this->isPersistent() && !empty($this->getSessionId())) {
            // Login via sessionId
            return $this->getSessionComponent()->apikey_login($this->getApiKey(), null, $this->getSessionId());
        }

        // Reqular login via api key, username and password
        if ($this->getSessionComponent()->apikey_login($this->getApiKey(), ['username' => $this->getUsername(), 'password' => $this->getPassword()])) {
            // Set sessionId and userId
            $this->setSessionId($this->getSessionComponent()->get_session_id())
                ->setUserId($this->getSessionComponent()->get_user_id());

            return true;
        }

        return false;
    }

    /**
     * Initialize login procedure
     *
     * @throws \Exception
     * @return bool
     */
    protected function initialize()
    {
        // Get mandator by api key
        $mandatorId = $this->getMandatorIdByApiKey($this->getApiKey());

        // Connect mandator now
        $this->getSessionComponent()
            ->connect_mandator($mandatorId);

        // Get user by credentials now
        $userId = $this->getUserIdByCredentials($this->getUsername(), $this->getPassword());

        // Check whether persistent session should be used
        if ($this->isPersistent()) {
            try {
                // Get session for user
                $this->setSessionId($this->getSessionByUser($userId));
            } catch (\Exception $e) {
            }
        }

        // Set mandatorId and userId
        $this->setMandatorId($mandatorId)
            ->setUserId($userId);

        return true;
    }

    /**
     * Get session by userId
     *
     * @param int $userId
     *
     * @return string
     * @throws \Exception
     */
    public function getSessionByUser($userId)
    {
        $database = isys_application::instance()->container->get('database');

        $sql = 'SELECT isys_user_session__php_sid
            FROM isys_user_session 
            WHERE isys_user_session__isys_obj__id = ' . $database->escape_string($userId) . ';';

        $resource = $database->query($sql);

        if ($database->num_rows($resource)) {
            return $database->fetch_row_assoc($resource)['isys_user_session__php_sid'];
        }

        throw new \Exception('Unable to find session for user.');
    }

    /**
     * Get mandatorId by apiKey
     *
     * @param string $apiKey
     *
     * @return int
     * @throws \Exception
     */
    public function getMandatorIdByApiKey($apiKey)
    {
        $systemDatabase = $this->getSystemDatabase();

        $sql = "SELECT isys_mandator__id 
            FROM isys_mandator 
            WHERE isys_mandator__apikey = '" . $systemDatabase->escape_string($apiKey) . "' 
            AND isys_mandator__active = 1;";

        $resource = $systemDatabase->query($sql);

        if ($systemDatabase->num_rows($resource)) {
            return (int)$systemDatabase->fetch_row_assoc($resource)['isys_mandator__id'];
        }

        throw new \Exception('Unable to retrieve mandator');
    }

    /**
     * Get userId by credentials
     *
     * @param string $username
     * @param string $password
     *
     * @return int
     * @throws \Exception
     */
    public function getUserIdByCredentials($username, $password)
    {
        if (method_exists(isys_helper_crypt::class, 'checkPassword')) {
            $database = isys_application::instance()->container->get('database');

            $sql = "SELECT 
                isys_obj__id, 
                isys_cats_person_list__id,
                isys_cats_person_list__title,
                isys_cats_person_list__description,
                isys_cats_person_list__status,
                isys_cats_person_list__user_pass,
                isys_cats_person_list__unmigrated_password
            FROM isys_obj 
                INNER JOIN isys_cats_person_list ON isys_obj__id = isys_cats_person_list__isys_obj__id
                WHERE BINARY LOWER(isys_cats_person_list__title) = LOWER('" . $database->escape_string($this->getUsername()) . "');";

            $resource = $database->query($sql);
            if ($database->num_rows($resource)) {
                $row = $database->fetch_row_assoc($resource);
                if (isys_helper_crypt::checkPassword($password, $row["isys_cats_person_list__user_pass"], $row['isys_cats_person_list__unmigrated_password'])) {
                    return (int)$row['isys_obj__id'];
                }
            }
            return 0;
        }

        return $this->legacyGetUserIdByCredentials($username, $password);
    }

    /**
     * Legacy method of getUserIdByCredentials before the new encryption methods
     *
     * @param $username
     * @param $password
     *
     * @return int
     * @throws \Exception
     */
    private function legacyGetUserIdByCredentials($username, $password)
    {
        $database = isys_application::instance()->container->get('database');

        $sql = "SELECT isys_obj__id FROM isys_obj 
            INNER JOIN isys_cats_person_list ON isys_obj__id = isys_cats_person_list__isys_obj__id
            WHERE BINARY LOWER(isys_cats_person_list__title) = LOWER('" . $database->escape_string($this->getUsername()) . "') 
            AND isys_cats_person_list__user_pass = md5('" . $database->escape_string($this->getPassword()) . "');";

        $resource = $database->query($sql);

        if ($database->num_rows($resource)) {
            return (int)$database->fetch_row_assoc($resource)['isys_obj__id'];
        }

        // @see ID-6511 Don't throw an exception, because this will break the API for all users that authenticate via LDAP.
        return 0;
    }

    /**
     * Destruction procedure
     */
    public function __destruct()
    {
        if ($this->isPersistent()) {
            try {
                $this->refreshSession();
            } catch (\Exception $e) {
                // We do not need to handle it!
            }
        } else {
            $this->getSessionComponent()->logout();
        }
    }

    /**
     * UserSession constructor.
     *
     * @param string                 $username
     * @param string                 $password
     * @param string                 $apiKey
     * @param bool                   $persistent
     * @param isys_component_session $sessionComponent
     */
    public function __construct($username, $password, $apiKey, $persistent, isys_component_session $sessionComponent)
    {
        $this->setUsername($username)
            ->setPassword($password)
            ->setApiKey($apiKey)
            ->setPersistent($persistent)
            ->setSessionComponent($sessionComponent);
    }

    /**
     * Logout
     */
    public function logout()
    {
        $this->__destruct();
    }
}
