<?php

/**
 * i-doit
 *
 * API model
 *
 * @package    i-doit
 * @subpackage API
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_cmdb_status extends isys_api_model_cmdb implements isys_api_model_interface
{
    /**
     * @var isys_component_template_language_manager
     */
    private $language;

    /**
     * @param array $params
     *
     * @return array
     * @throws isys_exception_database
     */
    public function read($params)
    {
        $cmdbStatus = [];

        $result = $this->getCmdbStatus();

        while ($row = $result->get_row()) {
            $cmdbStatus[] = [
                'id'       => (int)$row['id'],
                'title'    => $this->language->get($row['title']),
                'constant' => $row['constant'],
                'color'    => $row['color'],
                'editable' => $this->isEditable($row)
            ];
        }

        return $cmdbStatus;
    }

    /**
     * @param array $params
     *
     * @return array
     * @throws isys_exception_api_validation
     * @throws isys_exception_database
     */
    public function save($params)
    {
        $validationErrors = [];

        if (!isset($params['title']) || empty(trim($params['title']))) {
            $validationErrors['title'] = 'You need to provide a title';
        }

        if (!isset($params['constant']) || empty(trim($params['constant']))) {
            $validationErrors['constant'] = 'You need to provide a constant';
        } elseif (preg_match('/[^A-Z0-9_]/', $params['constant'])) {
            $validationErrors['constant'] = 'The provided constant is not valid (only a-z character, 0-9 numbers and underscores)';
        }

        if (!isset($params['color']) || empty(trim($params['color']))) {
            $validationErrors['color'] = 'You need to provide a color (six character hex format without leading "#")';
        } elseif (strlen($params['color']) !== 6 || preg_match('/[^a-f0-9]/i', $params['color'])) {
            $validationErrors['color'] = 'The provided color is not valid (needs to have six characters a-f or 0-9)';
        }

        if (count($validationErrors)) {
            throw new isys_exception_api_validation(
                'There was an validation error.',
                $validationErrors
            );
        }

        if (!isset($params['id']) || !is_int($params['id']) || $params['id'] <= 0) {
            $createResult = $this->m_dao->create($params['constant'], $params['title'], $params['color']);

            if ($createResult) {
                return [
                    'success' => true,
                    'message' => 'Status created',
                    'id'      => $createResult
                ];
            }

            return [
                'success' => false,
                'message' => 'Status could not be created',
                'id'      => null
            ];
        }

        $cmdbStatus = $this->getCmdbStatus($params['id'])->get_row();

        if (!is_array($cmdbStatus)) {
            throw new isys_exception_api_validation(
                'There was an validation error.',
                ['id' => 'The provided CMDB Status does not exist']
            );
        }

        if (!$this->isEditable($cmdbStatus)) {
            throw new isys_exception_api_validation(
                'There was an validation error.',
                ['id' => 'The provided CMDB Status ("' . $this->language->get($cmdbStatus['title']) . '") is not editable']
            );
        }

        if ($this->m_dao->save($params['id'], $params['constant'], $params['title'], $params['color'])) {
            return [
                'success' => true,
                'message' => 'Status updated',
                'id'      => (int)$params['id']
            ];
        }

        return [
            'success' => false,
            'message' => 'Status could not be updated',
            'id'      => null
        ];
    }

    /**
     * @param array $params
     *
     * @return array
     * @throws isys_exception_api_validation
     * @throws isys_exception_database
     */
    public function delete($params)
    {
        if (!isset($params['id']) || !is_int($params['id']) || $params['id'] <= 0) {
            throw new isys_exception_api_validation(
                'There was an validation error.',
                ['id' => 'Needs to be an integer, greater than zero']
            );
        }

        $cmdbStatus = $this->getCmdbStatus($params['id'])->get_row();

        if (!is_array($cmdbStatus)) {
            throw new isys_exception_api_validation(
                'There was an validation error.',
                ['id' => 'The provided CMDB Status does not exist']
            );
        }

        if (!$this->isEditable($cmdbStatus)) {
            throw new isys_exception_api_validation(
                'There was an validation error.',
                ['id' => 'The provided CMDB Status ("' . $this->language->get($cmdbStatus['title']) . '") is not editable']
            );
        }

        if ($this->m_dao->delete_status($params['id'])) {
            return [
                'success' => true,
                'message' => 'Status purged'
            ];
        }

        return [
            'success' => false,
            'message' => 'Status could not be purged'
        ];
    }

    /**
     * @param int|array $id
     *
     * @return isys_component_dao_result
     * @throws isys_exception_database
     */
    private function getCmdbStatus($id = null)
    {
        $sql = 'SELECT 
            isys_cmdb_status__id AS id, 
            isys_cmdb_status__title AS title, 
            isys_cmdb_status__const AS constant, 
            isys_cmdb_status__color AS color,
            isys_cmdb_status__editable AS editable
            FROM isys_cmdb_status';

        if ($id !== null) {
            $sql .= ' WHERE isys_cmdb_status__id ' . $this->m_dao->prepare_in_condition((array)$id);
        }

        return $this->m_dao->retrieve($sql . ';');
    }

    /**
     * @param array $row
     *
     * @return bool
     */
    private function isEditable(array $row)
    {
        return $row['editable'] && !in_array($row['constant'], ['C__CMDB_STATUS__IN_OPERATION', 'C__CMDB_STATUS__INOPERATIVE'], true);
    }

    /**
     * isys_api_model_cmdb_status constructor.
     */
    public function __construct()
    {
        parent::__construct();

        $this->m_dao = isys_cmdb_dao_status::instance($this->m_db);
        $this->language = isys_application::instance()->container->get('language');
    }
}
