<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Controller;

use Exception;
use idoit\Component\Provider\Singleton;
use idoit\Module\Api\Exception\JsonRpc\AbstractJsonRpcException;
use idoit\Module\Api\Exception\JsonRpc\AuthenticationException;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use idoit\Module\Api\Exception\JsonRpc\MethodException;
use idoit\Module\Api\Exception\JsonRpc\ParameterException;
use idoit\Module\Api\Exception\JsonRpc\RequestException;
use idoit\Module\Api\Exception\JsonRpc\SystemException;
use idoit\Module\Api\Request\RpcRequest;
use idoit\Module\Api\Response\ResponseStore;
use idoit\Module\Api\Response\RpcErrorResponse;
use idoit\Module\Api\Response\RpcNotificationResponse;
use idoit\Module\Api\Response\RpcResponse;
use idoit\Module\Api\SearchIndexRegister;
use idoit\Module\Api\Session\SessionIdSession;
use idoit\Module\Api\Session\SystemUserSession;
use idoit\Module\Api\Session\UserSession;
use idoit\Module\Cmdb\Search\Index\Data\CategoryCollector;
use idoit\Module\Cmdb\Search\Index\Signals;
use idoit\Module\Search\Index\Engine\Mysql;
use idoit\Module\Search\Index\Manager;
use isys_application;
use isys_exception_api_validation;
use isys_module_licence;
use isys_module_manager;
use isys_tenant;
use isys_tenantsettings;

/**
 * Class JsonRpcController
 *
 * @package idoit\Module\Api\Controller
 */
class JsonRpcController extends AbstractController
{
    use Singleton;

    /**
     * Current processed request
     *
     * @var RpcRequest
     */
    protected $currentRequest;

    /**
     * @var ResponseStore
     */
    protected $responseStore;

    /**
     * UserSession
     *
     * @var UserSession
     */
    protected $session;

    /**
     * Run api model
     *
     * @param \isys_api_model $modelInstance
     * @param array           $parameter
     *
     * @return mixed
     */
    public static function runModel($modelInstance, $parameter)
    {
        return call_user_func_array([
            $modelInstance,
            'route'
        ], $parameter);
    }

    /**
     * Validate Http request
     *
     * @return bool
     * @throws RequestException
     */
    public static function validateHttpRequest()
    {
        if ($_SERVER['REQUEST_METHOD'] != 'POST' || !isset($_SERVER['CONTENT_TYPE']) || substr($_SERVER['CONTENT_TYPE'], 0, 16) != 'application/json') {
            throw new RequestException('This is not a JSON-RPC. The content-type should be application/json, request method should be "post" ' .
                'and the http body should be a valid json-rpc 2.0 package.');
        }

        return true;
    }

    /**
     * Run JsonRpc Requests
     *
     * @return ResponseStore
     * @throws RequestException
     */
    public function run()
    {
        // Get application instance
        $app = \isys_application::instance();

        $this->getLogger()->info('-------------------------------------------------------------');
        $this->getLogger()->debug('i-doit JSON-RPC API v' . $this->getApiVersion() . ' - i-doit ' . $app->info['version'] . ' ' . $app->info['type']);
        $this->getLogger()->debug('Provided raw request: ', $this->getRequestStore()->getRawRequests());

        // Check whether requests are in store
        if ($this->getRequestStore()
            ->hasRequests()) {
            // Get total request number
            $totalRequestNumber = count($this->getRequestStore()->getRequests());
            $this->getLogger()->info($totalRequestNumber . ' request(s) to proceed.');

            // Initialize start timer
            $requestStartTimer = microtime(true);

            foreach ($this->getRequestStore()
                         ->getRequests() as $counter => $request) {
                $this->getLogger()
                    ->info('Processing request ' . ($counter + 1) . '/' . $totalRequestNumber . ': ', $request->getData());
                $this->getLogger()->debug('', $request->getData());

                // Set current request
                $this->setCurrentRequest($request);

                try {
                    try {
                        // Validate request
                        if (!$request->validate()) {
                            throw new RequestException('Request is not valid - aborting.');
                        }

                        // Check whether request is a notification by definition
                        if ($request->isNotification()) {
                            $this->getLogger()->warning('Request is an notification - aborting.');
                            $this->getResponseStore()
                                ->addResponse(new RpcNotificationResponse());
                            continue;
                        }

                        $this->getLogger()->info('Request is valid.');

                        // Login with given credentials and apiKey
                        if (!$this->login($request->getApiKey())) {
                            throw new AuthenticationException('Unable to login using given credentials and apiKey.');
                        }

                        // Disconnect search index for the moment
                        Signals::instance()
                            ->disconnectOnAfterCategoryEntrySave();

                        $this->getLogger()->info('Successfully logged in with user #' . $this->getSession()->getUserId() . ' ' . $this->getSession()->getUsername() . '.');

                        // Check whether api is active
                        if (!$this->isApiActive()) {
                            throw new InternalErrorException('JSON-RPC Api is not active - aborting.');
                        }

                        // Set api language
                        $this->setLanguage($request->getLanguage());
                        $this->getLogger()->info('Language set to \''. $request->getLanguage() .'\'.');

                        // Get model instance by method
                        $modelInstance = $this->getModelByMethod($request);
                        $this->getLogger()->debug('Controller \'' . get_class($modelInstance) . '\' created successfully.');

                        // Check whether authentication is needed
                        if ($modelInstance->needs_login() && !$this->getSession()->isLoggedIn()) {
                            throw new AuthenticationException('Request needs session but no active session exists.');
                        } else if (!$modelInstance->needs_login()) {
                            $this->getLogger()->debug('Controller does not require a valid user session.');
                        }

                        // @see  API-243, API-244  Because of how the bootstrapping works, the tenant variable may not be set in this context.
                        if (isys_application::instance()->tenant === null) {
                            $tenantData = isys_application::instance()->container->get('session')->get_mandator_data();

                            if (is_array($tenantData) && count($tenantData)) {
                                isys_application::instance()->tenant = new isys_tenant(
                                    $tenantData['isys_mandator__title'],
                                    $tenantData['isys_mandator__description'],
                                    $tenantData['isys_mandator__id'],
                                    $tenantData['isys_mandator__db_name'],
                                    $tenantData['isys_mandator__dir_cache']
                                );
                            }
                        }

                        // Build model parameter
                        $modelParameter = $this->buildModelParameter($request);
                        $this->getLogger()->debug('Calculated model parameters based on method "'. $request->getMethod() .'"": ', $modelParameter);

                        $this->getLogger()->debug('Prepare to run request...');

                        // Run model and get raw response
                        if ($rawResponse = self::runModel($modelInstance, $modelParameter)) {
                            $this->getLogger()->info('Response: ', (array)$rawResponse->get_data());

                            // Transform output to an RpcResponse
                            $response = new RpcResponse($request->getId(), $request->getVersion(), $rawResponse->get_data());

                            // Validate response before storing it
                            if ($response->validate()) {
                                $this->getLogger()->info('Response is valid. Add it to the store.');
                                $this->getResponseStore()
                                    ->addResponse($response);
                            } else {
                                $this->getLogger()->error('Response is not valid and will be skipped.');

                                // Invalid response returned
                                throw (new InternalErrorException('Invalid response for request.'))->setData([
                                    'request'  => $request->getData(),
                                    'response' => $rawResponse->get_data()
                                ]);
                            }
                        } else {
                            throw new MethodException('API Method "' . get_class($modelInstance) . '::route()" does not exist.');
                        }
                    } catch (isys_exception_api_validation $e) {
                        throw (new InternalErrorException($e->getMessage()))->setData((array)$e->get_validation_errors());
                    } catch (\isys_exception_api $e) {
                        // Resolve isys_exception_api
                        $exceptionClass = AbstractJsonRpcException::getExceptionByCode($e->getCode());

                        throw new $exceptionClass($e->getMessage());
                    } catch (\isys_exception_auth $e) {
                        throw new AuthenticationException($e->getMessage());
                    } catch (\isys_exception_database_mysql $e) {
                        // Get database error message
                        $errorMessage = $e->getMessage();

                        // Check whether database error refers to a mismatched foreign key constraint error
                        if ($e->getCode() == \MySQL\Error\Server::ER_NO_REFERENCED_ROW_2) {
                            // Reset database error message
                            $errorMessage = 'An database error occurred that indicates that you provided an Id which does not exist. 
                                             Please check all properties which are referencing an entry for validity: ';

                            // Match reference path
                            preg_match('/FOREIGN KEY \((.)*\)/m', $e->getMessage(), $matches);

                            if (is_array($matches) && !empty($matches[0])) {
                                // Add it to error message
                                $errorMessage .= $matches[0];
                            }
                        }

                        throw (new InternalErrorException($errorMessage))->setData([
                            'mysqlErrorCode'    => $e->getCode(),
                            'mysqlErrorMessage' => $e->getMessage(),
                        ]);
                    } catch (AbstractJsonRpcException $e) {
                        throw $e;
                    } catch (Exception $e) {
                        throw new SystemException($e->getMessage());
                    }
                } catch (AbstractJsonRpcException $e) {
                    $this->getResponseStore()
                        ->addResponse(new RpcErrorResponse($request->getId(), $request->getVersion() ?: '2.0', $e->getErrorCode(), $e->getErrorTopic() . ': ' . $e->getMessage(),
                            $e->getData()));

                    $this->getLogger()->error('An exception occured while processing request: ' . $e->getErrorCode() . ' ' . $e->getErrorTopic() . ': ' . $e->getMessage(), (is_array($e->getData()) ? $e->getData() : []));
                } catch (Exception $e) {
                    $this->getResponseStore()
                        ->addResponse(new RpcErrorResponse($request->getId(), $request->getVersion(), -32099, 'Internal error' . ': ' . $e->getMessage(), []));

                    $this->getLogger()->error('An exception occured while processing request: ' . -32099 . ' Internal error: ' . $e->getMessage());
                }

                $this->getLogger()->debug('-------------------------------------------------------------');
            }

            // Initialize request end timer
            $requestEndTimer = microtime(true);

            // Get search index
            $searchIndexRegister = SearchIndexRegister::getObjectOrientedRegister();

            // Initialize index timer
            $indexStartTimer = 0;
            $indexEndTimer = 0;

            // Check whether any categories needs to be reindexed
            if (count($searchIndexRegister) > 0) {
                $this->getLogger()->info('Refresh search index...');

                // Initialize index start timer
                $indexStartTimer = microtime(true);

                // Create needed resources
                $searchEngine = new Mysql(\isys_application::instance()->container->get('database'));
                $collector = new CategoryCollector(\isys_application::instance()->container->get('database'), [], []);
                $manager = new Manager($searchEngine, \isys_application::instance()->container->get('event_dispatcher'));
                $manager->addCollector($collector, 'idoit.cmdb.search.index.category_collector');

                // Get category oriented register
                $categoryOrientedSearchIndexRegister = SearchIndexRegister::getCategoryOrientedRegister();

                // Check which indexing strategy needes less resources
                if (count($searchIndexRegister) <= count($categoryOrientedSearchIndexRegister)) {
                    // Object oriented indexing
                    foreach ($searchIndexRegister as $objectId => $categoryConstants) {
                        $collector->setObjectIds([$objectId]);
                        $collector->setCategoryConstants(array_keys($categoryConstants));
                        $manager->setMode(Manager::MODE_OVERWRITE);
                        $manager->generateIndex();
                    }
                } else {
                    // Category oriented indexing
                    foreach ($categoryOrientedSearchIndexRegister as $categoryConstant => $objectIds) {
                        $collector->setObjectIds(array_keys($objectIds));
                        $collector->setCategoryConstants([$categoryConstant]);
                        $manager->setMode(Manager::MODE_OVERWRITE);
                        $manager->generateIndex();
                    }
                }

                // Initialize index end timer
                $indexEndTimer = microtime(true);

                $this->getLogger()->info('Done');
            }

            // Log some metrics
            $requestTime = $requestEndTimer-$requestStartTimer;
            $this->getLogger()->info('[METRICS] Request time: ' . $totalRequestNumber . ' request(s) took ' . $requestTime . 's.');
            $this->getLogger()->info('[METRICS] Requests per second: ' . ($totalRequestNumber / $requestTime) . 's.');

            $indexTime = $indexEndTimer - $indexStartTimer;
            $this->getLogger()->info('[METRICS] Indexing time: ' . $indexTime .'s.');
            $this->getLogger()->info('[METRICS] Total time: ' . ($requestTime + $indexTime) .'s.');


            return $this->getResponseStore();
        } else {
            $this->getLogger()->error('Provided request is not a valid json remote procedure call or does not include any valid json remote procedure calls.');

            throw new RequestException('Provided request is not a valid json remote procedure call.');
        }

        $this->getLogger()->debug('-------------------------------------------------------------');
    }

    /**
     * Get model by method
     *
     * @param RpcRequest $request
     *
     * @return mixed
     * @throws AuthenticationException
     * @throws MethodException
     * @throws ParameterException
     * @throws RequestException
     */
    public function getModelByMethod(RpcRequest $request)
    {
        $methodPieces = $this->getMethodPieces($request->getMethod());

        // Base model class
        $modelClassName = $methodPieces['modelClassName'];
        $targetMethod = $methodPieces['targetMethod'];

        if (class_exists('isys_module_licence')) {
            // Go sure to set the add-on licence information.
            $licence = new isys_module_licence();
            $licence->verify();

            if (method_exists($licence, 'isThisTheEnd') && $licence->isThisTheEnd()) {
                throw new AuthenticationException('Your i-doit license is missing for over 30 days. Please install a valid license in the i-doit admin center.');
            }
        }

        // Re-load the add-ons.
        isys_module_manager::instance()->module_loader(true);

        // Get model object.
        if (!class_exists($modelClassName)) {
            throw new MethodException('API Namespace "' . $methodPieces['namespace'] . '" (' . $modelClassName . ') does not exist.');
        } // if

        // Initiate the model.
        /** @var \isys_api_model $modelInstance */
        $modelInstance = new $modelClassName();

        global $g_comp_database;

        // Set database if not already done.
        if (!$modelInstance->get_database() && is_object($g_comp_database)) {
            $modelInstance->set_database($g_comp_database);
        }

        $this->validateModel($modelInstance->get_validation(), $request, $targetMethod);

        return $modelInstance;
    }

    /**
     * Build model parameter
     *
     * @param RpcRequest $request
     *
     * @return array
     * @throws RequestException
     */
    public function buildModelParameter(RpcRequest $request)
    {
        $modelPieces = $this->getMethodPieces($request->getMethod());
        $requestParams = $request->getParams();

        // Check if the option was set as last method parameter.
        if (isset($modelPieces['option']) && is_string($modelPieces['option'])) {
            $requestParams['option'] = $modelPieces['option'];
        } // if

        return [
            $modelPieces['targetMethod'],
            $requestParams,
        ];
    }

    /**
     * Get method pieces
     *
     * @param $method
     *
     * @return array
     * @throws RequestException
     */
    public function getMethodPieces($method)
    {
        // Extract request method.
        $methodPieces = explode('.', $method);

        // Fallback to _ exploding. This can be replaced by an ifsetor operation when we switched to php 5.3.
        if (!$methodPieces) {
            $methodPieces = explode('_', $method);
        } // if

        // If the explode went fine, go further and process the request.
        if (count($methodPieces) < 2) {
            // Invalid request.
            throw new MethodException('Request Method should be in this format: namespace.method (Example: cmdb.object)');
        } // if

        return [
            'modelClassName' => 'isys_api_model_' . $methodPieces[0],
            'namespace'      => $methodPieces[0],
            'targetMethod'   => $methodPieces[1],
            'option'         => $methodPieces[2],
        ];
    }

    /**
     * Validate request by model
     *
     * @param array      $modelValidationDefinition
     * @param RpcRequest $request
     * @param            $targetMethod
     *
     * @return bool
     * @throws ParameterException
     */
    public function validateModel(array $modelValidationDefinition, RpcRequest $request, $targetMethod)
    {
        // Check for mandatory parameters.
        if (isset($modelValidationDefinition[$targetMethod]) && is_array($modelValidationDefinition[$targetMethod])) {
            $requestData = $request->getData();

            foreach ($modelValidationDefinition[$targetMethod] as $validationRule) {
                if ($validationRule && !isset($requestData['params'][$validationRule])) {
                    throw new ParameterException('Mandatory parameter \'' . $validationRule . '\' not found in your request.');
                }
            }
        }

        return true;
    }

    /**
     * Check whether api is active
     *
     * @return bool
     */
    public function isApiActive()
    {
        return isys_tenantsettings::get('api.status', 0);
    }

    /**
     * Set request language
     *
     * @param $language
     *
     * @throws Exception
     */
    public function setLanguage($language)
    {
        $session = isys_application::instance()->container->get('session');
        $languageManager = isys_application::instance()->container->get('language');

        // Validate language string
        if (\is_string($language) && !empty($language)) {
            // Prevent constantly switching language if no change occured
            if ($language === $session->get_language() && $language === $languageManager->get_loaded_language()) {
                return;
            }

            // Set language.
            $session->set_language($language);

            // Force the language manager to load the language (reverting change frmo a51f654).
            $languageManager->load($language);
        }
    }

    /**
     * @return UserSession
     */
    public function getSession()
    {
        return $this->session;
    }

    /**
     * @param UserSession $session
     *
     * @return JsonRpcController
     */
    public function setSession($session)
    {
        $this->session = $session;

        return $this;
    }

    /**
     * Get current request
     *
     * @return RpcRequest
     */
    public function getCurrentRequest()
    {
        return $this->currentRequest;
    }

    /**
     * Set current request
     *
     * @param RpcRequest $request
     *
     * @return $this
     */
    public function setCurrentRequest(RpcRequest $request)
    {
        $this->currentRequest = $request;

        return $this;
    }

    /**
     * Get response store
     *
     * @return ResponseStore
     */
    public function getResponseStore()
    {
        return $this->responseStore;
    }

    /**
     * Set response store
     *
     * @param ResponseStore $responseStore
     */
    public function setResponseStore($responseStore)
    {
        $this->responseStore = $responseStore;
    }

    /**
     * Login procedure based on api key
     *
     * @param $apiKey
     *
     * @return bool
     * @throws AuthenticationException
     * @throws \isys_exception_api
     */
    protected function login($apiKey)
    {
        $loginSuccess = $this->getSession()->setApiKey($apiKey)->login();

        if ($loginSuccess && !$this->getSession()->getUserId()) {
            // This might be necessary with users that login via LDAP.
            $this->getSession()->setUserId((int) $this->getSession()->getSessionComponent()->get_user_id());
        }

        return $loginSuccess;
    }

    /**
     * Create session
     *
     * @param string $userName
     * @param string $password
     * @param string $sessionId
     * @param bool   $forceAuthentication
     *
     * @return SessionIdSession
     * @throws AuthenticationException
     */
    private function createSession($userName, $password, $sessionId, $forceAuthentication)
    {
        // Check whether authentication is forced by settings
        if ($forceAuthentication && (empty($userName) || empty($password)) && empty($sessionId)) {
            throw new AuthenticationException('Setting \'api.authenticated-users-only\' is enabled. ' .
                'Please provide valid user credentials by http basic auth or use an existing session id.');
        }

        // Check whether session id is setted by header
        if ($sessionId) {
            $session = new SessionIdSession($sessionId, null, \isys_application::instance()->container->get('session'));
        } else {
            // Check whether authentication is not enforced and credentials are not totally provided
            if (!$forceAuthentication && (empty($userName) || empty($password))) {
                // Fallback to 'systemapi' user
                $session = new SystemUserSession($userName, $password, null, false, \isys_application::instance()->container->get('session'));
            } else {
                // Use standard user session by default
                $session = new UserSession($userName, $password, null, false, \isys_application::instance()->container->get('session'));
            }
        }

        return $session;
    }

    /**
     * JsonRpcController constructor.
     *
     * @param ResponseStore $responseStore
     *
     * @throws AuthenticationException
     */
    public function __construct(ResponseStore $responseStore)
    {
        // Call parent constructor
        parent::__construct();

        /**
         * @todo Priority of login credentials BasicAuth and RpcHeader
         */
        // Create and set session component
        $this->setSession(
            $this->createSession(
                $_SERVER['PHP_AUTH_USER'] ?: \isys_core::header(\isys_core::HTTP_RPCAuthUser, false),
                $_SERVER['PHP_AUTH_PW'] ?: \isys_core::header(\isys_core::HTTP_RPCAuthPass, false),
                \isys_core::header(\isys_core::HTTP_RPCAuthSession, false),
                !!\isys_tenantsettings::get('api.authenticated-users-only', 1)
            )
        );

        // Disable authentication system by default
        $this->disableAuthenticationSystem();

        // Check whether a valid user session is used and auth system is active
        if (get_class($this->getSession()) != SystemUserSession::class && !!isys_tenantsettings::get('auth.active')) {
            // Enable it in cases a specific user session exists and authentication system is not disabled
            $this->enableAuthenticationSystem();
        }

        // Create response store
        $this->responseStore = $responseStore;
    }
}
