<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Dialog;

use idoit\Component\Property\Property;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use idoit\Module\Api\Exception\ValidationException;

/**
 * Class DialogAdmin
 *
 * @package idoit\Module\Api
 */
class DialogAdmin
{
    /**
     * Initialization modes
     */
    const MODE_TABLE    = 1;
    const MODE_CALLBACK = 2;
    const MODE_ARRAY    = 3;

    /**
     * Dialog data
     *
     * @var array
     */
    protected $data;

    /**
     * @var int
     */
    protected $mode;

    /**
     * @var array|Property
     */
    protected $propertyDefinition;

    /**
     * @param array          $data
     * @param array|Property $propertyDefinition
     * @param int            $mode
     *
     * @return DialogAdmin
     */
    public static function byArray(array $data, $propertyDefinition, $mode = self::MODE_ARRAY)
    {
        if (!($propertyDefinition instanceof Property) && !is_array($propertyDefinition)) {
            throw new InternalErrorException('The provided property definition needs to be an array or instance of Property.');
        }

        return new static($data, $propertyDefinition, $mode);
    }

    /**
     * @param string         $tableName
     * @param array|Property $propertyDefinition
     *
     * @return DialogAdmin
     * @throws \isys_exception_database
     * @throws \Exception
     */
    public static function byTableName($tableName, $propertyDefinition)
    {
        if (!($propertyDefinition instanceof Property) && !is_array($propertyDefinition)) {
            throw new InternalErrorException('The provided property definition needs to be an array or instance of Property.');
        }

        // Build sql to retrieve data from database.
        $sql = 'SELECT ' . $tableName . '__id AS id, ' . $tableName . '__title as title, ' . $tableName . '__const as const FROM ' . $tableName . ';';

        // Run query.
        $resource = \isys_application::instance()->container->get('cmdb_dao')->retrieve($sql);

        // Data store.
        $dialogData = [];

        // Check for filled result set.
        if ($resource->num_rows()) {
            while ($row = $resource->get_row()) {
                // Translate title before storing it.
                $dialogData[(int)$row['id']] = \isys_application::instance()->container->get('language')
                    ->get($row['title']);

                // Check whether constant is filled.
                if (!empty($row['const'])) {
                    // Store constant.
                    $dialogData[$row['id'] . '_const'] = $row['const'];
                }
            }
        }

        return self::byArray($dialogData, $propertyDefinition, self::MODE_TABLE);
    }

    /**
     * @return array
     */
    public function getData()
    {
        return $this->data;
    }

    /**
     * @param array $data
     *
     * @return DialogAdmin
     */
    public function setData($data)
    {
        $this->data = $data;

        return $this;
    }

    /**
     * @return array
     */
    public function getPropertyDefinition()
    {
        return $this->propertyDefinition;
    }

    /**
     * @param array|Property $propertyDefinition
     *
     * @return DialogAdmin
     */
    public function setPropertyDefinition($propertyDefinition)
    {
        if (!($propertyDefinition instanceof Property) && !is_array($propertyDefinition)) {
            throw new InternalErrorException('The provided property definition needs to be an array or instance of Property.');
        }

        $this->propertyDefinition = $propertyDefinition;

        return $this;
    }

    /**
     * @return int
     */
    public function getMode()
    {
        return $this->mode;
    }

    /**
     * @param \isys_callback $callback
     * @param \isys_request  $request
     * @param array|Property $propertyDefinition
     *
     * @return DialogAdmin
     * @throws \Exception
     */
    public static function byCallback(\isys_callback $callback, \isys_request $request, $propertyDefinition)
    {
        if (!($propertyDefinition instanceof Property) && !is_array($propertyDefinition)) {
            throw new InternalErrorException('The provided property definition needs to be an array or instance of Property.');
        }

        $data = $callback->execute($request);

        return self::byArray($data, $propertyDefinition, self::MODE_CALLBACK);
    }

    /**
     * Check whether id exists in dialog data
     *
     * @param int $id
     *
     * @return bool
     */
    public function idExists($id)
    {
        return !empty($this->getData()[$id]);
    }

    /**
     * Check whether value exists in dialog data
     *
     * @param string $value
     *
     * @return bool
     */
    public function valueExists($value)
    {
        return in_array($value, $this->getData());
    }

    /**
     * Get value by id
     *
     * @param int $id
     *
     * @return string
     */
    public function getValueById($id)
    {
        return $this->getData()[$id];
    }

    /**
     * Get id by value
     *
     * @param string $value
     *
     * @return false|int
     */
    public function getIdByValue($value)
    {
        return array_search($value, $this->getData());
    }

    /**
     * Find or create id for provided value
     *
     * @param string $value
     *
     * @see API-115 First check for ID (int), then constant and at last actual value.
     * @return null|string|int
     */
    public function findOrCreate($value)
    {
        if ($this->valueExists($value)) {
            // Get id by value
            $value = $this->getIdByValue($value);

            // Check whether it is an integer
            if (is_int($value)) {
                return $value;
            }

            // Check whether value needs to be cast
            if (strpos($value, '_const')) {
                return (int)$value;
            }

            // It is something specific
            return $value;
        }

        return null;
    }

    /**
     * @param int $mode
     *
     * @return DialogAdmin
     */
    public function setMode($mode)
    {
        $this->mode = $mode;

        return $this;
    }

    /**
     * DialogAdmin constructor.
     *
     * @param array          $data
     * @param array|Property $propertyDefinition
     * @param                $mode
     */
    public function __construct(array $data, $propertyDefinition, $mode)
    {
        if (!($propertyDefinition instanceof Property) && !is_array($propertyDefinition)) {
            throw new InternalErrorException('The provided property definition needs to be an array or instance of Property.');
        }

        $this->setData($data)
            ->setPropertyDefinition($propertyDefinition)
            ->setMode($mode);
    }
}
