<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Validation\Logical;

use idoit\Module\Api\Exception\ValidationException;
use idoit\Module\Api\Validation\Validation;

/**
 * Class DialogValue
 *
 * @package Idoit\Module\Api\Validation\Logical
 */
class DialogValue extends Validation
{
    /**
     * Data included in dialog field
     *
     * @var array
     */
    protected $dialogData;

    /**
     * @param array $value
     *
     * @return DialogValue
     */
    public function setValue($value)
    {
        $this->value = is_array($value) ? $value : [$value];

        return $this;
    }

    /**
     * Validate value
     *
     * @return bool
     * @throws \idoit\Module\Api\Exception\ValidationException
     */
    public function validate()
    {
        // Ensure one consistent data type for processing
        $integerValueCount = 0;
        $stringValueCount = 0;

        $propertyType = $this->getPropertyType();
        $dialogValues = (array)$this->getValue();
        $dialogData = $this->getDialogData();

        $dialogValuesCount = count($dialogValues);

        foreach ($dialogValues as $dialogValue) {
            if (is_string($dialogValue)) {
                $stringValueCount++;
            } elseif (is_int($dialogValue)) {
                $integerValueCount++;
            }
        }

        // Check whether provided values are mixed
        if ($integerValueCount !== $dialogValuesCount && $stringValueCount !== $dialogValuesCount) {
            throw new ValidationException($this->getPropertyType(), 'Mixed value types detected. Please provide only string or integer values.');
        }

        /**
         * Validate provided values
         */

        // Integer value handling
        if ($integerValueCount === $dialogValuesCount) {
            /**
             * This branch is relevant for all dialog, dialog+, dialogList and multiSelect
             */

            // Get difference between provided and available data
            $unknownValues = array_diff($dialogValues, array_keys($dialogData));

            // Check whether all provided Ids exists in dialog table
            if (count($unknownValues) !== 0) {
                throw new ValidationException(
                    $this->getPropertyType(),
                    'Id value(s) \'' . implode(',', $unknownValues) . '\' do not exist in dialog table. Please check your provided selection.'
                );

            }
        } elseif ($propertyType === C__PROPERTY__INFO__TYPE__DIALOG) {
            // Get difference between provided and available data
            $unknownValues = [];

            // @see  API-138  We check if the value exists as dialog key or value or constant.
            foreach ($dialogValues as $dialogValue) {
                if (isset($dialogData[$dialogValue]) || in_array($dialogValue, $dialogData, false) || defined($dialogValue)) {
                    continue;
                }

                $unknownValues[] = $dialogValue;
            }

            // Check whether given values exists all in dialog table
            if (count($unknownValues) !== 0) {
                throw new ValidationException(
                    $propertyType,
                    'Unable to resolve given value. Provided value(s) \'' . implode(',', $unknownValues) . '\' do not exist in dialog table.'
                );
            }
        }

        return true;
    }

    /**
     * @return array
     */
    public function getDialogData()
    {
        return $this->dialogData;
    }

    /**
     * @param array $dialogData
     *
     * @return DialogValue
     */
    public function setDialogData(array $dialogData)
    {
        $this->dialogData = $dialogData;

        return $this;
    }
}
