<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\ResponseModifier;

/**
 * IpAddressesProcessor
 *
 * @package    idoit\Module\Api\Model\Category
 */
class IpAddressesProcessor extends AbstractCategoryProcessor implements ResponseModifier
{

    /**
     * Modify api response
     *
     * @param array $response
     *
     * @return array
     */
    public function modifyResponse(array $response)
    {
        // Get request
        $request = $this->getRequest();

        // Check whether api request is 'read'
        if (is_array($request) && $request['option'] === 'read' && is_array($response) && count($response)) {
            // Calculate valid ips
            $netType = reset($response)['net_type']['id'];
            $filteredIps = $netType == C__CATS_NET_TYPE__IPV4 ? $this->getValidIpV4Addresses($request['objID']) : $this->getValidIpV6Addresses($request['objID']);

            // Validate data sets in result
            $response = array_filter($response, function ($result) use ($filteredIps) {
                return $result['object'] !== null && in_array($result['id'], $filteredIps, false);
            });

            // Modify response
            array_walk($response, function (&$result) {
                // Unset virtual property 'assigned_object'
                unset($result['assigned_object']);
            });
        }

        return array_values($response);
    }

    /**
     * Get Ids of valid iPv4 addresses
     *
     * @param $netObjectId
     *
     * @return array
     * @throws \Exception
     */
    private function getValidIpV4Addresses($netObjectId)
    {
        $netDao = \isys_cmdb_dao_category_s_net::instance(\isys_application::instance()->container->get('database'));
        $hostRessource = $netDao->get_assigned_hosts_with_dns($netObjectId, C__RECORD_STATUS__NORMAL);
        $filteredIps = [];
        $hosts = [];

        while ($hostEntry = $hostRessource->get_row()) {
            if (!in_array($hostEntry['isys_cats_net_ip_addresses_list__title'], ['', '0.0.0.0', 'D.H.C.P'], true)) {
                if ($hostEntry['isys_net_dns_domain__title'] !== null && isset($hosts[$hostEntry['isys_cats_net_ip_addresses_list__title']][$hostEntry['isys_catg_ip_list__id']])) {
                    continue;
                }

                // Add it to list
                $filteredIps[] = $hostEntry['isys_catg_ip_list__isys_cats_net_ip_addresses_list__id'];

                // Mark it as proceeded
                $hosts[$hostEntry['isys_cats_net_ip_addresses_list__title']][$hostEntry['isys_catg_ip_list__id']] = true;
            }
        }

        return $filteredIps;
    }

    /**
     * Get valid IPv6 addresses
     *
     * @param $netObjectId
     *
     * @return array
     * @throws \Exception
     */
    private function getValidIpV6Addresses($netObjectId)
    {
        $netDao = new \isys_cmdb_dao_category_s_net(\isys_application::instance()->container->get('database'));
        $hosts = [];
        $filteredIps = [];

        // Get the assigned hosts of our net.
        $hostRessource = $netDao->get_assigned_hosts($netObjectId);

        while ($hostEntry = $hostRessource->get_row()) {
            // Maybe we should check for more than just "empty".
            if (empty($hostEntry['isys_cats_net_ip_addresses_list__title'])) {
                continue;
            }

            $filteredIps[] = $hostEntry['isys_catg_ip_list__isys_cats_net_ip_addresses_list__id'];
        }

        return $filteredIps;
    }
}
