<?php

/**
 * i-doit
 *
 * CMDB API model.
 *
 * @package     i-doit
 * @subpackage  API
 * @author      Dennis Stücken <dstuecken@synetics.de>, Benjamin Heisig <bheisig@synetics.de>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_api_model_cmdb extends isys_api_model
{
    /**
     * CMDB DAO.
     *
     * @var  isys_cmdb_dao
     */
    protected $m_dao;

    /**
     * @var isys_component_dao_lock
     */
    protected $daoLock;

    /**
     * Creates new data.
     *
     * @param   array $p_params Parameters (depends on data method).
     *
     * @return  null
     */
    public function create($p_params)
    {
        return null;
    } // function

    /**
     * Updates data.
     *
     * @param   array $p_params Parameters (depends on data method).
     *
     * @return  null
     */
    public function update($p_params)
    {
        return null;
    } // function

    /**
     * Deletes data.
     *
     * @param   array $p_params Parameters (depends on data method).
     *
     * @return  null
     */
    public function delete($p_params)
    {
        return null;
    } // function

    /* ---------------------------------------------------------------------- */
    /* Helper methods ------------------------------------------------------- */
    /* ---------------------------------------------------------------------- */

    /**
     * Checks parameter 'category' for valid category types. Helper method.
     *
     * @param   mixed $p_category Int, string or array of ints or strings.
     *
     * @return  array
     * @throws  isys_exception_api
     */
    protected function check_category_type($p_category)
    {
        $l_return = [];
        if (is_numeric($p_category))
        {
            $l_category            = intval($p_category);
            $l_return[$l_category] = [];
        }
        else if (is_string($p_category))
        {
            $l_type = null;
            if (stripos($p_category, 'global') !== false)
            {
                $l_type = C__CMDB__CATEGORY__TYPE_GLOBAL;
            }
            else if (stripos($p_category, 'specific') !== false)
            {
                $l_type = C__CMDB__CATEGORY__TYPE_SPECIFIC;
            }
            else if (stripos($p_category, 'custom') !== false)
            {
                $l_type = C__CMDB__CATEGORY__TYPE_CUSTOM;
            }
            else
            {
                throw new isys_exception_api('unkown category type [naming]');
            } // if

            $l_return[$l_type] = [];
        }
        else
        {
            throw new isys_exception_api('unkown category type [format]');
        } // if

        return $l_return;
    } // function

    /* ---------------------------------------------------------------------- */
    /* Generic methods ------------------------------------------------------ */
    /* ---------------------------------------------------------------------- */

    /**
     * Formats data array by mapping and encodes data to UTF-8.
     *
     * @param   array $p_mapping The mapping itself
     * @param   array $p_row     Data array
     *
     * @return  array  Formatted data array
     */
    protected function format_by_mapping(array $p_mapping, $p_row)
    {
        $l_return = [];

        try {
            foreach ($p_mapping as $l_key => $l_map) {
                if (isset($p_row[$l_key]) || is_array($l_map)) {
                    if (is_array($l_map)) {
                        if (is_callable($l_map[0])) {
                            if (strpos($l_map[0], 'str') === 0 || $l_map[0] === '_L') {
                                $l_return[$l_map[1]] = @call_user_func($l_map[0], @$p_row[$l_key]);
                            } else {
                                $l_return[$l_map[1]] = @call_user_func($l_map[0], @$p_row[$l_key], $p_row);
                            }
                        }
                    } elseif (is_scalar($l_map)) {
                        $l_return[$l_map] = $p_row[$l_key];
                    }
                }
            }
        } catch (Exception $e) {
        }

        return $l_return;
    }

    /* ---------------------------------------------------------------------- */
    /* Helping methods ------------------------------------------------------ */
    /* ---------------------------------------------------------------------- */

    /**
     * Api success message. Used as default return message for deleting, updating or creating entries.
     *
     * @param   boolean $p_result
     * @param   string  $p_message
     * @param   integer $p_mysql_id
     *
     * @return  array
     */
    protected function api_success($p_result, $p_message, $p_mysql_id = null)
    {
        return [
            'success' => $p_result,
            'id'      => $p_mysql_id,
            'message' => $p_message
        ];
    }

    /**
     * Reads data.
     *
     * @param   string $model Data method.
     * @param   array  $p_params Parameters (depends on data method).
     *
     * @return  isys_api_model_cmdb  Returns itself.
     * @throws  isys_exception_api
     */
    public function route($model, $p_params)
    {
        // Build model class
        $modelClassName = 'isys_api_model_cmdb_' . $model;

        // Call data method and format data:
        if (class_exists($modelClassName)) {
            if (!is_object($this->m_db)) {
                throw new isys_exception_api('Database not loaded. Your login may did not work!');
            }

            // Initialize DAO:
            $this->m_dao = new isys_cmdb_dao($this->m_db, null);

            $modelInstance = new $modelClassName($this->m_dao);

            $allowedOptions = [
                'read',
                'create',
                'update',
                'delete',
                'quickpurge',
                'purge',
                'recycle',
                'archive',
                'delete',
                'markAsTemplate',
                'markAsMassChangeTemplate',
                'save'
            ];

            if (isset($p_params['option']) && in_array($p_params['option'], $allowedOptions, true)) {
                $modelMethod = $p_params['option'];
            } else {
                $modelMethod = 'read';
            }

            // Check for mandatory parameters.
            $mandatoryFields = $modelInstance->get_validation();

            $p_params = $this->formatParams($p_params);

            if (isset($mandatoryFields[$modelMethod]) && is_array($mandatoryFields[$modelMethod])) {
                foreach ($mandatoryFields[$modelMethod] as $mandatoryField) {
                    if ($mandatoryField && !isset($p_params[$mandatoryField])) {
                        throw new isys_exception_api('Mandatory parameter "' . $mandatoryField . '" not found in your request.', -32602);
                    }
                }
            }

            if (method_exists($modelInstance, $modelMethod)) {
                $this->m_log->info('Method: ' . $modelMethod);
                $this->format($modelInstance->$modelMethod($p_params));
            }
        } else {
            $this->m_log->error('Method "' . $model . '" does not exit.');
            throw new isys_exception_api('API Method "' . $model . '" (' . $modelClassName . ') does not exist.', -32601);
        }

        return $this;
    }

    /**
     * Compares mapped result by 'title'. Used by usort().
     *
     * @param   array $p_arr1
     * @param   array $p_arr2
     *
     * @return  integer
     */
    protected function sort_by_title($p_arr1, $p_arr2)
    {
        return strcasecmp($p_arr1['title'], $p_arr2['title']);
    } // function

    /**
     * @param $params
     *
     * @return mixed
     * @throws Exception
     */
    protected function formatParams($params)
    {
        if (isset($params['title']) && trim($params['title']) !== '') {
            $params['title'] = isys_cmdb_dao_category_g_accounting::instance($this->m_db)->replace_placeholders($params['title'], null, null, null, null, 'isys_obj');
        }

        return $params;
    }

    /**
     * Compares mapped result by 'title' sort DESC. Used by usort().
     *
     * @param   array $p_arr1
     * @param   array $p_arr2
     *
     * @return  integer
     */
    protected function sort_by_title_desc($p_arr1, $p_arr2)
    {
        return strcasecmp($p_arr2['title'], $p_arr1['title']);
    } // function

    /**
     * Validate status
     *
     * @param $status
     *
     * @return bool
     * @throws \idoit\Module\Api\Exception\JsonRpc\ParameterException
     */
    public static function validateStatus($status) {
        // Check whether status is empty
        if ($status === null) {
            return true;
        }

        // Validate status value
        if (!is_int($status) || !in_array($status, [C__RECORD_STATUS__NORMAL, C__RECORD_STATUS__ARCHIVED, C__RECORD_STATUS__DELETED])) {
            throw new \idoit\Module\Api\Exception\JsonRpc\ParameterException('Status parameter is not valid. Please provide an integer value which represents a valid cmdb status id.');
        }

        return true;
    }

    /**
     * @param int $objectId
     *
     * @return void
     * @throws isys_exception_api
     */
    protected function checkObjectLock(int $objectId)
    {
        if ($this->daoLock->is_locked($objectId)) {
            $lockInformation = $this->daoLock->get_lock_information($objectId)->get_row();

            $lockedBy = $this->m_dao->get_obj_name_by_id_as_string($lockInformation['isys_user_session__isys_obj__id']);
            $remainingSeconds = (C__LOCK__TIMEOUT - (time() - strtotime($lockInformation['isys_lock__datetime'])));

            throw new isys_exception_api('The object is currently locked by user "' . $lockedBy . '" for at least ' . $remainingSeconds . ' seconds.');
        }

        $this->daoLock->add_lock($objectId);
    }

    /**
     * @param int $objectId
     *
     * @return void
     * @throws isys_exception_api
     */
    protected function unlockObject(int $objectId)
    {
        $this->daoLock->delete_by_object_id($objectId);
    }

    /**
     * Constructor
     */
    public function __construct()
    {
        if (is_object($this->m_dao)) {
            $this->m_db = $this->m_dao->get_database_component();
        } else {
            global $g_comp_database;

            $this->m_db = $g_comp_database;
        }

        parent::__construct();

        $this->daoLock = isys_component_dao_lock::instance($this->m_db);
    }
}
