<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use idoit\Module\Api\Exception\ValidationException;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\RequestModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\ResponseModifier;
use isys_application;
use isys_cmdb_dao_category_g_location;
use isys_exception_api_validation;

/**
 * Class ModelProcessor
 *
 * @package idoit\Module\Api\Model\Cmdb\Category\Processor
 */
class LocationProcessor extends AbstractCategoryProcessor implements RequestModifier, ResponseModifier
{
    /**
     * Modify api request.
     *
     * @param array $request
     *
     * @return array
     * @throws ValidationException
     * @throws \isys_exception_database
     * @throws isys_exception_api_validation
     */
    public function modifyRequest(array $request)
    {
        if (in_array($request['option'], ['save', 'update', 'create'], true)) {
            // @see API-342 Throw an error if the user tries to write the virtual 'gps' property.
            if (isset($request['data']['gps'])) {
                throw new ValidationException('unknown', "Property 'gps' is not writeable.");
            }
        }

        // @see  API-190  Find the "real" HE to position the object.
        if (isset($request['data']['pos']) && is_numeric($request['data']['pos'])) {
            // First we need to get some information about the location parent.
            $enclosureData = $this->retrievePositionData($request[C__CMDB__GET__OBJECT]);

            // If there is a newly defined parent, we have to get the correct height of that object.
            if (isset($request['data']['parent'])) {
                $rackData = $this->retrieveRackData($request['data']['parent']);

                // Retrieve the rack height and slot sorting of the new parent.
                $enclosureData['rackHeight'] = $rackData['rackHeight'];
                $enclosureData['slotSorting'] = $rackData['slotSorting'];
            }

            // Set the default value, if no rackunits are set.
            if (!is_numeric($enclosureData['objectHeight'])) {
                $enclosureData['objectHeight'] = 1;
            }

            // Set the default value, if no rackunits are set.
            if (!is_numeric($enclosureData['rackHeight'])) {
                $enclosureData['rackHeight'] = 1;
            }

            // @todo  Use `isys_cmdb_dao_category_s_enclosure::C__SLOT_SORTING__DESC` in the future, available starting i-doit 1.13.1.
            if ($enclosureData['slotSorting'] === 'asc') {
                // We don't need further logic, since the rack is sorted ascending.
                return $request;
            }

            /*
             * @see  API-190
             *
             * Calculate the new position for this object. This is necessary since the position is ALWAYS counted from 1 to <rack height> but in the frontend the position
             * can (visibly) change when the slot sorting is set to "descending". So in this example we have a object with 3 HE and a enclosure with 40 HE.
             *
             * We always count ascending - that means if we want to position something on HE 30, the object would go from 33 to 30 (this was defined by product).
             *
             * The formula is: ((<rack height> - <desired position>) - <object height>) + 2
             * (+ 2 since we start with 1 instead of 0 and also we need to add one since we subtract the object height)
             *
             * 1) Want to position in HE 40: ((40 - 40) - 3) + 2 = -1 -> Should trigger an error during validation (since only values between 1 and 40 are allowed)
             * 2) Want to position in HE 38: ((40 - 38) - 3) + 2 =  1 (visually from 40 to 38)
             * 3) Want to position in HE  1: ((40 -  1) - 3) + 2 = 38 (visually from 3 to 1)
             * 4) Want to position in HE 20: ((40 - 20) - 3) + 2 = 19 (visually from 24 to 22)
             */
            $request['data']['pos'] = (($enclosureData['rackHeight'] - $request['data']['pos']) - $enclosureData['objectHeight']) + 2;

            // Provide some validation, since the location DAO does not do this.
            if ($request['data']['pos'] > $enclosureData['rackHeight']) {
                throw new isys_exception_api_validation(
                    'The calculated position inside the rack lies outside the rack.',
                    ['pos' => 'Position needs to be between 1 and ' . $enclosureData['rackHeight'] . ', got ' . $request['data']['pos'] . '.']
                );
            }
        }

        return $request;
    }

    /**
     * Modify api response
     *
     * @param array $response
     *
     * @return array
     */
    public function modifyResponse(array $response)
    {
        // Get request
        $request = $this->getRequest();

        // Check whether api request is 'read' and response object has results
        if (is_array($response) && is_array($request) && $request['option'] === 'read' && count($response)) {
            // Modify response.
            array_walk($response, function (&$result) {
                // First we need to get some information about the location parent.
                $enclosureData = $this->retrievePositionData($result['objID']);

                // @todo  Use `isys_cmdb_dao_category_s_enclosure::C__SLOT_SORTING__DESC` in the future, available starting i-doit 1.13.1.
                if ($enclosureData['slotSorting'] === 'asc') {
                    $from = (int)$result['pos']['title'];
                    $to = $from + $enclosureData['objectHeight'] - 1;
                } else {
                    $from = (int)$enclosureData['rackHeight'] - $result['pos']['title'] + 1;
                    $to = $from - $enclosureData['objectHeight'] + 1;
                }

                // Always display the lower number as "from", the higher number as "to"
                $result['pos']['visually_from'] = min($from, $to);
                $result['pos']['visually_to'] = max($from, $to);
            });
        }

        return array_values($response);
    }

    /**
     * @param int $rackObjectId
     *
     * @return array
     * @throws \isys_exception_database
     */
    private function retrieveRackData($rackObjectId) {
        $objectId = $this->getDao()->convert_sql_id($rackObjectId);

        $sql = "SELECT
            isys_catg_formfactor_list__rackunits AS rackHeight,
            isys_cats_enclosure_list__slot_sorting AS slotSorting
            FROM isys_catg_formfactor_list
            INNER JOIN isys_cats_enclosure_list ON isys_cats_enclosure_list__isys_obj__id = isys_catg_formfactor_list__isys_obj__id
            WHERE isys_catg_formfactor_list__isys_obj__id = {$objectId}
            LIMIT 1;";

        return $this->getDao()->retrieve($sql)->get_row();
    }

    /**
     * @param int $positionedObjectId
     *
     * @return array
     * @throws \isys_exception_database
     */
    private function retrievePositionData($positionedObjectId) {
        $objectId = $this->getDao()->convert_sql_id($positionedObjectId);

        $sql = "SELECT
                parent.isys_catg_formfactor_list__rackunits AS rackHeight,
                myself.isys_catg_formfactor_list__rackunits AS objectHeight,
                isys_cats_enclosure_list__slot_sorting AS slotSorting,
                isys_cats_enclosure_list__vertical_slots_front AS verticalSlotsFront,
                isys_cats_enclosure_list__vertical_slots_rear AS verticalSlotsRear
                FROM isys_obj
                INNER JOIN isys_catg_location_list ON isys_catg_location_list__isys_obj__id = isys_obj__id
                INNER JOIN isys_cats_enclosure_list ON isys_cats_enclosure_list__isys_obj__id = isys_catg_location_list__parentid
                LEFT JOIN isys_catg_formfactor_list AS myself ON myself.isys_catg_formfactor_list__isys_obj__id = isys_obj__id
                LEFT JOIN isys_catg_formfactor_list AS parent ON parent.isys_catg_formfactor_list__isys_obj__id = isys_catg_location_list__parentid
                WHERE isys_obj__id = {$objectId}
                LIMIT 1;";

        return $this->getDao()->retrieve($sql)->get_row();
    }
}
