<?php

use idoit\Module\Api\Category\Descriptor;
use idoit\Module\Api\Category\Entry;
use idoit\Module\Api\Dialog\AdminBuilder;
use idoit\Module\Api\Dialog\DialogAdmin;
use idoit\Module\Api\Exception\JsonRpc\AuthenticationException;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use idoit\Module\Api\Exception\JsonRpc\ParameterException;
use idoit\Module\Api\Exception\ValidationException;
use idoit\Module\Api\Model\Cmdb\Category\Processor\AbstractCategoryProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\ApplicationAssignedObjProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Layer2NetAssignedLogicalPortProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\LogicalPortProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\NetProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\AssignedSubscriptionsProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\ChassisDeviceProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\ClusterMembershipsProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\ClusterMembersProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\ConnectorProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\ContactProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\CustomFieldsProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\GeneralProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\GenericProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\IpAddressesProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\IpProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Layer2NetProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\LocallyAssignedObjectsProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\LocationProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\ModelProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\PersonAssignedGroupsProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\PersonGroupMembersProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\PersonProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\PortProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\RequestModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\ResponseModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\SyncModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\SlaProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\SoftwareAssignmentProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\StorProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\VirtualHostProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\VirtualSwitchProcessor;
use idoit\Module\Api\Model\Cmdb\Category\Processor\WanProcessor;
use idoit\Module\Api\Validation\Logical\DialogValue;
use idoit\Module\Api\Validation\Logical\IdValue;
use idoit\Module\Api\Validation\Logical\UnknownProperty;
use idoit\Module\Api\Validation\PropertyValidationFactory;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Layer2NetAssignProcessor;

/**
 * i-doit
 *
 * API model
 *
 * @package    i-doit
 * @subpackage API
 * @author     Dennis Stücken <dstuecken@synetics.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_cmdb_category extends isys_api_model_cmdb implements isys_api_model_interface
{

    /**
     * Possible options and their parameters
     *
     * @var array
     */
    protected $m_options = [
        'read' => []
    ];

    /**
     * Validation
     *
     * @var array
     */
    protected $m_validation = [];

    /**
     * Data formatting used in format methods
     *
     * @var array
     */
    protected $m_mapping = [];

    /**
     * Category type: global, specific or custom
     *
     * @var int
     */
    protected $m_category_type = null;

    /**
     * Field which includes the content
     *
     * @var string
     */
    protected $m_content_field = null;

    /**
     * Category id
     *
     * @var int
     */
    protected $m_category_id = null;

    /**
     * Category suffix
     *
     * @var string
     */
    protected $m_category_suffix = null;

    /**
     * Category get-param
     *
     * @var string
     */
    protected $m_category_get = null;

    /**
     * Category info from isysgui
     *
     * @var array
     */
    protected $m_category_info = null;

    /**
     * Category DAO
     *
     * @var isys_cmdb_dao_category
     */
    protected $m_category_dao = null;

    /**
     * Category source table
     *
     * @var string
     */
    protected $m_category_source_table = null;

    /**
     * Registry of category processors
     *
     * @var array
     */
    protected static $categoryProcessors = [
        // Global categories.
        'C__CATG__APPLICATION'                       => SoftwareAssignmentProcessor::class,
        'C__CATG__ASSIGNED_SUBSCRIPTIONS'            => AssignedSubscriptionsProcessor::class,
        'C__CATG__CLUSTER_MEMBERSHIPS'               => ClusterMembershipsProcessor::class,
        'C__CATG__CLUSTER_MEMBERS'                   => ClusterMembersProcessor::class,
        'C__CATG__CONNECTOR'                         => ConnectorProcessor::class,
        'C__CATG__CONTACT'                           => ContactProcessor::class,
        'C__CATG__GLOBAL'                            => GeneralProcessor::class,
        'C__CATG__IP'                                => IpProcessor::class,
        'C__CATG__LOCATION'                          => LocationProcessor::class,
        'C__CATG__MODEL'                             => ModelProcessor::class,
        'C__CATG__NETWORK_PORT'                      => PortProcessor::class,
        'C__CATG__NETWORK_LOG_PORT'                  => LogicalPortProcessor::class,
        'C__CATG__OBJECT'                            => LocallyAssignedObjectsProcessor::class,
        'C__CATG__OPERATING_SYSTEM'                  => SoftwareAssignmentProcessor::class,
        'C__CATG__SLA'                               => SlaProcessor::class,
        'C__CATG__STORAGE_DEVICE'                    => StorProcessor::class,
        'C__CATG__VIRTUAL_HOST'                      => VirtualHostProcessor::class,
        'C__CATG__VIRTUAL_SWITCH'                    => VirtualSwitchProcessor::class,
        'C__CATG__WAN'                               => WanProcessor::class,
        // Specific categories.
        'C__CATS__APPLICATION_ASSIGNED_OBJ'          => ApplicationAssignedObjProcessor::class,
        'C__CATS__CHASSIS_DEVICES'                   => ChassisDeviceProcessor::class,
        'C__CATS__LAYER2_NET'                        => Layer2NetProcessor::class,
        'C__CATS__LAYER2_NET_ASSIGN'                 => Layer2NetAssignProcessor::class,
        'C__CATS__LAYER2_NET_ASSIGNED_LOGICAL_PORTS' => Layer2NetAssignedLogicalPortProcessor::class,
        'C__CATS__NET'                               => NetProcessor::class,
        'C__CATS__NET_IP_ADDRESSES'                  => IpAddressesProcessor::class,
        'C__CATS__PERSON'                            => PersonProcessor::class,
        'C__CATS__PERSON_ASSIGNED_GROUPS'            => PersonAssignedGroupsProcessor::class,
        'C__CATS__PERSON_GROUP_MEMBERS'              => PersonGroupMembersProcessor::class,
        'C__CATS__PERSON_MASTER'                     => PersonProcessor::class,
        // Custom category
        'C__CATG__CUSTOM_FIELDS'                     => CustomFieldsProcessor::class,
    ];

    /**
     * Constructor
     *
     * @param \isys_cmdb_dao $p_dao
     */
    public function __construct(isys_cmdb_dao $p_dao)
    {
        $this->m_dao = $p_dao;
        parent::__construct();
    } // function

    /**
     * Read category data.
     * Parameter example:
     * [
     *    'objID'  => 132,
     *    'catgID' => 10
     * ]
     *
     * Or:
     * [
     *    'objID'  => 132,
     *    'catsID' => 20
     * ]
     *
     * Or:
     * [
     *    'objID'    => 132,
     *    'category' => "C__CATG__CPU"
     * ]
     *
     * @param array $p_params
     *
     * @return array
     * @throws isys_exception_api
     * @throws isys_exception_auth
     * @throws isys_exception_database
     * @throws ValidationException
     * @throws Exception
     */
    public function read($p_params)
    {
        // Init
        $l_return = [];
        $i = 0;
        $l_object_id = @$p_params[C__CMDB__GET__OBJECT];
        // @see API-317
        $entryId = $p_params['category_id'] ?? null;

        if (!is_scalar($l_object_id) || !is_numeric($l_object_id)) {
            throw new isys_exception_api('Parameter ' . C__CMDB__GET__OBJECT . ' has to be numeric.', -32602);
        }

        // Validate object id
        if (!$l_object_id || $l_object_id < 1) {
            throw new isys_exception_api('Parameter ' . C__CMDB__GET__OBJECT . ' invalid: ID must be positive and higher than one.', -32602);
        }

        if ($entryId !== null && (!is_int($entryId) || $entryId <= 0)) {
            throw new isys_exception_api('Parameter category_id invalid: ID must be numeric and higher than one.', -32602);
        }

        /**
         * Status handling
         *
         * @see API-99
         */
        if (isset($p_params['status'])) {
            // Valid statuses
            $statuses = [
                C__RECORD_STATUS__NORMAL,
                C__RECORD_STATUS__ARCHIVED,
                C__RECORD_STATUS__DELETED
            ];

            // Validate format
            if (!is_int($p_params['status']) && !is_string($p_params['status'])) {
                throw new ValidationException('status', 'Please provide a valid statusId as integer or a status constant as string value.');
            }

            // Validate constant value
            if (is_string($p_params['status'])) {
                if (!defined($p_params['status'])) {
                    throw new ValidationException('status', 'Please provide a valid status constant.');
                }

                $p_params['status'] = constant($p_params['status']);
            }

            // Validate integer and resolve it!
            if (is_int($p_params['status']) && $p_params['status'] !== -1 && !in_array($p_params['status'], $statuses, false)) {
                throw new ValidationException('status', 'Please provide a valid statusId.');
            }

            $l_status = $p_params['status'];
        } else {
            $l_status = C__RECORD_STATUS__NORMAL;
        }

        // Process data
        if (($l_cat = $this->prepare($p_params))) {
            $categoryProcessor = $this->getCategoryProcessor($l_cat, $p_params);

            if ($this->useAuth) {
                isys_auth_cmdb_categories::instance()
                    ->check_rights_obj_and_category(isys_auth::VIEW, $l_object_id, $l_cat->get_category_const());
            }

            if (!method_exists($l_cat, 'get_data')) {
                throw new isys_exception_api('get_data method does not exist for ' . get_class($l_cat), -32601);
            }

            // Check whether processor wants to modify response
            if ($categoryProcessor instanceof RequestModifier) {
                $p_params = $categoryProcessor->modifyRequest($p_params);
            }

            $condition = null;

            // Condition-Handling and retrieve data
            if (isset($p_params['condition'])) {
                $condition = urldecode($p_params['condition']);
            }

            $l_catentries = $l_cat->get_data_as_array($entryId, $l_object_id, $condition, null, ($l_status === -1 ? null : $l_status));

            // Count category result
            if (count($l_catentries)) {
                if (!isset($p_params['raw']) || !$p_params['raw']) {
                    $l_properties = $l_cat->get_properties();

                    // Get content field
                    $l_content_field = $this->get_content_field();

                    $offset = 0;
                    $length = null;

                    if (isset($p_params['offset']) && is_numeric($p_params['offset']) && $p_params['offset'] > 0) {
                        $offset = (int)$p_params['offset'];
                    }

                    if (isset($p_params['limit']) && is_numeric($p_params['limit']) && $p_params['limit'] > 0) {
                        $length = (int)$p_params['limit'];
                    }

                    $l_catentries = array_slice($l_catentries, $offset, $length);

                    // Format category result
                    foreach ($l_catentries as $l_row) {
                        // Set object and category entry id
                        $l_return[$i]['id'] = $l_row[$this->get_category_source_table('__id')];
                        $l_return[$i][C__CMDB__GET__OBJECT] = $l_row[$l_cat->get_object_id_field()];

                        // Property-Handling
                        foreach ($l_properties as $l_key => $l_propdata) {
                            if (is_string($l_key)) {
                                if (isset($l_propdata[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][0], $l_propdata[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1])) {
                                    if (is_object($l_propdata[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][0])) {
                                        $l_method = $l_propdata[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1];
                                        $l_return[$i][$l_key] = $l_propdata[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][0]->$l_method($l_row[$l_propdata[C__PROPERTY__DATA][$l_content_field]],
                                            $l_row);
                                    } elseif (class_exists($l_propdata[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][0])) {
                                        /**
                                         * @see API-30 Execute p_arData callback here
                                         *      to guarantee usage of right isys_request instance
                                         *      with filled properties.
                                         */
                                        if ($l_propdata['ui']['params']['p_arData'] instanceof isys_callback) {
                                            // Create and setup request for callback
                                            $request = isys_request::factory();
                                            $request->set_object_id($l_row[$this->get_category_source_table('__isys_obj__id')])
                                                ->set_category_data_id($l_row[$this->get_category_source_table('__id')])
                                                ->set_row($l_row);

                                            // Execute callback and save returned value in data array
                                            $l_propdata['ui']['params']['p_arData'] = $l_propdata['ui']['params']['p_arData']->execute($request);
                                        }

                                        $l_helper = new $l_propdata[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][0]($l_row, $this->m_dao->get_database_component(),
                                            $l_propdata[C__PROPERTY__DATA], $l_propdata[C__PROPERTY__FORMAT], $l_propdata[C__PROPERTY__UI], false);

                                        // Set the Unit constant for the convert-helper
                                        if ($l_propdata[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1] === 'convert' && method_exists($l_helper, 'set_unit_const')) {
                                            $l_row_unit = $l_properties[$l_propdata[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__UNIT]][C__PROPERTY__DATA][$l_content_field];
                                            $l_helper->set_unit_const($l_row[$l_row_unit]);
                                        } // if

                                        // Call callback
                                        $l_return[$i][$l_key] = call_user_func([
                                            $l_helper,
                                            $l_propdata[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1]
                                        ], $l_row[$l_propdata[C__PROPERTY__DATA][$l_content_field]]);

                                        /**
                                         * @see API-104 This will prevent empty array outputs
                                         */
                                        if (is_array($l_return[$i][$l_key]) && count($l_return[$i][$l_key]) === 0) {
                                            $l_return[$i][$l_key] = null;
                                        }

                                        // Set data id
                                        $p_params['data']['id'] = $this->get_category_id();

                                        // Retrieve data from isys_export_data
                                        if ($l_return[$i][$l_key] instanceof isys_export_data) {
                                            $l_export_data = $l_return[$i][$l_key];

                                            /** @var isys_export_data $l_export_data */
                                            $l_return[$i][$l_key] = $l_export_data->get_data();
                                        } // if
                                    } // if

                                } else {
                                    $l_return[$i][$l_key] = $l_row[$l_propdata[C__PROPERTY__DATA][$l_content_field]];

                                    if ($l_propdata[C__PROPERTY__DATA][C__PROPERTY__DATA__ENCRYPT]) {
                                        // @see  API-28
                                        $l_return[$i][$l_key] = isys_helper_crypt::decrypt($l_return[$i][$l_key]);
                                    }
                                }

                                // @see API-251
                                if (isys_tenantsettings::get('api.strip-tags', 0) && $l_propdata[C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE] == C__PROPERTY__INFO__TYPE__COMMENTARY) {
                                    $l_return[$i][$l_key] = trim(strip_tags($l_return[$i][$l_key]));
                                }

                                unset($l_helper_class, $l_helper);
                            }
                        }

                        $i++;
                    }
                } else {
                    $l_return = $l_catentries;
                }
            }

            // Check whether processor wants to modify response
            if ($categoryProcessor instanceof ResponseModifier) {
                $l_return = $categoryProcessor->modifyResponse($l_return);
            }
        }

        return $l_return;
    }

    /**
     * Prepare api request
     *
     * @param array $p_params
     *
     * @return bool|isys_cmdb_dao_category
     * @throws isys_exception_api
     * @throws isys_exception_database
     */
    protected function prepare($p_params)
    {
        // Get category params.
        [$l_get_param, $l_cat_suffix] = $this->prepare_category_params($p_params);

        $this->set_category_suffix($l_cat_suffix);

        // Convert constant to ID.
        if (is_string($p_params[$l_get_param]) && defined($p_params[$l_get_param])) {
            $p_params[$l_get_param] = constant($p_params[$l_get_param]);
        }

        $l_isysgui = [];

        // Get category info.
        if (is_numeric($p_params[$l_get_param])) {
            $l_isysgui = $this->m_dao->get_isysgui('isysgui_cat' . $l_cat_suffix, (int)$p_params[$l_get_param])
                ->__to_array();

            $this->set_category_id($p_params[$l_get_param]);
        }

        // Check if category exists.
        if (!count($l_isysgui)) {
            throw new isys_exception_api('Unable to find desired category with ID "' . $p_params[$l_get_param] . '". Please check the delivered category ID and try again',
                -32602);
        }

        $this->set_category_info($l_isysgui);

        // Check if class exists.
        if (!class_exists($l_isysgui["isysgui_cat{$l_cat_suffix}__class_name"])) {
            throw new isys_exception_api('Category DAO "' . $l_isysgui["isysgui_cat{$l_cat_suffix}__class_name"] . '" does not exist.', -32602);
        }

        $_GET[$l_get_param] = $this->get_category_id();

        // Set category DAO
        if (($l_cat = new $l_isysgui["isysgui_cat{$l_cat_suffix}__class_name"]($this->m_dao->get_database_component()))) {
            if (method_exists($l_cat, 'set_catg_custom_id')) {
                /** @var $l_cat isys_cmdb_dao_category_g_custom_fields */
                $l_cat->set_catg_custom_id($this->get_category_id())->set_catgory_const($p_params['category']);
            }

            /**
             * @see API-72 Check whether selected category is virtual by definition
             */
            if (Descriptor::isCategoryVirtual($l_cat->get_category_const())) {
                throw new isys_exception_api('Virtual category ' . _L($l_cat->getCategoryTitle()) . ' cannot be handled by the API.');
            }

            // Set source table.
            $this->set_category_source_table($l_cat->get_table());

            return $l_cat;
        } else {
            // Unable to instantiate DAO class.
            throw new isys_exception_api('Unable to instantiate category DAO "' . $l_isysgui["isysgui_cat{$l_cat_suffix}__class_name"] . '".', -32602);
        }
    }

    /**
     * Prepare category parameter for globa, specific and custom
     *
     * @param $p_params
     *
     * @return array
     * @throws isys_exception_api
     */
    private function prepare_category_params(&$p_params)
    {
        // Process Parameters
        if (isset($p_params[C__CMDB__GET__CATS]) && $p_params[C__CMDB__GET__CATS]) {
            $l_get_param = C__CMDB__GET__CATS;
            $l_cat_suffix = 's';

            $this->set_category_type(C__CMDB__CATEGORY__TYPE_SPECIFIC);
            $this->set_content_field(C__PROPERTY__DATA__FIELD);
        } else {
            if (isset($p_params[C__CMDB__GET__CATG]) && $p_params[C__CMDB__GET__CATG]) {
                $l_get_param = C__CMDB__GET__CATG;
                $l_cat_suffix = 'g';

                $this->set_category_type(C__CMDB__CATEGORY__TYPE_GLOBAL);
                $this->set_content_field(C__PROPERTY__DATA__FIELD);
            } else {
                if (isset($p_params[C__CMDB__GET__CATG_CUSTOM]) && $p_params[C__CMDB__GET__CATG_CUSTOM]) {
                    $l_get_param = C__CMDB__GET__CATG_CUSTOM;
                    $l_cat_suffix = 'g_custom';

                    $this->set_category_type(C__CMDB__CATEGORY__TYPE_CUSTOM);
                    $this->set_content_field(C__PROPERTY__DATA__FIELD_ALIAS);
                } else {
                    if (isset($p_params['category']) && is_string($p_params['category'])) {
                        if (defined($p_params['category'])) {
                            // Default is a global category
                            $l_get_param = C__CMDB__GET__CATG;
                            $l_cat_suffix = 'g';

                            if (strpos($p_params['category'], 'C__CATG') === 0 && strpos($p_params['category'], 'C__CATG__CUSTOM_FIELDS_') !== 0) {
                                $this->set_category_type(C__CMDB__CATEGORY__TYPE_GLOBAL);
                                $this->set_content_field(C__PROPERTY__DATA__FIELD);
                            } else {
                                if (strpos($p_params['category'], 'C__CATS') === 0) {
                                    $l_get_param = C__CMDB__GET__CATS;
                                    $l_cat_suffix = 's';

                                    $this->set_category_type(C__CMDB__CATEGORY__TYPE_SPECIFIC);
                                    $this->set_content_field(C__PROPERTY__DATA__FIELD);
                                } else if (strpos($p_params['category'], 'C__CMDB__SUBCAT') === 0) {
                                    /**
                                     * @todo see ID-948 and ID-934
                                     */
                                    $l_dao = new isys_cmdb_dao($this->m_dao->get_database_component());

                                    // Try to retrieve category by constant of any type
                                    $l_cat_data = $l_dao->get_cat_by_const($p_params['category']);

                                    // Is there a result?
                                    if (is_array($l_cat_data) && count($l_cat_data)) {
                                        if (isset($l_cat_data['type'])) {
                                            // Check for type now
                                            if ($l_cat_data['type'] == C__CMDB__CATEGORY__TYPE_SPECIFIC) {
                                                $l_get_param = C__CMDB__GET__CATS;
                                                $l_cat_suffix = 's';

                                                $this->set_category_type(C__CMDB__CATEGORY__TYPE_SPECIFIC);
                                                $this->set_content_field(C__PROPERTY__DATA__FIELD);
                                            } else if ($l_cat_data['type'] == C__CMDB__CATEGORY__TYPE_GLOBAL) {
                                                $l_get_param = C__CMDB__GET__CATG;
                                                $l_cat_suffix = 'g';

                                                $this->set_category_type(C__CMDB__CATEGORY__TYPE_GLOBAL);
                                                $this->set_content_field(C__PROPERTY__DATA__FIELD);
                                            } // if
                                        } // if
                                    } // if
                                } else {
                                    $l_get_param = C__CMDB__GET__CATG_CUSTOM;
                                    $l_cat_suffix = 'g_custom';

                                    $this->set_category_type(C__CMDB__CATEGORY__TYPE_CUSTOM);
                                    $this->set_content_field(C__PROPERTY__DATA__FIELD_ALIAS);
                                } // if
                            } // if

                            $p_params[$l_get_param] = $p_params['category'];
                        } else {
                            throw new isys_exception_api(sprintf('Category "%s" not found. Delete your i-doit cache if you are sure it exists.', $p_params['category']),
                                -32602);
                        }
                    } else {
                        throw new isys_exception_api('Category type missing. You must specify the parameter int \'catsID\', int \'catgID\', int \'customID\' or string \'category\' ' .
                            'in order to identify the corresponding category you would like to use.', -32602);
                    } // if
                } // if
            } // if
        } // if

        return [$l_get_param, $l_cat_suffix];
    } // function

    /**
     * Set the category type
     *
     * @param int $p_category_type
     *
     * @return $this
     */
    protected function set_category_type($p_category_type)
    {
        $this->m_category_type = $p_category_type;

        return $this;
    } // function

    /**
     * Set the content field
     *
     * @param string $p_content_field
     *
     * @return $this
     */
    protected function set_content_field($p_content_field)
    {
        $this->m_content_field = $p_content_field;

        return $this;
    } // function

    /**
     * Set category suffix
     *
     * @param string $p_category_suffix
     *
     * @return $this
     */
    protected function set_category_suffix($p_category_suffix)
    {
        $this->m_category_suffix = $p_category_suffix;

        return $this;
    } // function

    /**
     * Set category id
     *
     * @param int $p_category_id
     *
     * @return $this
     */
    protected function set_category_id($p_category_id)
    {
        $this->m_category_id = $p_category_id;

        return $this;
    } // function

    /**
     * Set category info from isysgui
     *
     * @param array $p_category_info
     *
     * @return $this
     */
    protected function set_category_info($p_category_info)
    {
        $this->m_category_info = $p_category_info;

        return $this;
    } // function

    /**
     * Is category type global?
     *
     * @return bool
     */
    protected function is_global()
    {
        return (C__CMDB__CATEGORY__TYPE_GLOBAL == $this->get_category_type());
    } // function

    /**
     * Set category source table
     *
     * @param string $p_category_source_table
     *
     * @return $this
     */
    protected function set_category_source_table($p_category_source_table)
    {
        $this->m_category_source_table = $p_category_source_table;

        return $this;
    } // function

    /**
     * Get category id
     *
     * @return int
     */
    public function get_category_id()
    {
        return $this->m_category_id;
    } // function

    /**
     * Is category type custom?
     *
     * @return bool
     */
    protected function is_custom()
    {
        return (C__CMDB__CATEGORY__TYPE_CUSTOM == $this->get_category_type());
    } // function

    /**
     * Get category's source table
     *
     * @param string $p_add
     *
     * @return string
     */
    protected function get_category_source_table($p_add = '')
    {
        return $this->m_category_source_table . $p_add;
    } // function

    /**
     * Get content field
     *
     * @return string
     */
    protected function get_content_field()
    {
        return $this->m_content_field;
    } // function

    /**
     * Create category entry.
     *
     * @param   array $p_params Parameters (depends on data method)
     *
     * @return  isys_api_model_cmdb Returns itself.
     * @throws  Exception
     * @throws  isys_exception_api
     * @throws  isys_exception_api_validation
     */
    public function create($p_params)
    {
        $l_id = $this->sync_wrapper($p_params, isys_import_handler_cmdb::C__CREATE);

        if ($l_id === true || $l_id > 0) {
            // Update 'isys_obj__updated'
            $this->m_dao->object_changed($p_params[C__CMDB__GET__OBJECT]);

            if (is_bool($l_id)) {
                $l_return['id'] = null;

                if ($l_id === true) {
                    $l_return['message'] = 'Category entry was already existing and has been updated.';
                }
            } else {
                $l_return['id'] = $l_id;
                $l_return['message'] = 'Category entry successfully created.';
            }

            $l_return['success'] = true;
        } else {
            $l_return['id'] = null;
            $l_return['message'] = 'Error while creating category entry';
            $l_return['success'] = false;
        } // if

        // Add deprecation info to message
        if (!empty($l_return['message'])) {
            $l_return['message'] .= ' [This method is deprecated and will be removed in a feature release. Use \'cmdb.category.save\' instead.]';
        }

        return $l_return;
    } // function

    /**
     * Get category processor
     *
     * @param isys_cmdb_dao_category $categoryDao
     * @param array                  $request
     *
     * @return AbstractCategoryProcessor
     * @throws Exception
     */
    public function getCategoryProcessor(isys_cmdb_dao_category $categoryDao, array $request)
    {
        // Use GenericProcessor as default
        $categoryProcessorClass = GenericProcessor::class;

        // Check whether category has an specific processor
        if (isset(self::$categoryProcessors[$categoryDao->get_category_const()])) {
            $categoryProcessorClass = self::$categoryProcessors[$categoryDao->get_category_const()];
        } else if ($categoryDao instanceof isys_cmdb_dao_category_g_custom_fields) {
            $categoryProcessorClass = CustomFieldsProcessor::class;
        }

        // Check whether processor is an child of AbstractCategoryProcessor
        if (!is_subclass_of($categoryProcessorClass, AbstractCategoryProcessor::class)) {
            throw new Exception('Category processor \'' . $categoryProcessorClass . '\' does not extend AbstractCategoryProcessor.');
        }

        // Create needed processor and return it!
        return new $categoryProcessorClass($request, $categoryDao);
    }

    /**
     * Validate request
     *
     * Method to validate parameter types
     *
     * @see API-79
     *
     * @param $request
     * @param $properties
     *
     * @return bool
     *
     * @throws isys_exception_api_validation
     */
    public function validateRequest($request, $properties)
    {
        // Validation error store
        $validationErrors = [];
        $propertiesCount = 0;
        // Iterate over provided property names via request
        foreach ($request as $propertyName => $value) {
            try {
                // Validation: Id values
                if ($propertyName === 'id' || $propertyName === 'category_id') {
                    (new IdValue($propertyName, $value, null))->validate();
                    continue;
                }

                // Validation: UnknownProperty
                (new UnknownProperty($propertyName, $value, $properties[$propertyName]))->validate();

                // Do not validate null values
                if ($value === null) {
                    // @see API-359 Setting values empty should still be considered a used property.
                    $propertiesCount++;
                    continue;
                }

                // Validation: Property type
                PropertyValidationFactory::getValidator($propertyName, $value, $properties[$propertyName])->validate();
                $propertiesCount++;
            } catch (ValidationException $e) {
                $validationErrors[$propertyName] = $e->getMessage();

                continue;
            }
        }

        if ($propertiesCount < 1) {
            throw new isys_exception_api_validation('There are no usable properties for the category in the request.', [
                'Unusable keys' => array_keys($request),
                'Available' => array_keys($properties)
            ]);
        }

        // Check for failed validations
        if (!empty($validationErrors)) {
            // Throw an validation exception
            throw new isys_exception_api_validation('There was an validation error', $validationErrors);
        }

        return true;
    }

    /**
     * Sync data
     *
     * @param array $p_params
     * @param int   $p_mode
     *
     * @return bool
     * @throws \Exception
     * @throws \isys_exception_api
     * @throws \isys_exception_api_validation
     */
    protected function sync_wrapper($p_params, $p_mode)
    {
        global $g_comp_database;
        /** @var isys_cmdb_dao_dialog_admin $l_dialog_admin */
        $l_dialog_admin = isys_cmdb_dao_dialog_admin::instance($g_comp_database);
        // Init
        $l_object_id = isset($p_params[C__CMDB__GET__OBJECT]) ? $p_params[C__CMDB__GET__OBJECT] : false;

        // Validate object id
        if (!$l_object_id || $l_object_id < 2) {
            throw new isys_exception_api('Object id invalid. ID must be positive and higher than two.', -32602);
        } // if

        if (!isset($p_params['data']) || !is_array($p_params['data'])) {
            throw new isys_exception_api('Mandatory array parameter \'data\' missing or wrong format given.');
        } // if

        try {
            // Process data
            if (($l_cat = $this->prepare($p_params))) {
                // Get category processor
                $categoryProcessor = $this->getCategoryProcessor($l_cat, $p_params);
                // Prevent foreign key issues and check for object existence
                if (!$l_cat->obj_exists($l_object_id)) {
                    throw new isys_exception_api(sprintf('Object with id "%s" does not exist.', $l_object_id));
                }

                if ($this->useAuth) {
                    $checkRight = ($p_mode == isys_import_handler_cmdb::C__UPDATE ? isys_auth::EDIT : isys_auth::CREATE);

                    $categoryConstant = $l_cat->get_category_const();

                    // @see  API-211  If we process a custom category, we need to get the specific constant.
                    if ($l_cat instanceof isys_cmdb_dao_category_g_custom_fields) {
                        $categoryConstant = $l_cat->get_catg_custom_const();
                    }

                    isys_auth_cmdb_categories::instance()
                        ->check_rights_obj_and_category($checkRight, $l_object_id, $categoryConstant);
                }

                $l_cat_suffix = $this->get_category_suffix();
                $l_isysgui = $this->get_category_info();

                $l_cat->set_object_id($l_object_id);
                $l_cat->set_object_type_id($l_cat->get_objTypeID($l_object_id));

                // Check whether categoryProcessor wants to modify request before further processing
                if ($categoryProcessor instanceof RequestModifier) {
                    $p_params = $categoryProcessor->modifyRequest($p_params);

                    // Check whether request modifier wants to change mode
                    if ($p_mode == isys_import_handler_cmdb::C__CREATE && $p_params['option'] === 'update') {
                        $p_mode = isys_import_handler_cmdb::C__UPDATE;
                    }
                }

                if (method_exists($l_cat, 'sync')) {
                    // Get properties
                    $l_properties = $l_cat->get_properties();

                    // Validate request
                    if (isys_tenantsettings::get('api.validation', 1)) {
                        $this->validateRequest($p_params['data'], $l_properties);
                    }

                    // @see  API-22  Special case for our lovely reverse contact assignment categories.
                    if (in_array($l_cat->get_category_const(),
                        ['C__CATS__ORGANIZATION_CONTACT_ASSIGNMENT', 'C__CATS__PERSON_CONTACT_ASSIGNMENT', 'C__CATS__PERSON_GROUP_CONTACT_ASSIGNMENT'])
                    ) {
                        $dataObject = $p_params['data']['object'];
                        $p_params['data']['object'] = $p_params[C__CMDB__GET__OBJECT];
                        $l_object_id = $p_params[C__CMDB__GET__OBJECT] = $dataObject;

                        if (!$l_cat->obj_exists($l_object_id)) {
                            throw new isys_exception_api(sprintf('Object with id "%s" does not exist.', $l_object_id));
                        }

                        $l_cat->set_object_id($l_object_id);
                        $l_cat->set_object_type_id($l_cat->get_objTypeID($l_object_id));
                    }
                    // Try to validate incoming data
                    $l_validation = $l_cat->validate($p_params['data']);

                    if (is_array($l_validation) && count($l_validation) > 0) {
                        $l_validation_errors = '';
                        foreach ($l_validation AS $l_field => $l_problem) {
                            $l_validation_errors .= $l_field . '(' . $l_properties[$l_field][C__PROPERTY__DATA][C__PROPERTY__DATA__TYPE] . '): ' . $l_problem . ', ';
                        } // foreach

                        throw new isys_exception_api_validation('There was a validation error: ' . rtrim($l_validation_errors, ', '), $l_validation);
                    } else {
                        if ($l_validation === true) {
                            // Create sync conform data array
                            $l_data = [];

                            // Prepare category ressource
                            if ($l_isysgui["isysgui_cat{$l_cat_suffix}__list_multi_value"] == 1) {
                                // Mode: Update
                                if ($p_mode == isys_import_handler_cmdb::C__UPDATE) {
                                    // Set category id
                                    if (isset($p_params['data']['category_id']) && $p_params['data']['category_id'] > 0) {
                                        $l_data['data_id'] = $p_params['data']['category_id'];
                                    } else {
                                        if (isset($p_params['data']['id']) && $p_params['data']['id'] > 0) {
                                            $l_data['data_id'] = $p_params['data']['id'];
                                        } else {
                                            $l_data['data_id'] = null;
                                        }
                                    }

                                    // Check for category entry id first
                                    if (isset($l_data['data_id'])) {
                                        // Try to get specified entry
                                        $l_catentries = $l_cat->get_data_as_array($l_data['data_id'], $l_object_id);

                                        if (count($l_catentries) != 1) {
                                            // Category entry does not exist or the object does not own it.
                                            throw new isys_exception_api(sprintf('Unable to find a category entry with id %d for object %d.', $l_data['data_id'], $l_object_id));
                                        }
                                    } else {
                                        throw new isys_exception_api('Please define the category entry you want to update by setting data.id or data.category_id.');
                                    }
                                }

                                $l_cat->set_list_id($l_data['data_id']);

                                /**
                                 * @todo This causes massive performance problems
                                 */
                                //$l_catentries = $l_cat->get_data_as_array($l_data['data_id'], $l_object_id);
                            } else {

                                /*
                                 * If we are updating a single value category we obviously only need the object id for identifying the right category entry.
                                 * So let's check for it.
                                 */
                                $l_catentries = $l_cat->get_data_as_array(null, $l_object_id);
                            } // if

                            // Update-Handling
                            if ($l_isysgui["isysgui_cat{$l_cat_suffix}__list_multi_value"] != 1) {
                                if (count($l_catentries)) {
                                    $l_data['data_id'] = $l_catentries[0][$this->get_category_source_table('__id')];

                                    if (!$l_data['data_id']) {
                                        $p_mode = isys_import_handler_cmdb::C__CREATE;
                                    } else {
                                        $p_params['data_id'] = $p_params['data']['id'] = $l_data['data_id'];
                                        $p_mode = isys_import_handler_cmdb::C__UPDATE;
                                    } // if
                                } else {
                                    $p_mode = isys_import_handler_cmdb::C__CREATE;
                                } // if
                            } // if

                            // Build changes array
                            $l_changes = [];
                            foreach ($p_params['data'] as $l_key => $l_value) {
                                if (is_string($l_value) && defined($l_value)) {
                                    $l_changes[isys_import_handler_cmdb::C__PROPERTIES][$l_key][C__DATA__VALUE] = constant($l_value);
                                } else {
                                    $l_changes[isys_import_handler_cmdb::C__PROPERTIES][$l_key][C__DATA__VALUE] = $l_value;
                                    $l_changes[isys_import_handler_cmdb::C__PROPERTIES][$l_key]['title_lang'] = $l_value;
                                } // if
                            } // foreach

                            // Create and setup request
                            $request = new isys_request();

                            $request->set_object_id($l_object_id)
                                ->set_category_data_id($p_params['data_id'] ?: $p_params['id'] ?: $l_data['data_id'] ?: $l_data['id'])
                                ->set_row($l_catentries[0] ?: []);

                            // Validate request
                            if (isys_tenantsettings::get('api.validation', 1)) {
                                // Validate dialog properties
                                $this->validateDialogProperties($p_params['data'], $l_properties, $request, $p_mode);
                            }

                            // Dialog+ handling before merge.
                            foreach ($p_params['data'] as $l_key => $l_value) {
                                // Do not consider empty properties.
                                if ($l_value === null) {
                                    continue;
                                }

                                $dialogTypes = [
                                    C__PROPERTY__INFO__TYPE__DIALOG,
                                    C__PROPERTY__INFO__TYPE__DIALOG_PLUS,
                                    C__PROPERTY__INFO__TYPE__DIALOG_LIST,
                                    C__PROPERTY__UI__TYPE__MULTISELECT
                                ];

                                // Check for type.
                                if (!in_array($l_properties[$l_key][C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE], $dialogTypes, true)) {
                                    continue;
                                }

                                // Arrayize value
                                $dialogValues = is_array($l_value) ? $l_value : [$l_value];

                                // Check whether we have strings to proceed.
                                if (is_string($dialogValues[0])) {
                                    // Build dialog admin.
                                    $dialogAdmin = (new AdminBuilder(
                                        $l_properties[$l_key][C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE],
                                        $l_properties[$l_key],
                                        $request
                                    ))->build();

                                    if ($l_properties[$l_key][C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE] === C__PROPERTY__INFO__TYPE__DIALOG_PLUS) {
                                        $dialogAdmin->setMode(DialogAdmin::MODE_TABLE);
                                    }

                                    // Check every value.
                                    foreach ($dialogValues as $dialogIndex => $dialogValue) {
                                        try {
                                            // Resolve it.
                                            $resolvedValue = $dialogAdmin->findOrCreate($dialogValue);

                                            // Write it back to values.
                                            if (empty($resolvedValue)) {
                                                unset($dialogValues[$dialogIndex]);
                                            } else {
                                                $dialogValues[$dialogIndex] = $resolvedValue;
                                            }
                                        } catch (Exception $e) {
                                            // An error occurred during resolving of dialog value.
                                            unset($dialogValues[$dialogIndex]);
                                        }
                                    }

                                    // Write it back to params.
                                    if (count($dialogValues)) {
                                        $p_params['data'][$l_key] = is_array($l_value) ? $dialogValues : $dialogValues[0];
                                    }
                                }
                            }

                            // Merge new and old data from the database to prevent overwrites through nulls.
                            if (isset($l_data['data_id']) && is_numeric($l_data['data_id']) && count($l_catentries)) {
                                $p_params['data'] = array_merge(self::to_sync_structure($l_catentries[0], $l_cat->get_properties()), $p_params['data']);
                            }

                            /**
                             * @todo Should resolution of constants not happen earlier?
                             */
                            // Build Data-Structure for the sync-routine
                            foreach ($p_params['data'] as $l_key => $l_value) {
                                if (is_string($l_value) && defined($l_value)) {
                                    $l_data['properties'][$l_key][C__DATA__VALUE] = constant($l_value);
                                } else {
                                    $l_data['properties'][$l_key][C__DATA__VALUE] = $l_value;
                                }
                            }

                            // Check whether processor provides sync-modification capabilities
                            if ($categoryProcessor instanceof SyncModifier) {
                                // Run Processor for sync data
                                $l_data = $categoryProcessor->modifySyncData($l_data);
                            }

                            // Emit signal.
                            isys_component_signalcollection::get_instance()
                                ->emit('mod.api.beforeCmdbCategorySync', $l_data, $l_object_id, $p_mode, $l_cat);

                            // Call sync of corresponding category
                            $l_sync = $l_cat->sync($l_data, $l_object_id, $p_mode);

                            if ($l_sync) {
                                // Get category title
                                $l_category_info = $this->get_category_info();
                                $l_category_title = $l_category_info['isysgui_cat' . $this->get_category_suffix() . '__title'];

                                // Create array of changes
                                $l_changes = (new isys_module_logbook())->prepare_changes($l_cat, $l_catentries[0], $l_changes);

                                if (count($l_changes)) {
                                    // @see API-47 Add the custom category ID to provide a property title.
                                    if ($this->get_category_suffix() === 'g_custom') {
                                        foreach ($l_changes as $attribute => $change) {
                                            if ($l_category_info['isysgui_catg_custom__id'] > 0 && substr_count($attribute, '::') === 1) {
                                                $l_changes[$attribute . '::' . $l_category_info['isysgui_catg_custom__id']] = $change;
                                                unset($l_changes[$attribute]);
                                            }
                                        }
                                    }

                                    if ($p_mode == isys_import_handler_cmdb::C__CREATE) {
                                        // Sometimes we get a "from" value here - but that is incorrect.
                                        $l_changes = array_map(function ($arr) {
                                            $arr['from'] = '';

                                            return $arr;
                                        }, $l_changes);
                                    }

                                    // Create logbook entry
                                    isys_event_manager::getInstance()->triggerCMDBEvent(
                                        'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                                        null,
                                        $p_params[C__CMDB__GET__OBJECT],
                                        $this->m_dao->get_objTypeID($l_object_id),
                                        $l_category_title,
                                        serialize($l_changes)
                                    );
                                }
                            }

                            // Emit category signal (afterCategoryEntrySave).
                            isys_component_signalcollection::get_instance()->emit(
                                'mod.cmdb.afterCategoryEntrySave',
                                $l_cat,
                                isset($l_data['data_id']) ? $l_data['data_id'] : null,
                                $l_sync,
                                $l_object_id,
                                $l_data,
                                isset($l_changes) ? $l_changes : []
                            );

                            // Add object and category to the index
                            \idoit\Module\Api\SearchIndexRegister::register($l_object_id, [$l_cat->get_category_const()]);

                            return $l_sync;
                        }

                        throw new isys_exception_api('There was an unknown validation error.');
                    }
                }
            } else {
                throw new isys_exception_api('Unable to prepare requested category data. Please make sure the given category ID or constant exists.');
            }
        } catch (isys_exception_api_validation $e) {
            throw $e;
        } catch (isys_exception_api $e) {
            throw $e;
        }

        return false;
    }

    /**
     * Validate dialog properties
     *
     * @see API-79
     *
     * @param array        $propertyData
     * @param array        $propertyDefinition
     * @param isys_request $request
     * @param string       $mode
     *
     * @throws isys_exception_api_validation
     */
    public function validateDialogProperties(array $propertyData, array $propertyDefinition, isys_request $request, $mode)
    {
        // Properties to handle
        $propertyTypesToHandle = [
            C__PROPERTY__INFO__TYPE__DIALOG,
            C__PROPERTY__INFO__TYPE__DIALOG_PLUS,
            C__PROPERTY__INFO__TYPE__DIALOG_LIST,
            C__PROPERTY__INFO__TYPE__MULTISELECT,
        ];

        // Validation error store
        $validationErrors = [];

        // Dialog, Dialog(+), DialogList, MultiList - Handling
        foreach ($propertyData as $propertyName => $propertyValue) {
            // Check for filled value
            if ($propertyValue === null) {
                continue;
            }

            // Get property types
            $propertyType = $propertyDefinition[$propertyName][C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE];

            // Check whether property type should be handled here
            if (empty($propertyType) || !in_array($propertyType, $propertyTypesToHandle)) {
                continue;
            }

            /**
             * Get dialog data as "id" => "title"
             */

            // Dialog data store
            $dialogData = [];

            try {
                $dialogAdmin = (new AdminBuilder($propertyType, $propertyDefinition[$propertyName], $request))->build();

                $dialogData = $dialogAdmin->getData();
            } catch (Exception $e) {
                // Unset dialog data explicitly to prevent further processing and calculating a wrong validation result
                $dialogData = null;
            }

            // Validate calculated dialog data
            if (!is_array($dialogData)) {
                /**
                 * We have to skip further validation checks
                 * because there is no data or data retrieval
                 */
                continue;
            }

            try {
                /**
                 * Skip dialog validation if following conditions are met:
                 *
                 * 1. We are going to create a new category entry
                 * 2. Dialog data will be provided by a callback function
                 */
                if ($mode === isys_import_handler_cmdb::C__CREATE &&
                    ($propertyDefinition[$propertyName][C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_arData'] instanceof isys_callback)) {
                    continue;
                }

                (new DialogValue($propertyName, $propertyValue, $propertyDefinition[$propertyName]))->setDialogData($dialogData)
                    ->validate();
            } catch (ValidationException $e) {
                $validationErrors[$propertyName] = $e->getMessage();
            } catch (\Exception $e) {
                continue;
            }
        }

        // Check whether errors occurred and throw a corresponding validation exception
        if (count($validationErrors) > 0) {
            throw new isys_exception_api_validation('Validation errors in dialog related properties occured.', $validationErrors);
        }
    }

    /**
     * Get category suffix
     *
     * @return string
     */
    public function get_category_suffix()
    {
        return $this->m_category_suffix;
    }

    /**
     * Get category info
     *
     * @return array
     */
    public function get_category_info()
    {
        return $this->m_category_info;
    }

    /**
     * Transform stored data for merging it with the delivered data from the API.
     *
     * @param  array $p_data          API-DATA
     * @param  array $p_category_info Category Properties
     *
     * @return array
     */
    protected function to_sync_structure($p_data, $p_category_info)
    {
        $l_result = [];

        if (is_array($p_category_info) && count($p_category_info) && is_array($p_data) && count($p_data)) {
            foreach ($p_category_info AS $l_property_key => $l_property_data) {
                // ID-3103 n2m fields will mostly use the category ID as their data field... But that is wrong.
                if ($l_property_data[C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE] == C__PROPERTY__INFO__TYPE__N2M) {
                    continue;
                }

                // @see  API-188  We seem to be dealing with a unit field, we should calculate the correct value.
                if (isset($l_property_data[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__UNIT]) && $l_property_data[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][1] === 'convert') {
                    $dataField = $l_property_data[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD];
                    $unitDataField = $p_category_info[$l_property_data[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__UNIT]][C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD];
                    $convertCallback = $l_property_data[C__PROPERTY__FORMAT][C__PROPERTY__FORMAT__CALLBACK][2][0];

                    if ($p_data[$dataField] > 0 && $p_data[$unitDataField] > 0 && method_exists('isys_convert', $convertCallback)) {
                        $l_result[$l_property_key] = isys_convert::{$convertCallback}($p_data[$dataField], $p_data[$unitDataField], C__CONVERT_DIRECTION__BACKWARD);

                        // No need to process the value any furhter...
                        continue;
                    }
                }

                if (isset($p_data[$l_property_data[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD]])) {
                    // Handling for connectionFields
                    if (isset($l_property_data[C__PROPERTY__FORMAT]['callback'][1]) && $l_property_data[C__PROPERTY__FORMAT]['callback'][1] == 'connection') {
                        $l_result[$l_property_key] = $p_data['isys_connection__isys_obj__id'];
                    } else {
                        $l_result[$l_property_key] = $p_data[$l_property_data[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD]];
                    }
                }
            }
        }

        return $l_result;
    }

    /**
     * CategoryMethod: cmdb.category.set
     *
     * @param $params
     *
     * @return array
     * @throws Exception
     */
    public function save(array $params)
    {
        // Validate request
        if (isys_tenantsettings::get('api.validation', 1)) {
            // Validate request first
            $this->validateSetRequest($params);
        }
        // Prepare category
        $categoryDao = $this->prepare($params);

        // Calculate needed mode
        $mode = isys_import_handler_cmdb::C__CREATE;

        // Do not allow specifiying entry id for single value categories
        if (isset($params['entry']) && !$categoryDao->is_multivalued()) {
            unset($params['entry']);
        }

        // Check whether entryId is setted
        if (!empty($params['entry'])) {
            // We will update a category entry
            $mode = isys_import_handler_cmdb::C__UPDATE;

            // Set category entry id
            $params['data']['id'] = $params['entry'];
        }

        // Remap object id
        $params['objID'] = $params['object'];

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$params['objID']);

        // Handle category request
        if ($categoryEntryId = $this->sync_wrapper($params, $mode)) {
            // Update last changed time o object
            $this->m_dao->object_changed($params['object']);

            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$params['objID']);

            // Create response
            return [
                'success' => true,
                'message' => 'Category entry successfully saved',
                'entry'   => (int)$categoryEntryId,
            ];
        }

        // @see  API-191  Unlock the object once the request has finished.
        $this->unlockObject((int)$params['objID']);

        return [
            'success' => false,
            'message' => 'Error while saving category'
        ];
    }

    /**
     * Validate set request
     *
     * @param array $params
     *
     * @throws ValidationException
     * @throws isys_exception_database
     */
    public function validateSetRequest(array $params)
    {
        // Collected needed parameter
        $objectId = $params['object'];
        $entryId = $params['entry'];
        $categoryConstant = $params['category'];

        // Validate object id
        if (empty($objectId) || !is_int($objectId) || !$this->m_dao->obj_exists($objectId)) {
            throw new ValidationException('object', 'ObjectId needs to be a integer value which references an existing object.');
        }

        // Validate categoryConstant
        if (empty($categoryConstant) || !is_string($categoryConstant) || !defined($categoryConstant)) {
            throw new ValidationException('category', 'Please provide a valid i-doit category constant as string value.');
        }

        // Validate entry id
        if (!empty($entryId) && !is_int($entryId)) {
            throw new ValidationException('entry', 'Entry id needs to be a positive integer value.');
        }
    }

    /**
     * Save category
     *
     * @param array $p_params Parameters (depends on data method)
     *
     * @return array
     * @throws \Exception
     */
    public function update($p_params)
    {
        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$p_params[C__CMDB__GET__OBJECT]);
        // Remap object ID, because we use it in some category processors.
        $p_params['object'] = (int)$p_params[C__CMDB__GET__OBJECT];

        if ($this->sync_wrapper($p_params, isys_import_handler_cmdb::C__UPDATE)) {
            // Update 'isys_obj__updated'
            $this->m_dao->object_changed($p_params[C__CMDB__GET__OBJECT]);

            $result = [
                'success' => true,
                'message' => 'Category entry successfully saved.'
            ];
        } else {
            $result = [
                'success' => false,
                'message' => 'Error while saving category.'
            ];
        }

        // Add deprecation info to message
        if (!empty($result['message'])) {
            $result['message'] .= ' [This method is deprecated and will be removed in a feature release. Use \'cmdb.category.save\' instead.]';
        }

        // @see  API-191  Unlock the object once the request has finished.
        $this->unlockObject((int)$p_params[C__CMDB__GET__OBJECT]);

        return $result;
    }

    /**
     * Rank category entry:
     *
     * This method only moves the given category entry
     * to the next deletion status: normal -> archived, archived -> deleted...
     * It will not purge the entry!
     *
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     * @throws AuthenticationException
     * @throws InternalErrorException
     * @internal param string $p_method Data method
     * @return array|boolean
     */
    public function delete($p_params)
    {
        // New delete handling
        if (isset($p_params['object']) && isset($p_params['category']) && isset($p_params['entry'])) {
            // Create category entry instance
            $categoryEntry = new Entry($p_params['object'], $p_params['category'], $p_params['entry']);

            // Check fopr rights
            if ($this->useAuth && !isys_auth_cmdb::instance()->has_rights_in_obj_and_category(isys_auth::ARCHIVE, $categoryEntry->getObjectId(), $categoryEntry->getCategoryConstant()) && !isys_auth_cmdb::instance()->has_rights_in_obj_and_category(isys_auth::DELETE, $categoryEntry->getObjectId(), $categoryEntry->getCategoryConstant())) {
                throw new AuthenticationException('You are not allowed to rank this category entry.');
            }

            // @see  API-191  Check and react to the object lock.
            $this->checkObjectLock((int)$categoryEntry->getObjectId());

            // Recycle entry
            if ($categoryEntry->delete()) {
                // @see  API-191  Unlock the object once the request has finished.
                $this->unlockObject((int)$categoryEntry->getObjectId());

                return [
                    'success' => true,
                    'message' => 'Entry ' . $categoryEntry->getEntryId() . ' has been successfully ranked from ' . $categoryEntry->getEntryStatus() . ' to ' .
                        $categoryEntry->getTargetStatus() . '.'
                ];
            }

            throw new InternalErrorException('Unable to delete category entry.');
        }

        // Init
        $l_object_id = @$p_params[C__CMDB__GET__OBJECT];

        // Validate object id
        if (!$l_object_id || $l_object_id < 2) {
            throw new isys_exception_api('Object id invalid. ID must be positive and higher than one.', -32602);
        }

        // Get category params
        if (!($l_cat = $this->prepare($p_params))) {
            return false;
        }

        $l_cat_suffix = $this->get_category_suffix();
        $l_isysgui = $this->get_category_info();
        $l_source_table = $this->get_category_source_table();

        if (isset($p_params[C__CMDB__GET__CATLEVEL]) && $p_params[C__CMDB__GET__CATLEVEL]) {
            $l_category_id = $p_params[C__CMDB__GET__CATLEVEL];
        } elseif (isset($p_params['id']) && $p_params['id']) {
            $l_category_id = $p_params['id'];
        } else {
            throw new isys_exception_api('Category ID missing. You must specify the parameter \'id\' with the content of the entry ID of the category.');
        }

        // @see  API-166  We simply create an instance because this will throw an exception if the entry does not exist.
        new Entry($l_object_id, $l_cat->get_category_const(), $l_category_id);

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$l_object_id);

        if ($this->useAuth) {
            isys_auth_cmdb_categories::instance()
                ->check_rights_obj_and_category(isys_auth::DELETE, $l_object_id, $l_cat->get_category_const());
        }

        if ($l_isysgui["isysgui_cat{$l_cat_suffix}__list_multi_value"] == 0) {
            throw new isys_exception_api('Your category is not multi-valued. It is not possible to delete a single value category.');
        }

        if ($this->get_category_source_table() === 'isys_catg_virtual_list') {
            throw new isys_exception_api('The category does not support deletion.');
        }

        if (!method_exists($l_cat, 'rank_record')) {
            throw new isys_exception_api('CMDB-Category-Error! Rank method not supported by the category.');
        }

        $l_return = $l_cat->rank_record($l_category_id, C__CMDB__RANK__DIRECTION_DELETE, $l_source_table);

        // @see  API-191  Unlock the object once the request has finished.
        $this->unlockObject((int)$l_object_id);

        if ($l_return) {
            return [
                'success' => true,
                'message' => 'Category entry \'' . $l_category_id . '\' successfully deleted'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Category entry \'' . $l_category_id . '\' was not deleted. It may not exists.'
            ];
        }
    }

    /**
     * Quickpurge category entry:
     * This method will purge the entry and therefore remove it from the database.
     *
     * @param   array $p_params Parameters (depends on data method).
     *
     * @return array|bool Returns itself.
     * @throws isys_exception_api
     * @throws isys_exception_cmdb
     * @throws isys_exception_dao
     * @throws isys_exception_database
     * @throws isys_exception_general
     */
    public function quickpurge($p_params)
    {
        /**
         * @todo We should delete this method and link to the new purge() method instead
         */

        if (!isys_tenantsettings::get('cmdb.quickpurge', false)) {
            throw new isys_exception_api('Quickpurge is not enabled');
        }

        // Init.
        $l_object_id = @$p_params[C__CMDB__GET__OBJECT];

        // Validate object id.
        if (!$l_object_id || $l_object_id < 2) {
            throw new isys_exception_api('Object id invalid. ID must be positive and higher than one.', -32602);
        }

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$l_object_id);

        $l_isysgui = $this->get_category_info();

        // Get category params
        if (($l_cat = $this->prepare($p_params))) {
            $l_source_table = $this->get_category_source_table();

            if (@$p_params[C__CMDB__GET__CATLEVEL]) {
                $l_category_id = $p_params[C__CMDB__GET__CATLEVEL];
            } elseif (@$p_params['id']) {
                $l_category_id = $p_params['id'];
            } else {
                throw new isys_exception_api('Category ID missing. You must specify the parameter \'id\' with the content of the entry ID of the category.');
            }

            // @see  API-166  We simply create an instance because this will throw an exception if the entry does not exist.
            new Entry($l_object_id, $l_cat->get_category_const(), $l_category_id);

            // Check class and instantiate it.
            if ($this->get_category_source_table() !== 'isys_catg_virtual_list' ||
                ($l_cat->category_entries_purgable() && $l_isysgui["isysgui_cat" . $this->get_category_suffix() . "__list_multi_value"] == 0)) {
                if (method_exists($l_cat, 'rank_record')) {
                    if ($l_cat->rank_record($l_category_id, C__CMDB__RANK__DIRECTION_DELETE, $l_source_table, null, true)) {
                        // @see  API-191  Unlock the object once the request has finished.
                        $this->unlockObject((int)$l_object_id);

                        return [
                            'success' => true,
                            'message' => 'Category entry \'' . $l_category_id . '\' successfully purged'
                        ];
                    }

                    // @see  API-191  Unlock the object once the request has finished.
                    $this->unlockObject((int)$l_object_id);

                    return [
                        'success' => false,
                        'message' => 'Category entry \'' . $l_category_id . '\' was not purged. It may not exists.'
                    ];
                }

                throw new isys_exception_api('CMDB-Category-Error! Rank method not supported by the category.');
            }

            throw new isys_exception_api('The category does not support purge.');
        }

        return false;
    }

    /**
     * Purge category entry
     *
     * @param array $params
     *
     * @return array
     * @throws InternalErrorException
     * @throws ParameterException
     * @throws isys_exception_cmdb
     * @throws isys_exception_dao
     * @throws isys_exception_database
     * @throws isys_exception_general
     * @throws \Exception
     */
    public function purge(array $params)
    {
        // Create category entry instance
        $categoryEntry = new Entry($params['object'], $params['category'], $params['entry']);

        // Check rights
        if ($this->useAuth && !isys_auth_cmdb::instance()->has_rights_in_obj_and_category(isys_auth::ARCHIVE, $categoryEntry->getObjectId(), $categoryEntry->getCategoryConstant()) && !isys_auth_cmdb::instance()->has_rights_in_obj_and_category(isys_auth::DELETE, $categoryEntry->getObjectId(), $categoryEntry->getCategoryConstant())) {
            throw new AuthenticationException('You are not allowed to rank this category entry.');
        }

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$categoryEntry->getObjectId());

        // Recycle entry
        if ($categoryEntry->purge()) {
            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$categoryEntry->getObjectId());

            return [
                'success' => true,
                'message' => 'Entry ' . $categoryEntry->getEntryId() . ' has been successfully purged from ' . $categoryEntry->getEntryStatus() . ' to ' .
                    $categoryEntry->getTargetStatus() . '.'
            ];
        }

        throw new InternalErrorException('Unable to purge category entry.');
    }

    /**
     * Recycle entries
     *
     * @param array $params
     *
     * @return array
     *
     * @throws Exception
     */
    public function recycle(array $params)
    {
        // Create category entry instance
        $categoryEntry = new Entry($params['object'], $params['category'], $params['entry']);

        // Check rights
        if ($this->useAuth && !isys_auth_cmdb::instance()->has_rights_in_obj_and_category(isys_auth::ARCHIVE, $categoryEntry->getObjectId(), $categoryEntry->getCategoryConstant()) && !isys_auth_cmdb::instance()->has_rights_in_obj_and_category(isys_auth::DELETE, $categoryEntry->getObjectId(), $categoryEntry->getCategoryConstant())) {
            throw new AuthenticationException('You are not allowed to rank this category entry.');
        }

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$params['object']);

        // Recycle entry
        if ($categoryEntry->recycle()) {
            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$params['object']);

            return [
                'success' => true,
                'message' => 'Entry ' . $categoryEntry->getEntryId() . ' has been successfully recycled from ' . $categoryEntry->getEntryStatus() . ' to ' .
                    $categoryEntry->getTargetStatus() . '.'
            ];
        }

        throw new InternalErrorException('Unable to recycle category entry.');
    }

    /**
     * Archive entry
     *
     * @param array $params
     *
     * @return array
     * @throws InternalErrorException
     * @throws \Exception
     */
    public function archive(array $params)
    {
        // Create category entry instance
        $categoryEntry = new Entry($params['object'], $params['category'], $params['entry']);

        // Check rights
        if ($this->useAuth && !isys_auth_cmdb::instance()->has_rights_in_obj_and_category(isys_auth::ARCHIVE, $categoryEntry->getObjectId(), $categoryEntry->getCategoryConstant()) && !isys_auth_cmdb::instance()->has_rights_in_obj_and_category(isys_auth::DELETE, $categoryEntry->getObjectId(), $categoryEntry->getCategoryConstant())) {
            throw new AuthenticationException('You are not allowed to rank this category entry.');
        }

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$categoryEntry->getObjectId());

        // Archive entry
        if ($categoryEntry->archive()) {
            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$categoryEntry->getObjectId());

            return [
                'success' => true,
                'message' => 'Entry ' . $categoryEntry->getEntryId() . ' has been successfully archived from ' . $categoryEntry->getEntryStatus() . ' to ' . $categoryEntry->getTargetStatus() . '.'
            ];
        }

        throw new InternalErrorException('Unable to archive category entry.');
    }

    /**
     * Set the get-param for
     * retrieving the categoryID
     *
     * @param   string $p_category_get
     *
     * @return $this
     */
    protected function set_category_get($p_category_get)
    {
        $this->m_category_get = $p_category_get;

        return $this;
    }

    /**
     * Get category get paramater: catgID, catsID, customID
     *
     * @return string
     */
    protected function get_category_get()
    {
        return $this->m_category_get;
    }

    /**
     * Is category type specific?
     *
     * @return bool
     */
    protected function is_specific()
    {
        return (C__CMDB__CATEGORY__TYPE_SPECIFIC == $this->get_category_type());
    }

    /**
     * Get the current category type
     *
     * @return int
     */
    protected function get_category_type()
    {
        return $this->m_category_type;
    }
}
