<?php

/**
 * i-doit
 *
 * API model
 *
 * @package    i-doit
 * @subpackage API
 * @author     Dennis Stücken <dstuecken@synetics.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_cmdb_object_types extends isys_api_model_cmdb implements isys_api_model_interface
{
    /**
     * Data formatting used in format methods
     *
     * @var array
     */
    protected $m_mapping = [
        'isys_obj_type__id'                      => 'id',
        'isys_obj_type__title'                   => ['_L', 'title'],
        'isys_obj_type__container'               => 'container',
        'isys_obj_type__const'                   => 'const',
        'isys_obj_type__color'                   => 'color',
        'isys_obj_type__obj_img_name'            => ['isys_api_model_cmdb_object_types::get_image_url', 'image'],
        'isys_obj_type__icon'                    => 'icon',
        'isys_obj_type__isysgui_cats__id'        => 'cats',
        'isys_obj_type__isys_obj_type_group__id' => 'tree_group',
        'isys_obj_type__status'                  => 'status',
        'isys_obj_type_group__id'                => 'type_group',
        'isys_obj_type_group__title'             => ['_L', 'type_group_title'],
    ];

    /**
     * Possible options and their parameters
     *
     * @var array
     */
    protected $m_options = [
        'read' => [
            'filter'   => [
                'type'        => 'array',
                'description' => 'Filter array',
                'optional'    => true
            ],
            'limit'    => [
                'type'        => 'int',
                'description' => 'Resultset limiting',
                'optional'    => true
            ],
            'sort'     => [
                'type'        => 'string',
                'description' => 'ASC or DESC',
                'optional'    => true
            ],
            'order_by' => [
                'type'        => 'string',
                'description' => 'Ordering by title, id or status',
                'optional'    => true
            ]
        ]
    ];

    /**
     * Validation
     *
     * @var array
     */
    protected $m_validation = [];

    /**
     * @param string $unused
     * @param array  $row
     *
     * @return string
     * @throws Exception
     */
    public static function get_image_url($unused, array $row): string
    {
        return isys_application::instance()->container->get('route_generator')
            ->generate('cmdb.object-type.image', ['objectTypeId' => $row['isys_obj_type__id']]);
    }

    /**
     * Fetches object types by filter.
     * [
     *    integer  $p_params['filter']['id']       (optional) Object type identifier or Constants
     *    array    $p_params['filter']['ids']      (optional) Object type identifiers or Constants
     *    string   $p_params['filter']['title']    (optional) Object type title
     *    array    $p_params['filter']['titles']   (optional) Object type titles
     *    boolean  $p_params['filter']['enabled']  (optional) Show only object types enabled or disabled in GUI.
     *    boolean  $p_params['raw']                (optional) Formatting.
     *    string   $p_params['order_by']           (optional) Order by one of the supported filter arguments. Defaults to null that means result will be ordered by object identifiers.
     *    string   $p_params['sort']               (optional) Order result ascending ('ASC') or descending ('DESC').
     *    integer  $p_params['limit']              (optional) Limitation: where to start and number of elements, i.e. 0 or 0,10. Defaults to null that means no limitation.
     * ]
     *
     * @param array $p_params Parameters:
     *
     * @return array Objects types. Returns an empty array on error.
     * @author Benjamin Heisig <bheisig@synetics.de>
     */
    public function read($p_params)
    {
        if (!array_key_exists('filter', $p_params) || !is_array($p_params['filter']))
        {
            $p_params['filter'] = [];
        }

        // Force limit to record status 'normal':
        $p_params['filter']['status'] = C__RECORD_STATUS__NORMAL;

        // Raw mode:
        $l_raw      = (bool) $p_params['raw'];
        $l_order_by = null;
        $l_sort     = null;
        $l_limit    = null;
        $l_count    = false;

        // Order by.
        if (isset($p_params['order_by']) && !empty($p_params['order_by']))
        {
            $l_order_by = $p_params['order_by'];
        }

        // Sort.
        if (isset($p_params['sort']) && !empty($p_params['sort']))
        {
            $l_sort = $p_params['sort'];
        }

        // Limitation.
        if (isset($p_params['limit']) && !empty($p_params['limit']))
        {
            $l_limit = $p_params['limit'];
        }

        // Show object count.
        if (isset($p_params['countobjects']))
        {
            $l_count = (bool) $p_params['countobjects'];
        }

        // Data retrieval:
        $l_data = $this->m_dao->get_object_types_by_properties(
            $p_params['filter'],
            $l_order_by,
            $l_sort,
            $l_limit,
            $l_count
        );

        $l_return = [];

        // Data formatting:
        while ($l_row = $l_data->get_row())
        {
            if ($l_row['isys_obj_type__id'] == C__OBJTYPE__LOCATION_GENERIC || $l_row['isys_obj_type__id'] == C__OBJTYPE__MIGRATION_OBJECT)
            {
                continue;
            }

            if ($this->useAuth)
            {
                try
                {
                    isys_auth_cmdb::instance()->obj_type(isys_auth::VIEW, $l_row['isys_obj_type__const']);
                }
                catch (isys_exception_auth $e)
                {
                    $this->m_log->error($e->getMessage());
                    continue;
                }
            }

            if ($l_raw)
            {
                $l_new = $l_row;
            }
            else
            {
                $l_new = $this->format_by_mapping($this->m_mapping, $l_row);

                if ($l_count)
                {
                    $l_new['objectcount'] = $this->m_dao->count_objects(null, $l_row['isys_obj_type__id'], true);
                }
            }

            $l_return[] = $l_new;
        }

        // Order by translated titles:
        if ($l_order_by === 'title' && $l_raw === false && (!$l_sort || (strtoupper($l_sort)==='ASC'))) {
            usort($l_return, [$this, 'sort_by_title']);
        } elseif ($l_order_by === 'title' && $l_raw === false && (strtoupper($l_sort)==='DESC')) {
            usort($l_return, [$this, 'sort_by_title_desc']);
        }

        return $l_return;
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @return isys_api_model_cmdb Returns itself.
     * @throws isys_exception_api
     */
    public function create($p_params)
    {
        throw new isys_exception_api('Creating is not possible here.');
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @return isys_api_model_cmdb Returns itself.
     * @throws isys_exception_api
     */
    public function delete($p_params)
    {
        throw new isys_exception_api('Deleting is not possible here.');
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @return isys_api_model_cmdb Returns itself.
     * @throws isys_exception_api
     */
    public function update($p_params)
    {
        throw new isys_exception_api('Updating is not possible here.');
    }

    /**
     * Constructor
     */
    public function __construct(isys_cmdb_dao $p_dao)
    {
        $this->m_dao = $p_dao;
        parent::__construct();
    }
}
