<?php

use idoit\Module\Api\Exception\JsonRpc\ParameterException;

/**
 * i-doit APi
 *
 * @package    i-doit
 * @subpackage API
 * @author     Dennis Stücken <dstuecken@synetics.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_idoit_constants implements isys_api_model_interface
{

    /**
     * Possible options and their parameters
     *
     * @var array
     */
    protected $m_options = [
        'read' => []
    ];

    /**
     * Documentation missing
     *
     * @param $params
     *
     * @return array
     * @throws ParameterException
     * @throws isys_exception_api
     * @throws Exception
     */
    public function read($params)
    {
        // Mapping of constant groups to method
        $mapping = [
            'objectTypes'   => 'getObjectTypes',
            'categories'    => 'getCategories',
            'recordStates'  => 'getRecordStates',
            'relationTypes' => 'getRelationTypes',
            'staticObjects' => 'getStaticObjects',
        ];

        // Calculate required constant groups
        $requiredConstantGroups = $params['constants'] ?: array_keys($mapping);

        if (!is_array($requiredConstantGroups)) {
            throw new ParameterException('Parameter \'constants\' needs to be an array which includes several constant groups. 
                                Possible groups are: ' . implode(', ', array_keys($mapping)));
        }

        // Check whether database component is initialized
        if (method_exists(isys_application::instance()->container, 'get') && isys_application::instance()->container->get('database') instanceof isys_component_database) {
            // Resolved constants store
            $constants = [];

            // Iterate over constant groups
            foreach ($requiredConstantGroups as $constantGroup) {
                // Check whether constant group exists and has an resolver method
                if (!method_exists($this, $mapping[$constantGroup])) {
                    continue;
                }

                // Resolve constant groups constants
                $constants[$constantGroup] = call_user_func_array([$this, $mapping[$constantGroup]], []);
            }
        } else {
            throw new isys_exception_api('Error getting constants: The database component is not available.');
        }

        return $constants;
    }

    /**
     * Get object types
     *
     * @return array
     * @throws Exception
     */
    private function getObjectTypes()
    {
        $results = [];
        $resource = isys_application::instance()->container->get('cmdb_dao')
            ->get_object_types();

        if ($resource->num_rows()) {
            while ($objectRow = $resource->get_row()) {
                $results[$objectRow['isys_obj_type__const']] = ucwords(strtolower(isys_application::instance()->container->get('language')
                    ->get($objectRow['isys_obj_type__title'])));
            }
        }

        return $results;
    }

    /**
     * Get categories
     *
     * @return array
     * @throws isys_exception_database
     * @throws Exception
     */
    private function getCategories()
    {
        $results = [];

        foreach ([
                     'g',
                     's',
                     'g_custom'
                 ] as $categoryType) {

            // Set category type specific resource
            $categoryTypeResource = isys_application::instance()->container->get('cmdb_dao')
                ->get_isysgui('isysgui_cat' . $categoryType);

            // Check whether resource is valid
            if (isset($categoryTypeResource) && is_object($categoryTypeResource)) {
                while ($l_row = $categoryTypeResource->get_row()) {
                    $results[$categoryType][$l_row['isysgui_cat' . $categoryType . '__const']] = _L($l_row['isysgui_cat' . $categoryType . '__title']);
                }
            }
        }

        return $results;
    }

    /**
     * Get static objects
     *
     * @return array
     * @throws isys_exception_database
     * @throws Exception
     */
    private function getStaticObjects()
    {
        $results = [];
        $resource = isys_application::instance()->container->get('cmdb_dao')
            ->retrieve('SELECT * FROM isys_obj WHERE isys_obj__const <> \'\';');

        if ($resource->num_rows()) {
            while ($objectRow = $resource->get_row()) {
                $results[$objectRow['isys_obj__const']] = isys_application::instance()->container->get('language')
                    ->get($objectRow['isys_obj__title']);
            }
        }

        return $results;
    }

    /**
     * Get relation types
     *
     * @return array
     * @throws isys_exception_database
     * @throws Exception
     */
    private function getRelationTypes()
    {
        $results = [];

        $resource = isys_application::instance()->container->get('cmdb_dao')
            ->retrieve('SELECT * FROM isys_relation_type WHERE TRUE;');

        if ($resource->num_rows()) {
            while ($objectRow = $resource->get_row()) {
                $results[$objectRow['isys_relation_type__const']] = isys_application::instance()->container->get('language')
                    ->get($objectRow['isys_relation_type__title']);
            }
        }

        return $results;
    }

    /**
     * Get record statuses
     *
     * @return array
     * @throws Exception
     */
    private function getRecordStates()
    {
        // Get language manager
        $languageManager = isys_application::instance()->container->get('language');

        return [
            'C__RECORD_STATUS__NORMAL'   => $languageManager->get('LC__CMDB__RECORD_STATUS__NORMAL'),
            'C__RECORD_STATUS__ARCHIVED' => $languageManager->get('LC__CMDB__RECORD_STATUS__ARCHIVED'),
            'C__RECORD_STATUS__DELETED'  => $languageManager->get('LC__CMDB__RECORD_STATUS__DELETED'),
            'C__RECORD_STATUS__PURGE'    => $languageManager->get('LC__CMDB__RECORD_STATUS__PURGE'),
        ];
    }
}