<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Dialog;

use idoit\Component\Property\Property;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use idoit\Module\Api\Exception\ValidationException;
use isys_request;

/**
 * Class AdminBuilder
 *
 * @package idoit\Module\Api\Dialog
 */
class AdminBuilder
{
    /**
     * @var array|\isys_callback
     */
    protected $array;

    /**
     * @var string
     */
    protected $implementationClass;

    /**
     * @var array|Property
     */
    protected $propertyDefinition;

    /**
     * @var isys_request
     */
    protected $request;

    /**
     * @var string
     */
    protected $table;

    /**
     * @return array|\isys_callback
     */
    public function getArray()
    {
        return $this->array;
    }

    /**
     * @param array|\isys_callback $array
     *
     * @return AdminBuilder
     */
    public function setArray($array)
    {
        $this->array = $array;

        return $this;
    }

    /**
     * @return string
     */
    public function getTable()
    {
        return $this->table;
    }

    /**
     * @param string $table
     *
     * @return AdminBuilder
     */
    public function setTable($table)
    {
        $this->table = $table;

        return $this;
    }

    /**
     * @return isys_request
     */
    public function getRequest()
    {
        return $this->request;
    }

    /**
     * @param isys_request $request
     *
     * @return AdminBuilder
     */
    public function setRequest($request)
    {
        $this->request = $request;

        return $this;
    }

    /**
     * @return array
     */
    public function getPropertyDefinition()
    {
        return $this->propertyDefinition;
    }

    /**
     * @param array|Property $propertyDefinition
     *
     * @return AdminBuilder
     */
    public function setPropertyDefinition($propertyDefinition)
    {
        if (!($propertyDefinition instanceof Property) && !is_array($propertyDefinition)) {
            throw new InternalErrorException('The provided property definition needs to be an array or instance of Property.');
        }

        $this->propertyDefinition = $propertyDefinition;

        return $this;
    }

    /**
     * Build admin instance.
     *
     * @return DialogAdmin
     * @throws \Exception
     */
    public function build()
    {
        // @todo  Custom categories need specific logic to provide CONST information.

        // Build Implementation based on provided information.
        if ($this->getArray() instanceof \isys_callback) {
            if (!$this->getRequest()) {
                throw new \Exception('Unable to build Administration object without request.');
            }

            return call_user_func([$this->implementationClass, 'byCallback'], $this->getArray(), $this->getRequest(), $this->getPropertyDefinition());
        }

        if (is_array($this->getArray())) {
            return call_user_func([$this->implementationClass, 'byArray'], $this->getArray(), $this->getPropertyDefinition());
        }

        if (is_string($this->getTable())) {
            return call_user_func([$this->implementationClass, 'byTableName'], $this->getTable(), $this->getPropertyDefinition());
        }

        throw new \Exception('Unable to instantiate Dialog administrator with given parameters.');
    }

    /**
     * AdminBuilder constructor.
     *
     * @param string              $propertyType
     * @param array|Property|null $propertyDefinition
     * @param isys_request|null   $request
     *
     * @throws \Exception
     */
    public function __construct($propertyType, $propertyDefinition = null, \isys_request $request = null)
    {
        if ($propertyDefinition !== null && !($propertyDefinition instanceof Property) && !is_array($propertyDefinition)) {
            throw new InternalErrorException('The provided property definition needs to be an array or instance of Property.');
        }

        // Determine needed implementation.
        switch ($propertyType) {
            case C__PROPERTY__INFO__TYPE__DIALOG:
                $this->implementationClass = DialogAdmin::class;
                break;
            case C__PROPERTY__INFO__TYPE__DIALOG_LIST:
            case C__PROPERTY__INFO__TYPE__MULTISELECT:
            case C__PROPERTY__INFO__TYPE__DIALOG_PLUS:
                $this->implementationClass = DialogPlusAdmin::class;
                break;
            default:
                throw new \Exception('Unable to build Administration for given property type.');
        }

        if (!empty($propertyDefinition)) {
            if ($propertyDefinition[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_arData'] instanceof \isys_callback && $request instanceof \isys_request) {
                $propertyDefinition[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_arData'] =
                    $propertyDefinition[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_arData']->execute($request);
            }

            $this->setPropertyDefinition($propertyDefinition)
                ->setArray($propertyDefinition[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_arData'])
                ->setTable($propertyDefinition[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_strTable']);
        }

        $this->setRequest($request);
    }
}
